package com.mapr.db.rowcol;

import static com.mapr.db.impl.Constants.DEFAULT_FAMILY_MAP;
import static com.mapr.db.impl.Constants.DEFAULT_NAME_MAP;
import static com.mapr.db.impl.Constants.EMPTY_ROWKEY;
import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.ojai.Document;
import org.ojai.FieldPath;
import org.ojai.store.DocumentMutation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.impl.IdCodec;

/*
 * Class to encode / decode a Ojai document into the rowcol format bytebuffer
 */
public class RowcolCodec {

  static Logger logger = LoggerFactory.getLogger(RowcolCodec.class);
  public static DBDocumentImpl getDBDocument(Document r) {
    if (r instanceof DBDocumentImpl){
      return (DBDocumentImpl) r;
    } else {
      return (DBDocumentImpl) KeyValueBuilder.initFrom(r);
    }
  }

  /*
   * In scenarios like copytable, the original jsonPathMap and cachedBufferMap inside the 
   * document's cachedBuffer are from the src table, which may be different from the
   * ones in the dst table, when we encode this for dst table, we create a new cachedBufferMap
   * for the dst. Here is an exmaple:
   * At src table, we have familyId and field path mapping
   * origJsonPathMap {""->1, "a.b"->5, "c.d"->8}
   * origCachedBufferMap {1->ByteBuff1, 5->ByteBuff5, 8->ByteBuff8}
   * At dst table, we have a different mapping
   * newJsonPathMap  {""->1, "a.b"->2, "c.d"->10}
   * this function will create a new CachedBufferMap
   * newCachedBufferMap {1->ByteBuff1, 2->ByteBuff5, 10->ByteBuff8}
   */
  private static
  Map<Integer, ByteBuffer> getUpdatedCachedBufferMap (
                                Map <FieldPath, Integer> newJsonPathMap,
                                DBDocumentImpl origRec) {
    if (origRec.cachedBuffer == null) {
      return null;
    }
    Map<FieldPath, Integer> origJsonPathMap = origRec.cachedBuffer.jsonPathMap;
    Map<Integer, ByteBuffer> origCachedBufferMap = origRec.cachedBuffer.map;

    if (newJsonPathMap == null) {
      return origCachedBufferMap;
    }
    // The mapping from familyId in origJsonPathMap to familyId in newJsonPathMap
    Map<Integer, Integer> fidMap = new HashMap<Integer, Integer>();
    boolean jsonPathMapChanged = false;
    for (Entry<FieldPath, Integer> kv : origJsonPathMap.entrySet()) {
      Integer oldInt = kv.getValue();
      Integer newInt = newJsonPathMap.get(kv.getKey());

      if (newInt == null) {
        //new json map does not have this family
        jsonPathMapChanged = true;
        continue;
      }
      if (oldInt.compareTo(newInt) != 0) {
        jsonPathMapChanged = true;
      }
      fidMap.put(oldInt, newInt);
    }
    if (!jsonPathMapChanged) {
      return origCachedBufferMap;
    }

    Map<Integer, ByteBuffer> newCachedBufferMap = new LinkedHashMap<Integer, ByteBuffer>();
    for (Entry<Integer, ByteBuffer> kv : origCachedBufferMap.entrySet()) {
      Integer oldInt = kv.getKey();
      Integer newInt = fidMap.get(oldInt);
      if (newInt == null) {
        //new json map does not have this family, we should not use cachedBufferMap
        return null;
      }

      ByteBuffer buf = kv.getValue();
      newCachedBufferMap.put(newInt, buf);
    }
    return newCachedBufferMap;
  }

  /*
   * Encodes a given document into a set of Bytebuffers. One Bytebuffer
   * for each column family.
   * @param document : Ojai document which needs to be serialized
   * @param jsonPathMap : Its a mapping from the JSON path for the
   * column family to the familyId of the column family. The default
   * CF would be with path "".
   * @param useEncoded : A boolean flag that is used to select whether to
   * use cached, encoded buffer or serialize from the Document DOM.
   * @return - array of serialized family info. Info contains the action
   * and corresponding bytebuffer (if required) for each column family.
   * Order of family in the array would be same as the order of elements
   * in the input map
   */
  public static
  SerializedFamilyInfo[] encode(Document document,
                                Map <FieldPath, Integer> jsonPathMap,
                                boolean isBulkLoad,
                                boolean useCached) {
    DBDocumentImpl rec = getDBDocument(document);

    if (useCached && rec.cachedBuffer != null) {
      SerializationContext ctx = new SerializationContext();
      Map<Integer, ByteBuffer> cachedBufferMap = getUpdatedCachedBufferMap(
          jsonPathMap, rec);
      if (cachedBufferMap != null) {
        //We use cachedBufferMap only when we could find the family mapping on dst table
        return ctx.getSerializedBuffers(jsonPathMap, cachedBufferMap);
      }
    }

    if (rec.getNeedDOMStruct()) {
      rec.getDOMFromCachedBuffer();
    }

    SerializationContext ctx = new SerializationContext();
    ctx.setFullRecordOp(true);
    ctx.setStoreRowTS(isBulkLoad);
    ctx.serializeFamilies(rec, jsonPathMap);
    return ctx.getSerializedBuffers();
  }

  public static
  SerializedFamilyInfo[] encode(Document document,
                                Map <FieldPath, Integer> jsonPathMap) {
    return RowcolCodec.encode(document,
                              jsonPathMap,
                              false /*isBulkLoad*/,
                              false /*useEncoded*/);
  }

  /**
   * Encodes an input OJAI document into a byte array in rowcol format.
   * Currently it assumes that there is only one column family but in the future if
   * we add support for multiple column families, then this function would also take
   * a mapping from column name to column family name and return an array of Byte buffer,
   * one for each column family.
   * @param document OJAI document
   * @return an array of Byte buffer
   */
  public static ByteBuffer encode(Document document) {
    SerializedFamilyInfo [] familyInfo = encode(document,
                                                DEFAULT_FAMILY_MAP,
                                                false /*isBulkload*/,
                                                false /*useEncoded*/);
    assert(familyInfo.length == 1);
    return familyInfo[0].getByteBuffer();
  }

  public static
  void decodeInternal(Map <Integer, ByteBuffer> map,
                          Map <FieldPath, Integer> jsonPathMap,
                          Map <Integer, String> idToCFNameMap,
                          boolean insertionOrder,
                          boolean decodeTimestamp,
                          boolean preserveDeleteFlags,
                          DBDocumentImpl outDoc, String[] paths
                          ) {
    SerializationContext ctx = new SerializationContext();
    if (decodeTimestamp) {
      ctx.setDecodeTimestamp(decodeTimestamp);
    }
    ctx.setPreserveDeleteTime(preserveDeleteFlags);
    if (paths != null && paths.length > 0) {
      ctx.setProjectedPaths(paths);
    }
    ctx.decode(map, jsonPathMap, idToCFNameMap,
               insertionOrder, outDoc);

  }

  /* This will always decode the document */
  public static DBDocumentImpl decode(Map <Integer, ByteBuffer> map,
                                      Map <FieldPath, Integer> jsonPathMap,
                                      Map <Integer, String> idToCFNameMap) {

    DBDocumentImpl doc = new DBDocumentImpl();
    decodeInternal(map, jsonPathMap, idToCFNameMap,
                  false /*insertion order*/,
                  false /*decodeTimestamp*/,
                  false /*preserveDeleteFlags*/,
                  doc, null);
    return doc;
  }


  public static DBDocumentImpl decode(Map <Integer, ByteBuffer> map,
      Map <FieldPath, Integer> jsonPathMap, Map <Integer, String> idToCFNameMap,
      ByteBuffer rowKey, boolean excludeId, boolean cacheEncoded,
      boolean insertionOrder, boolean decodeTimestamp, boolean preserveDeleteFlags,
      String[] paths) {

    if ((rowKey != null) && (!rowKey.hasRemaining())) {
      // The first empty segment key. The rowKey has start position 0, and 0 remaining bytes.
      KeyValue kv = KeyValueBuilder.initFrom(EMPTY_ROWKEY);
      logger.info("create rowkey " + kv.getString() + " from buffer=("+rowKey+")");
      DBDocumentImpl doc = new DBDocumentImpl();
      doc.setId(kv, excludeId);
      return doc;
    }
    /*
     * if user has explicitly asked for decoding then decode it right away
     * else delay the decode
     */
    if (!cacheEncoded) {
      DBDocumentImpl doc = new DBDocumentImpl();
      doc.setId(IdCodec.decode(rowKey), excludeId);
      decodeInternal(map, jsonPathMap, idToCFNameMap,
                    insertionOrder, decodeTimestamp, preserveDeleteFlags, doc, paths);
      return doc;
    }

    DBDocumentImpl r = new DBDocumentImpl();
    r.setId(IdCodec.decode(rowKey), excludeId);
    r.setSerializedJson(map, jsonPathMap, idToCFNameMap,
        IdCodec.decode(rowKey), excludeId, insertionOrder, decodeTimestamp, preserveDeleteFlags, paths);
    return r;
  }

  /* this method will decode the document always */
  public static Document decode(ByteBuffer input) {
    return decode(input, null /*rowKey*/, true /*excludeId*/,
                  false /*decodeTimestamp*/, true /*preserveDeleteFlags*/);
  }


  /**
   * Decodes a serialized byte buffer and returns a document.
   * If the passed rowkey is not null, then it is included as the "_id"
   * key at the root level with type binary.<br/><br/>
   * This method will always decode the document.
   */

  public static Document decode(ByteBuffer input, ByteBuffer rowKey,
                                  boolean excludeId, boolean decodeTimestamp, boolean preserveDeleteFlags) {

    assert(input.order() == ByteOrder.LITTLE_ENDIAN);

    // If its an empty bytebuffer then return null
    if (input.position() == input.limit()) {
      return null;
    }

    Map <Integer, ByteBuffer> bufferMap = new HashMap<Integer, ByteBuffer>();
    ByteBuffer cachedBuffer = input.duplicate();
    cachedBuffer.mark();
    cachedBuffer.order(ByteOrder.LITTLE_ENDIAN);
    bufferMap.put(0, cachedBuffer);

    DBDocumentImpl doc = new DBDocumentImpl();
    doc.setId(IdCodec.decode(rowKey), excludeId);
    decodeInternal(bufferMap, DEFAULT_FAMILY_MAP, DEFAULT_NAME_MAP,
                  false /*restoreOrder*/,
                  decodeTimestamp, preserveDeleteFlags, doc, null);
    return doc;
  }

  public static DocumentMutation decodeMutation(ByteBuffer input, boolean needsRead) {
    return MutationImpl.fromSerializedValue(input, needsRead);
  }

}
