package com.mapr.db.rowcol;

import java.nio.ByteBuffer;

public class RootTimeDescriptor {

/*
 * RootTimeDescriptor contains information about the root document
 * base timestamp. It is 1 byte field with following information
 *
 * Each stored time contains two parts - Time in millisecond, and optional
 * uniqifier. Uniqifier is optional and is stored when same row gets multiple
 * update within same millisecond time. In most of the cases it would be zero.
 * It will not be store if its zero.
 *
 * bit 0 : 0 - Base timestamp is zero and would be set by the server
 *       : 1 - Base timestamp is valid and its 6byte abs time in millisec
 * bit 1 & 2
 *   0x0 : uniq is zero
 *   0x1 : uniq is 1 byte long
 *   0x2 : uniq is 2 byte long
 *   0x3 : uniq is 4 byte long
 *
 * bit 3 - 0 doc doesn't have any delete only keyvalue
 *         1 doc has delete only keyvalue
 *
 * bit 4-7 : Reserved - will be initialized to zero currently and will
 *           be used as version information in future
 *
 * If timestamp is non zero then it will follow the descriptor byte
 * if uniq is non zero size then it will follow next
 */

  // Version of rowcol format for json data
  // This is stored in the firstbyte of serialized data
  enum JsonRowColVersion {
    JSONROWCOLVERSION0
  };

  static final byte TimeStampShift = 0;
  static final byte TimeStampMask = 1 << TimeStampShift;
  static final byte UniqSizeShift = 1;
  static final byte UniqSizeMask = 0x3 << UniqSizeShift;
  static final byte HasDeletesShift  = 3;
  static final byte HasDeletesMask = (byte) (0x1 << HasDeletesShift);

  static final byte VersionShift = 4;
  static final byte VersionMask = (byte) (0xf << VersionShift);


  public static void serialize(DBDocumentImpl rec, ByteWriter w, SerializationContext ctx) {
    rec.rootTimeDescriptor = 0;
    if (ctx.storeRowTS()) {
      // For bulkload serialization we need to put the base row timestamp
      // in the rowcol format as it might get written directly to the spill file
      rec.rootTimeDescriptor = 1 << TimeStampShift;
      w.put(rec.rootTimeDescriptor);
      long time = System.currentTimeMillis();
      // Store the first 6 bytes in little endian format
      byte[] dummyBuf = new byte[6];
      for (int i = 0; i < 6; ++i) {
        dummyBuf[i] = (byte) ((time >> 8 * i) & 0xff);
      }
      w.put(dummyBuf);
    } else {
      if (rec.hasDeletes()) {
        rec.rootTimeDescriptor |= HasDeletesMask;
      }
      w.put(rec.rootTimeDescriptor);
    }
  }

  public static void deserialize(ByteBuffer input, SerializationContext ctx) {
    byte buf = input.get();
    long time = 0;
    int uniq = 0;
    byte[] dummyBuf = new byte[6];

    // If there is a valid rowtime in the document then read 6 bytes for time
    if (((buf & TimeStampMask) >> TimeStampShift) > 0) {
      input.get(dummyBuf);
      for (int i = 0; i < dummyBuf.length; ++i) {
        time |= ((long)(dummyBuf[i] & 0xff)) << (8 * i);
      }
    }

    byte uniqSize = (byte) ((buf & UniqSizeMask) >> UniqSizeShift);
    switch (uniqSize) {
    case 0: break;
    case 1: uniq = input.get(); break;
    case 2: uniq = input.getShort(); break;
    case 3: uniq = input.getInt(); break;
    }
    // Store the row base timestamp in the context
    // Other timestamps can be relative to this hence need to save it in the context
    ctx.setBaseTime(time, uniq);
    return;
  }

  public static void rewriteHasDeleteFlag(DBDocumentImpl rec, ByteWriter w, SerializationContext ctx) {
    if (rec.hasDeletes()) {
      rec.rootTimeDescriptor |= HasDeletesMask;
    } else {
      rec.rootTimeDescriptor &= (byte)~HasDeletesMask;
    }
    w.putByteAtOffset(0, rec.rootTimeDescriptor);
  }
}
