/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.rowcol;

import static com.mapr.db.impl.Constants.DEFAULT_FAMILY_MAP;
import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;
import static org.ojai.DocumentConstants.ID_KEY;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.ojai.Document;
import org.ojai.FieldPath;
import org.ojai.Value;
import org.ojai.annotation.API;
import org.ojai.store.DocumentMutation;
import org.ojai.store.MutationOp;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;

import com.mapr.db.impl.MapRDBTableImplHelper;
import com.mapr.db.rowcol.InsertContext.OpType;

@API.Internal
public class MutationImpl implements DocumentMutation {
  /* dbrecord to keep track of operations and rowcol encoding */
  private DBDocumentImpl record;
  /* if applying this operation on server requires a read-modify-write operation*/
  private boolean needsReadOnServer;

  // List of field paths which would require read on server
  private List <FieldPath> fields;

  // Template for set
  // set (String path, <type> value);
  // set (FieldPath path, <type> value);
  @API.Internal
  public MutationImpl() {
    record = new DBDocumentImpl();
    needsReadOnServer = false;
    fields = new ArrayList<FieldPath>();
  }

  /* throws exception if _id is being modified */
  private void checkForId(FieldPath path) {
    if (ID_KEY.equals(path.getRootSegment().getNameSegment().getName())) {
      throw new UnsupportedOperationException("DocumentMutation can't be used to change _id");
    }
  }

  private MutationImpl setCommon(FieldPath path, KeyValue value) {
    checkForId(path);
    fields.add(path);
    MutationHelper.mutationCommon(OpType.SET, record, path, value);
    /* set operation always requires a read on server to validate the type */
    needsReadOnServer = true;
    return this;
  }

  private MutationImpl setOrReplaceCommon(FieldPath path, KeyValue value) {
    boolean containsArray =
        MutationHelper.mutationCommon(OpType.SET_OR_REPLACE, record, path,value);
    // If it contains array then it will require a read on server even for
    // the setOrreplace op
    if (containsArray) {
      fields.add(path);
      needsReadOnServer = true;
    }
    return this;
  }

  private DocumentMutation appendCommon(FieldPath path, KeyValue value) {
    fields.add(path);
    MutationHelper.mutationCommon(OpType.APPEND, record, path, value);
    needsReadOnServer = true;
    return this;
  }

  private DocumentMutation mergeCommon(FieldPath path, KeyValue value) {
    fields.add(path);
    MutationHelper.mutationCommon(OpType.MERGE, record, path, value);
    needsReadOnServer = true;
    return this;
  }

  private MutationImpl incrementCommon(FieldPath path, KeyValue value) {
    fields.add(path);
    MutationHelper.mutationCommon(OpType.INCREMENT, record, path, value);
    needsReadOnServer = true;
    return this;
  }

  @Override
  public DocumentMutation empty() {
    record.empty();
    needsReadOnServer = false;
    fields = new ArrayList<FieldPath>();
    return this;
  }

  @Override
  public DocumentMutation setNull(String path) {
    return setNull(FieldPath.parseFrom(path));
  }

  @Override
  public DocumentMutation setNull(FieldPath path) {
    return setCommon(path, KeyValueBuilder.initFromObject(null));
  }

  @Override
  public DocumentMutation set(String path, Value v) {
    return set(FieldPath.parseFrom(path), v);
  }

  @Override
  public DocumentMutation set(FieldPath path, Value v) {
    return setCommon(path, KeyValueBuilder.initFrom(v));
  }

  @Override
  public DocumentMutation set(String path, boolean b) {
    return set(FieldPath.parseFrom(path), b);
  }

  @Override
  public DocumentMutation set(FieldPath path, boolean b) {
    return setCommon(path, KeyValueBuilder.initFrom(b));
  }

  @Override
  public DocumentMutation set(String path, short s) {
    return set(FieldPath.parseFrom(path), s);
  }

  @Override
  public DocumentMutation set(FieldPath path, short s) {
    return setCommon(path, KeyValueBuilder.initFrom(s));
  }

  @Override
  public DocumentMutation set(String path, byte b) {
    return set(FieldPath.parseFrom(path), b);
  }

  @Override
  public DocumentMutation set(FieldPath path, byte b) {
    return setCommon(path, KeyValueBuilder.initFrom(b));
  }

  @Override
  public DocumentMutation set(String path, int i) {
    return set(FieldPath.parseFrom(path), i);
  }

  @Override
  public DocumentMutation set(FieldPath path, int i) {
    return setCommon(path, KeyValueBuilder.initFrom(i));
  }

  @Override
  public DocumentMutation set(String path, long l) {
    return set(FieldPath.parseFrom(path), l);
  }

  @Override
  public DocumentMutation set(FieldPath path, long l) {
    return setCommon(path, KeyValueBuilder.initFrom(l));
  }

  @Override
  public DocumentMutation set(String path, float f) {
    return set(FieldPath.parseFrom(path), f);
  }

  @Override
  public DocumentMutation set(FieldPath path, float f) {
    return setCommon(path, KeyValueBuilder.initFrom(f));
  }

  @Override
  public DocumentMutation set(String path, double d) {
    return set(FieldPath.parseFrom(path), d);
  }

  @Override
  public DocumentMutation set(FieldPath path, double d) {
    return setCommon(path, KeyValueBuilder.initFrom(d));
  }

  @Override
  public MutationImpl set(String path, String value) {
    return set(FieldPath.parseFrom(path), value);
  }

  @Override
  public MutationImpl set(FieldPath path, String value) {
    return setCommon(path, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DocumentMutation set(String path, BigDecimal bd) {
    return set(FieldPath.parseFrom(path), bd);
  }

  @Override
  public DocumentMutation set(FieldPath path, BigDecimal bd) {
    return setCommon(path, KeyValueBuilder.initFrom(bd));
  }

  @Override
  public DocumentMutation set(String path, OTime t) {
    return set(FieldPath.parseFrom(path), t);
  }

  @Override
  public DocumentMutation set(FieldPath path, OTime t) {
    return setCommon(path, KeyValueBuilder.initFrom(t));
  }

  @Override
  public DocumentMutation set(String path, OTimestamp t) {
    return set(FieldPath.parseFrom(path), t);
  }

  @Override
  public DocumentMutation set(FieldPath path, OTimestamp t) {
    return setCommon(path, KeyValueBuilder.initFrom(t));
  }

  @Override
  public DocumentMutation set(String path, ODate d) {
    return set(FieldPath.parseFrom(path), d);
  }

  @Override
  public DocumentMutation set(FieldPath path, ODate d) {
    return setCommon(path, KeyValueBuilder.initFrom(d));
  }

  @Override
  public DocumentMutation set(String path, List<? extends Object> value) {
    return set(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation set(FieldPath path, List<? extends Object> value) {
    return setCommon(path, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DocumentMutation set(String path, OInterval intv) {
    return set(FieldPath.parseFrom(path), intv);
  }

  @Override
  public DocumentMutation set(FieldPath path, OInterval intv) {
    return setCommon(path, KeyValueBuilder.initFrom(intv));
  }

  @Override
  public DocumentMutation set(String path, ByteBuffer bb) {
    return set(FieldPath.parseFrom(path), bb);
  }

  @Override
  public DocumentMutation set(FieldPath path, ByteBuffer bb) {
    return setCommon(path, KeyValueBuilder.initFrom(bb));
  }

  @Override
  public DocumentMutation set(String path, Map<String, ? extends Object> value) {
    return set(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation set(FieldPath path, Map<String, ? extends Object> value) {
    return setCommon(path, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DocumentMutation set(String path, Document value) {
    return set(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation set(FieldPath path, Document value) {
    return setCommon(path, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DocumentMutation setOrReplace(String path, Value v) {
    return setOrReplace(FieldPath.parseFrom(path), v);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, Value v) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFromObject(v));
  }

  @Override
  public DocumentMutation setOrReplaceNull(String path) {
    return setOrReplaceNull(FieldPath.parseFrom(path));
  }

  @Override
  public DocumentMutation setOrReplaceNull(FieldPath path) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFromObject(null));
  }

  @Override
  public DocumentMutation setOrReplace(String path, boolean b) {
    return setOrReplace(FieldPath.parseFrom(path), b);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, boolean b) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(b));
  }

  @Override
  public DocumentMutation setOrReplace(String path, short s) {
    return setOrReplace(FieldPath.parseFrom(path), s);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, short s) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(s));
  }

  @Override
  public DocumentMutation setOrReplace(String path, byte b) {
    return setOrReplace(FieldPath.parseFrom(path), b);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, byte b) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(b));
  }

  @Override
  public DocumentMutation setOrReplace(String path, int i) {
    return setOrReplace(FieldPath.parseFrom(path), i);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, int i) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(i));
  }

  @Override
  public DocumentMutation setOrReplace(String path, long l) {
    return setOrReplace(FieldPath.parseFrom(path), l);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, long l) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(l));
  }

  @Override
  public DocumentMutation setOrReplace(String path, float f) {
    return setOrReplace(FieldPath.parseFrom(path), f);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, float f) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(f));
  }

  @Override
  public DocumentMutation setOrReplace(String path, double d) {
    return setOrReplace(FieldPath.parseFrom(path), d);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, double d) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(d));
  }

  @Override
  public MutationImpl setOrReplace(String path, String value) {
    return setOrReplace(FieldPath.parseFrom(path), value);
  }

  @Override
  public MutationImpl setOrReplace(FieldPath path, String value) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DocumentMutation setOrReplace(String path, BigDecimal bd) {
    return setOrReplace(FieldPath.parseFrom(path), bd);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, BigDecimal bd) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(bd));
  }

  @Override
  public DocumentMutation setOrReplace(String path, OTime t) {
    return setOrReplace(FieldPath.parseFrom(path), t);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, OTime t) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(t));
  }

  @Override
  public DocumentMutation setOrReplace(String path, OTimestamp t) {
    return setOrReplace(FieldPath.parseFrom(path), t);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, OTimestamp t) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(t));
  }

  @Override
  public DocumentMutation setOrReplace(String path, ODate d) {
    return setOrReplace(FieldPath.parseFrom(path), d);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, ODate d) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(d));
  }

  @Override
  public DocumentMutation setOrReplace(String path, List<? extends Object> value) {
    return setOrReplace(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, List< ? extends Object> value) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DocumentMutation setOrReplace(String path, OInterval intv) {
    return setOrReplace(FieldPath.parseFrom(path), intv);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, OInterval intv) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(intv));
  }

  @Override
  public DocumentMutation setOrReplace(String path, ByteBuffer bb) {
    return setOrReplace(FieldPath.parseFrom(path), bb);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, ByteBuffer bb) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(bb));
  }

  @Override
  public DocumentMutation setOrReplace(String path, Map<String, ? extends Object> value) {
    return setOrReplace(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, Map<String, ? extends Object> value) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DocumentMutation setOrReplace(String path, Document value) {
    return setOrReplace(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation setOrReplace(FieldPath path, Document value) {
    return setOrReplaceCommon(path, KeyValueBuilder.initFrom(value));
  }


  @Override
  public DocumentMutation append(String path, List<? extends Object> value) {
    return append(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation append(FieldPath path, List<? extends Object> value) {
    KeyValue kv = KeyValueBuilder.initFrom(value);
    appendCommon(path, kv);
    return this;
  }

  @Override
  public DocumentMutation append(String path, String value) {
    return append(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation append(FieldPath path, String  value) {
    KeyValue kv = KeyValueBuilder.initFrom(value);
    appendCommon(path, kv);
    return this;
  }

  @Override
  public DocumentMutation append(String path, byte[] value, int offset, int len) {
    return append(FieldPath.parseFrom(path), ByteBuffer.wrap(value, offset, len));
  }

  @Override
  public DocumentMutation append(FieldPath path, byte[] value, int offset, int len) {
    return append(path, ByteBuffer.wrap(value, offset, len));
  }

  @Override
  public DocumentMutation append(String path, byte[] value) {
    return append(FieldPath.parseFrom(path), ByteBuffer.wrap(value));
  }

  @Override
  public DocumentMutation append(FieldPath path, byte[] value) {
    return append(path, ByteBuffer.wrap(value));
  }

  @Override
  public DocumentMutation append(String path, ByteBuffer value) {
    return append(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation append(FieldPath path, ByteBuffer value) {
    return appendCommon(path, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DocumentMutation merge(String path, Document value) {
    return merge(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation merge(FieldPath path, Document value) {
    return mergeCommon(path, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DocumentMutation merge(String path, Map<String, Object> value) {
    return merge(FieldPath.parseFrom(path), value);
  }

  @Override
  public DocumentMutation merge(FieldPath path, Map<String, Object> value) {
    return mergeCommon(path, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DocumentMutation increment(FieldPath path, byte inc) {
    return incrementCommon(path, KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(String path, byte inc) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(FieldPath path, short inc) {
    return incrementCommon(path, KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(String path, short inc) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(inc));
  }

  @Override
  public MutationImpl increment(String path, int inc) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(FieldPath path, int inc) {
    return incrementCommon(path, KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(FieldPath path, long inc) {
    return incrementCommon(path, KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(String path, long inc) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(String path, float inc) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(FieldPath path, float inc) {
    return incrementCommon(path, KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(String path, double inc) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(FieldPath path, double inc) {
    return incrementCommon(path, KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(String path, BigDecimal inc) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(inc));
  }

  @Override
  public DocumentMutation increment(FieldPath path, BigDecimal inc) {
    return incrementCommon(path, KeyValueBuilder.initFrom(inc));
  }

  /**
   * Deletes the field at the given path.
   * If the field does not exist, the method will silently return without doing anything.
   * For example, if a.b.c is being asked to be deleted, and a.b is an array on the
   * server, then we will not delete it.
   */
  @Override
  public DocumentMutation delete(String path) {
    return delete(FieldPath.parseFrom(path));
  }

  @Override
  public DocumentMutation delete(FieldPath path) {
    checkForId(path);
    boolean readNeeded = MutationHelper.mutationCommon(OpType.DELETE, record,
        path, new KeyValue(Value.Type.NULL));
    if (readNeeded) {
      fields.add(path);
      needsReadOnServer = true;
    }
    return this;
  }

  /**
   * Indicates whether or not the processing of this mutation requires a read on
   * the server. If this function returns false then encoding from the mutation
   * can be sent in the fast part of put, delete, set_or_replace where it
   * can be batched and sent to the server.
   * For other operations, if any modified field contains an array index
   * or if the operation is one that requires verification on the server,
   * such as append, set, increment, or merge, this function will return true.
   */
  @API.Internal
  public boolean needsReadOnServer() {
    return needsReadOnServer;
  }

  /**
   * Performs rowcol serialization of the mutation. This is temporary
   * and will go away after the caller moves to use the multi-column family version
   * of this API.
   */
  @API.Internal
  public ByteBuffer rowcolSerialize() {
    SerializedFamilyInfo [] familyInfo = rowcolSerialize(DEFAULT_FAMILY_MAP);
    assert(familyInfo.length == 1);
    return familyInfo[0].getByteBuffer();
  }

  /**
   * Gives the rowcol encoding for multiple column families for a
   * given mutation operation.
   * @param jsonPathMap a mapping from the JSON path for the
   * column family to the familyId of the column family. The default
   * column family path will have the path ""
   * @return An array of serialized family information. Information includes the action
   * and corresponding bytebuffer (if required) for each column family.
   * The order of column families in the array will be same as the order of elements
   * in the input map.
   */
  @API.Internal
  public SerializedFamilyInfo[] rowcolSerialize(Map <FieldPath, Integer> jsonPathMap,
      boolean isBulkLoad) {
    SerializationContext ctx = new SerializationContext();
    ctx.setFullRecordOp(false);
    // for bulkload serialization store the base rowts
    // also in the rowcol format.
    ctx.setStoreRowTS(isBulkLoad);
    ctx.serializeFamilies(record, jsonPathMap);
    return ctx.getSerializedBuffers();
  }

  /*
   * Returns the fields which requires read on server for this mutation.
   * Return value is a map with key as family Id and value as list of paths in that family
   * For non default family the paths are prefixed with "v".
   * Input - the map from each field path to the family id.
   */
  @API.Internal
  public Map<Integer, List <String>>
    getFieldsNeedRead(Map <FieldPath, Integer> pathCFidMap) {
    if (fields.size() > 0) {
      String[] strFields = new String[fields.size()];
      for (int i = 0; i < fields.size(); ++i) {
        strFields[i] = fields.get(i).asPathString();
      }

      Map<Integer, List<String>> pathsPerFam =
          MapRDBTableImplHelper.getMultipleCFQualifiers(pathCFidMap, false /*excludeEmbeddedFamily*/, strFields);

      class StringPathComparator implements Comparator<String> {
        @Override
        public int compare(String o1, String o2) {
          if (o1 == o2) return 0;
          if (o1 == null) return -1;
          if (o2 == null) return 1;
          return FieldPath.parseFrom(o1).compareTo(FieldPath.parseFrom(o2));
        }
      }

      if (pathsPerFam != null) {
        for (Map.Entry<Integer, List<String>> e : pathsPerFam.entrySet()) {
          Collections.sort(e.getValue(), new StringPathComparator());
        }
      }
      return pathsPerFam;
    }
    return null;
  }

  @API.Internal
  public SerializedFamilyInfo[] rowcolSerialize(Map <FieldPath, Integer> jsonPathMap) {
    return this.rowcolSerialize(jsonPathMap, false);
  }

  @API.Internal
  public static MutationImpl fromSerializedValue(ByteBuffer input, boolean needsRead) {
    MutationImpl rm = new MutationImpl();
    rm.needsReadOnServer = needsRead;
    rm.record = (DBDocumentImpl) RowcolCodec.decode(input);
    return rm;
  }

  @Override
  public Iterator<MutationOp> iterator() {
    throw new UnsupportedOperationException("");
  }

  @Override
  public DocumentMutation decrement(FieldPath path, byte dec) {
    return incrementCommon(path, KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(String path, byte dec) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(FieldPath path, short dec) {
    return incrementCommon(path, KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(String path, short dec) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(String path, int dec) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(FieldPath path, int dec) {
    return incrementCommon(path, KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(FieldPath path, long dec) {
    return incrementCommon(path, KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(String path, long dec) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(String path, float dec) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(FieldPath path, float dec) {
    return incrementCommon(path, KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(String path, double dec) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(FieldPath path, double dec) {
    return incrementCommon(path, KeyValueBuilder.initFrom(-dec));
  }

  @Override
  public DocumentMutation decrement(String path, BigDecimal dec) {
    return incrementCommon(FieldPath.parseFrom(path), KeyValueBuilder.initFrom(dec.negate()));
  }

  @Override
  public DocumentMutation decrement(FieldPath path, BigDecimal dec) {
    return incrementCommon(path, KeyValueBuilder.initFrom(dec.negate()));
  }

}
