/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.rowcol;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.zip.Checksum;

import org.ojai.Value;
import org.ojai.annotation.API;
import org.ojai.util.Values;

import com.mapr.db.impl.Constants;
//import com.mapr.db.mapreduce.tools.impl.DiffTableUtils;

public class KeyValueWithTS extends KeyValue {
  /*
   * DBRecord and DBList class will be subclass of this class hence
   * we can return those classes directly when needed as KeyValue
   */

  // byte containing flag about the create / delete / update of the value

  TimeAndUniq times[];
  TimeAndUniq arrayIndex;
  byte [] arrayIndexUniq;


  public KeyValueWithTS() {

  }
  public KeyValueWithTS(Type type) {
    super(type);
  }

  // Map and List overrides this function.
  // This is called only for base types for which tostring
  // is same as toStringWithTimestamp
  @Override
  public String toStringWithTimestamp() {
    return Values.asJsonString(this);
  }


  /*
   * compares two maprdb record implementations for equality
   * returns true if they are equal (have same number of fields
   * and also same keyname and value).
   * matchiTime - if this is set to true then timedescriptor is also matched
   * keepOnlyMismatch - If this is true then r1 and r2 are modified to keep
   * only the mismatching entries -- rest all are removed.
   *
   */
  public static boolean equals(
      KeyValue kv1, KeyValue kv2,
      boolean matchTime,
      boolean keepOnlyMismatch,
      TimeAndUniq parentDelTime1,
      TimeAndUniq parentDelTime2) {

    TimeAndUniq delTime1 = null;
    TimeAndUniq delTime2 = null;

    boolean matched = true;
    if ((kv1 == null) && (kv2 == null)) {
      return true;
    } else if (kv1 == null || kv2 == null) {
      return false;
    } else if (kv1.type_ != kv2.type_) {
      // If the types are not same then return false and terminate
      return false;
    }

    if (kv1 instanceof DBDocumentImpl) {
      DBDocumentImpl doc1 = (DBDocumentImpl) kv1;
      DBDocumentImpl doc2 = (DBDocumentImpl) kv2;
      doc1.setMapOnDemand();
      doc2.setMapOnDemand();
    }

    assert(kv1.isRoot() == kv2.isRoot());

    if (matchTime) {
      KeyValueWithTS kvTS1 = (KeyValueWithTS)kv1;
      KeyValueWithTS kvTS2 = (KeyValueWithTS)kv2;
      for (int i = 0; i < 2; ++i) {
        TimeAndUniq t1 = kvTS1.times[i];
        TimeAndUniq t2 = kvTS2.times[i];
        if (TimeAndUniq.equals(t1, t2) == false) {
          matched = false;
        }
      }

      /* for deltime check if the parent deltime is more than
       * or equal to my deltime. If it is then consider my del
       * time as parentDelTime
       */
      delTime1 = kvTS1.times[2];
      delTime2 = kvTS2.times[2];
      if (parentDelTime1 != null &&
          TimeAndUniq.cmp(parentDelTime1, delTime1) >= 0) {
        delTime1 = parentDelTime1;
      }
      if (parentDelTime2 != null &&
          TimeAndUniq.cmp(parentDelTime2, delTime2) >= 0) {
        delTime2 = parentDelTime2;
      }
      if (TimeAndUniq.equals(delTime1, delTime2) == false) {
        matched = false;
      }
    }

    assert(kv1.isArrayElement() == kv2.isArrayElement());
    Type type = kv1.getType();

    // document type of keyvalues
    if (type == Value.Type.MAP) {
      Map <String, KeyValue> map1 = ((DBDocumentImpl) kv1).map;
      Map <String, KeyValue> map2 = ((DBDocumentImpl) kv2).map;

      KeyValue child1 = null;
      KeyValue child2 = null;
      String key1 = null;
      String key2 = null;
      Iterator<Map.Entry<String, KeyValue>> iter1 = map1.entrySet().iterator();
      Iterator<Map.Entry<String, KeyValue>> iter2 = map2.entrySet().iterator();

      while (iter1.hasNext() || iter2.hasNext()) {
        if ((child1 == null) && iter1.hasNext()) {
          Map.Entry<String, KeyValue> e1 = iter1.next();
          child1 =  e1.getValue();
          key1 = e1.getKey();
        }

        if ((child2 == null) && iter2.hasNext()) {
          Map.Entry<String, KeyValue> e2 = iter2.next();
          child2 = e2.getValue();
          key2 = e2.getKey();
        }

        /* check for matching key */
        if (!Objects.equals(key1, key2)) {
          return false;
        }

        assert((child1 != null) || (child2 != null));
        if ((child1 == null) || (child2 == null)) {
          return false;
        }

          boolean childMatched =
              KeyValueWithTS.equals(child1, child2, matchTime, keepOnlyMismatch,
                              delTime1, delTime2);
          // If child1 and child2 matched and we need to keep only mismatch
          // then remove them from the map
          if (childMatched && keepOnlyMismatch) {
            iter1.remove();
            iter2.remove();
          }
          if (!childMatched) {
            matched = false;
          }
          child1 = child2 = null;
          key1 = key2 = null;
      }
      return matched;
    }

    // array type of keyvalues
    if (type == Value.Type.ARRAY) {
      List <KeyValue> list1 = ((DBList) kv1).list;
      List <KeyValue> list2 = ((DBList) kv2).list;

      KeyValue child1 = null;
      KeyValue child2 = null;
      Iterator<KeyValue> iter1 = list1.iterator();
      Iterator<KeyValue> iter2 = list2.iterator();

      while (iter1.hasNext() || iter2.hasNext()) {
        if ((child1 == null) && iter1.hasNext()) {
          child1 =  iter1.next();
        }

        if ((child2 == null) && iter2.hasNext()) {
          child2 = iter2.next();
        }

        assert((child1 != null) || (child2 != null));
        if ((child1 == null) || (child2 == null)) {
          return false;
        }

        // compare their key names
        TimeAndUniq index1 = ((KeyValueWithTS)child1).arrayIndex;
        TimeAndUniq index2 = ((KeyValueWithTS)child2).arrayIndex;
        byte[] index1Uniq = ((KeyValueWithTS)child1).arrayIndexUniq;
        byte[] index2Uniq = ((KeyValueWithTS)child2).arrayIndexUniq;
        int cmp = ArrayIndexDescriptor.compareIndexTimeAndUniq(index1, index1Uniq, index2, index2Uniq);
        if (cmp == 0) {
          boolean childMatched =
              KeyValueWithTS.equals(child1, child2, matchTime, keepOnlyMismatch,
                              delTime1, delTime2);
          // If child1 and child2 matched and we need to keep only mismatch
          // then remove them from the map
          if (childMatched && keepOnlyMismatch) {
            iter1.remove();
            iter2.remove();
          }
          if (!childMatched) {
            matched = false;
          }
          child1 = child2 = null;
        } else if (cmp < 0) {
          // kv1 key is less than kv2 key
          matched = false;
          child1 = null;
        } else {
          // kv1 key is more than kv2 key
          matched = false;
          child2 = null;
        }
      }
      return matched;
    }
    // This is non container type so just match their values
    return kv1.equals(kv2);
  }

  @API.Internal
  public static void updateChecksumKeyValue(Checksum checksum, KeyValue kv, boolean includeTime, TimeAndUniq parentDelTime) {

    if (kv == null) {
      return;
    }
    if (kv instanceof DBDocumentImpl) {
      DBDocumentImpl doc = (DBDocumentImpl) kv;
      doc.setMapOnDemand();
    }

    if (kv.isRoot()) {
      checksum.update(Constants.ISROOT_CHECKSUM);
    } else {
      checksum.update(Constants.ISNOTROOT_CHECKSUM);
    }

    TimeAndUniq delTime = null;
    if (includeTime) {
      KeyValueWithTS kvTS = (KeyValueWithTS)kv;
      //The length of TimeAndUniq can only be 3
      TimeAndUniq[] tau = kvTS.times;
      for (int i = 0; i < 2; ++i) {
        TimeAndUniq t = tau[i];
        byte[] tsbytes = ByteBuffer.allocate(Long.SIZE).putLong(t.time()).array();
        checksum.update(tsbytes, 0, Long.SIZE);
        checksum.update(t.uniq());
      }

      delTime = tau[2];
      if (parentDelTime != null &&
          TimeAndUniq.cmp(parentDelTime, delTime) >= 0) {
        delTime = parentDelTime;
      }
      byte[] dtsbytes = ByteBuffer.allocate(Long.SIZE).putLong(delTime.time()).array();
      checksum.update(dtsbytes, 0, Long.SIZE);
      checksum.update(delTime.uniq());
    }

    Type type = kv.getType();
    checksum.update(type.ordinal());

    // document type of keyvalues
    if (type == Value.Type.MAP) {
      checksum.update(Constants.MAP_BEGIN_CHECKSUM);
      Map <String, KeyValue> map = ((DBDocumentImpl) kv).map;

      KeyValueWithTS child = null;
      //The map will be same order for logically same document
      Iterator<Map.Entry<String, KeyValue>> iter = map.entrySet().iterator();

      while (iter.hasNext()) {
        Entry<String, KeyValue> mapEntry = iter.next();
        child =  (KeyValueWithTS)mapEntry.getValue();
        assert(child != null);

        //String sk = child.getKey();
        String sk = mapEntry.getKey();
        checksum.update(sk.length());
        checksum.update(sk.getBytes(), 0, sk.length());
        updateChecksumKeyValue(checksum, child, includeTime, delTime);
      }
      checksum.update(Constants.MAP_END_CHECKSUM);
    }

    // array type of keyvalues
    if (type == Value.Type.ARRAY) {
      checksum.update(Constants.ARRAY_BEGIN_CHECKSUM);
      List <KeyValue> list = ((DBList) kv).list;
      KeyValueWithTS child = null;
      Iterator<KeyValue> iter = list.iterator();

      while (iter.hasNext()) {
        child =  (KeyValueWithTS)iter.next();
        assert(child != null);

        // compare their key names
        TimeAndUniq index = child.arrayIndex;
        byte[] indexUniq = child.arrayIndexUniq;
        updateChecksumIndexTimeAndUniq(checksum, index, indexUniq);
        updateChecksumKeyValue(checksum, child, includeTime, delTime);
      }
      checksum.update(Constants.ARRAY_END_CHECKSUM);
    }
    updateChecksumPrimaryType(checksum, kv);

  }

  private static void updateChecksumIndexTimeAndUniq(
      Checksum checksum, TimeAndUniq tau, byte[] uniq) {

    byte[] tsbytes = ByteBuffer.allocate(Long.SIZE).putLong(tau.time()).array();
    checksum.update(tsbytes, 0, Long.SIZE);
    checksum.update(tau.uniq());

    if (uniq != null) {
      checksum.update(uniq, 0, uniq.length);
    }
  }

  public static void updateChecksumPrimaryType(Checksum checksum, KeyValue kv) {
    if (kv == null) {
      return;
    }
    Type type = kv.getType();
    checksum.update(type.ordinal());
    switch (type) {
      case BOOLEAN:
      case BYTE:
      case SHORT:
      case INT:
      case LONG:
      case FLOAT:
      case DOUBLE:
      case DATE:
      case TIMESTAMP:
      case TIME:
      case INTERVAL:
        byte[] pvbytes = ByteBuffer.allocate(Long.SIZE).putLong(kv.primValue).array();
        checksum.update(Long.SIZE);
        checksum.update(pvbytes, 0, Long.SIZE);
        return;
      case NULL:
        checksum.update(Integer.SIZE);
        checksum.update(Constants.VALUENULL_CHECKSUM);
        return;

      case BINARY:
        ByteBuffer bf = ((ByteBuffer) kv.objValue).duplicate();
        checksum.update(bf.limit());
        checksum.update(bf.array(), 0, bf.limit());
        return;
      case DECIMAL:
        BigDecimal bd = kv.getDecimal();
        byte[] usvbytes = bd.unscaledValue().toByteArray();
        checksum.update(usvbytes.length);
        checksum.update(usvbytes, 0, usvbytes.length);
        checksum.update(bd.scale());
        return;

      case STRING:
        String str = kv.getString();
        checksum.update(str.length());
        checksum.update(str.getBytes(), 0, str.length());
        return;

      case MAP:
      case ARRAY:
      default:
        assert(false);
    }
  }

  /*
   * makes a shallow copy of the KeyValue
   * It will copy the fields which are not dependent on
   * which tree this key value belongs to
   * Caller needs to recalculate the size of key value
   * after inserting it into the new tree
   */
  @Override
  public KeyValue shallowCopy() {
    return this.clone();
  }

  @Override
  public KeyValueWithTS clone() {
    return (KeyValueWithTS)super.clone();
  }

}
