package com.mapr.db.rowcol;

import org.ojai.exceptions.TypeException;

import com.mapr.org.apache.hadoop.hbase.util.Bytes;

/**
 * A helper class to calculate estimated size of different types of key values
 */
public class KeyValueSizeHelper {

  /* 
   * based on the type of keyvalue return the number of bytes for the value
   * For array and map we return INT_MAX as the size here so that the caller
   * reserves four bytes to encode their size. The actual size would be calculated
   * by the caller after encoding the children and then it will write the size
   * at appropriate place in the output buffer.
   */
  static int getValueSize(KeyValue kv) {
    switch (kv.getType()) {
    case MAP: 
    case ARRAY: return Integer.MAX_VALUE;
    case BINARY: {
      return kv.getBinaryInternal().remaining();
    }
    case BOOLEAN: return 0;
    case BYTE: return 1;
    case DATE: return varLongSize(kv);
    case DECIMAL: return BigDecimalSizeDescriptor.getSerializedSize(kv.getDecimal());
    case DOUBLE: return 8;
    case FLOAT: return 4;
    case INT: return 4;
    case INTERVAL: return varLongSize(kv);
    case LONG: return 8;
    case NULL: return 0;
    case SHORT: return 2;
    // FIXME: bad fix, we are decoding string twice
    case STRING: return Bytes.toBytes(kv.getString()).length;
    case TIME: return varLongSize(kv);
    case TIMESTAMP: return varLongSize(kv);
    }
    throw new TypeException("Unknown type " + kv.getType());

  }

  /* returns the serialized size of the long value */
  private static int varLongSize(KeyValue kv) {
    long v = kv.getPrimValue();
    if (v < 0) {
      return 8;
    } else if (v <= 0xff) {
      return 1;
    } else if (v <= 0xffff) {
      return 2;
    } else if (v <= 0xffffffffL) {
      return 4;
    } else {
      return 8;
    }
  }
}
