package com.mapr.db.rowcol;

import java.nio.ByteBuffer;

import org.ojai.Value;

/*
 * Helper class to encode and decode key value type of elements
 * Each is given 4 bits to encode their size a
 */
public class KeyValueSizeDescriptor {
  static int varIntSize(int value) {
    if (value < 0) {
      return 4;
    }
    if (value <= 0xff) {
      return 1;
    } else if (value <= 0xffff) {
      return 2;
    } else {
      return 4;
    }
  }
  /*
   * returns the number of additional bytes needed to store the key size
   * If the keySize can be stored inline then it returns 0 otherwise it will
   * return the number of additional bytes to store size of size.
   * numBitsForValue is the maximum size of value that can be inline
   * for some types it can be zero as well if the size is decided based on the type
   */
  public static int getKeySizeOfSize(int numBitsForValue, int keySize) {

    if (numBitsForValue == 8) {
      return 0;
    }
    int maxInlineKeySize = (1 << (8 - numBitsForValue)) - 4;
    if (keySize <= 0) {
      return 0;
    }
    if (keySize < maxInlineKeySize) {
      return 0;
    }
    return varIntSize(keySize);
  }

  public static int getValueSizeOfSize(int numBitsForValue, int valueSize) {
    if ((numBitsForValue == 0) || (valueSize == 0)) {
      return 0;
    }
    int maxInlineValueSize = (1 << numBitsForValue) - 4;
    if (valueSize < maxInlineValueSize) {
      return 0;
    }
    return varIntSize(valueSize);
  }

  /*
   * Encodes the keysize and valuesize in the output buffer
   * it would take space of 1 + getKeySizeOfSize() + getValueSizeOfSize() bytes
   */
  public static int encodeKeyValueSize(int numBitsForValue,
                                       int keySizeOfSize,
                                       int valueSizeOfSize,
                                       int keySize, int valueSize,
                                       ByteWriter output) {
    int sizeDesc = 0;
    /*
     * if the key size is stored inline then store
     * 8 - numBitsForValue in sizedesc
     */
    if (keySizeOfSize == 0) {
      sizeDesc = keySize;
    } else {
      sizeDesc = (1 << (8 - numBitsForValue)) - 4 - 1 + keySizeOfSize;
    }

    if (valueSizeOfSize == 0) {
      sizeDesc |= (valueSize << (8 - numBitsForValue)) & 0xff;
    } else {
      sizeDesc |= ((1 << numBitsForValue) - 4 - 1 + valueSizeOfSize) << (8 - numBitsForValue);
    }

    output.put((byte) sizeDesc);
    encodeVarInt(keySize, keySizeOfSize, output);
    int valueOffset = output.position();
    encodeVarInt(valueSize, valueSizeOfSize, output);
    // Return the value offset so that caller can put the size
    // later on in the encoded stream. This is needed forthe 
    // map and array type of values where the size is not
    // known until the children are also encoded. So we always
    // have the 4 bytes saved for such type of values
    return valueOffset;
  }

  public static void encodeVarInt(int value, int numBytes, ByteWriter output) {
    switch (numBytes) {
    case 0: break;
    case 1: output.put((byte) value); break;
    case 2: output.putShort((short) value); break;
    default : output.putInt(value);
    }
  }

  public static int readVarInt(int numBytes, ByteBuffer input) {
    switch (numBytes) {
    case 0: return 0;
    case 1: return input.get() & 0xff;
    case 2: return input.getShort()  & 0xffff;
    default : return input.getInt();
    }
  }

  /*
   * reads the required number of bytes from buffer and returns
   * an array of integers output[0] = keysize and output[1] = valuesize
   */
  public static int[] decodeKeyValueSize(int numBitsForValue, ByteBuffer b) {
    int output[] = new int[2];
    int sizeDesc = b.get() & 0xff;
    int keySize = sizeDesc & ((1 << (8 - numBitsForValue)) - 1);
    int valueSize = (sizeDesc >> (8 - numBitsForValue)) & 0xff;
    int maxInlineKeySize = (1 << (8 - numBitsForValue)) - 4;
    int maxInlineValueSize = (1 << numBitsForValue) - 4;

    // special case when all the bits are used for key and value size is 0
    if (maxInlineKeySize < 0) {
      maxInlineKeySize = 1;
    }
    if (maxInlineValueSize < 0) {
      maxInlineValueSize = 1 << numBitsForValue;
    }

    if (keySize < maxInlineKeySize) {
      output[0] = keySize;
    } else {
      output[0] = readVarInt(keySize - maxInlineKeySize + 1, b);
    }

    if (valueSize < maxInlineValueSize) {
      output[1] = valueSize;
    } else {
      output[1] = readVarInt(valueSize - maxInlineValueSize + 1, b);
    }
    return output;
  }

  /*
   * returns the number of bits required to store the size of value
   * for the given type. For fixed length entries (like boolean, byte, short, long)
   * etc it returns 0 as the type itself gives the size information
   */

  public static int valueSizeBitsForType(Value.Type type) {
    switch (type) {
    // For boolean we store the value also in the size desc hence take 1 bit
    case BOOLEAN : return 1;
    case BYTE :
    case SHORT :
    case INT :
    case LONG :
    case FLOAT :
    case DOUBLE :
    case NULL :
      return 0;

      /*
       * for other types return 4 as we share 4 bits each for key and value
       */
    default : return 4;
    }
  }
 }
