package com.mapr.db.rowcol;

import java.nio.ByteBuffer;
import java.util.Map;
import java.util.TreeMap;

import org.ojai.Value.Type;
import org.ojai.exceptions.TypeException;

import com.mapr.db.rowcol.ArrayIndexDescriptor.ArrayIndexType;
import com.mapr.db.rowcol.InsertContext.OpType;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

/*
 * Class to help with the serialization of keyvalue into the byte buffer
 * the output bytebuffer should be preallocated with estimated serialized
 * size and should also be configured in the little endian ordering
 */
public class KeyValueSerializeHelper {

  /*
   * It serializes the keyvalue into the output byte buffer.
   */
  public static void serialize(String k, KeyValue kv, KeyValue parentKv, int indexInParent,
                               SerializationContext context, ByteWriter w) {
    // write the type of field first
    int startOffset = w.position();
    byte [] keyNameBytes = null;
    /* encode the optype also in the higher three bits of the type */
    byte type = RowcolType.rowColType(kv.getType());
    type |= kv.opType << RowcolType.OPTYPE_SHIFT;
    w.put(type);

    int numBitsForValue;
    int keySize;
    if (kv.isArrayElement()) {
      keySize = 0;
      numBitsForValue = 8;
    } else {
      keyNameBytes = Bytes.toBytes(k);
      keySize = keyNameBytes.length;
      numBitsForValue = KeyValueSizeDescriptor.valueSizeBitsForType(kv.getType());
    }
    int valueSize = KeyValueSizeHelper.getValueSize(kv);
    int keySizeOfSize = KeyValueSizeDescriptor.getKeySizeOfSize(numBitsForValue, keySize);
    int valueSizeOfSize = KeyValueSizeDescriptor.getValueSizeOfSize(numBitsForValue, valueSize);
    // if the type is boolean than set the value size equal to
    // the boolean value - this will result in storing the
    // boolean value inline with the sizedesc
    if (kv.getType() == Type.BOOLEAN) {
      valueSize = kv.getBoolean() ? 1 : 0;
    }
    int valueOffset;
    valueOffset =
        KeyValueSizeDescriptor.encodeKeyValueSize(numBitsForValue, keySizeOfSize, valueSizeOfSize, keySize, valueSize, w);

    // For array and map types we should have encoded 4 bytes for the valuesize
    assert((!kv.isContainerType()) || (valueSize == Integer.MAX_VALUE));

    TimeDescriptor.serialize(kv,  w);
    if (kv.isArrayElement() == false) {
      serializeOrderOfField(kv, w);
      // serialize the key name
      w.put(keyNameBytes);
    } else {
      DBList l = (DBList) parentKv;
      if (l.IsAbsoluteIndexType()) {
        ArrayIndexDescriptor.serialize(ArrayIndexType.ARRAY_INDEX_TYPE_ABSOLUTE,
            indexInParent, l.map.size(), w);
      } else {
        ArrayIndexDescriptor.serialize(ArrayIndexType.ARRAY_INDEX_TYPE_ASSOCIATIVE,
            indexInParent, l.list.size(), w);
      }
    }
    // serialize the value
    serializeValue(kv, valueSize, context, w);

    /* for map and array store the real size now */
    if (kv.isContainerType()) {
      valueSize = w.position() - startOffset;
      w.putAtOffset(valueOffset, valueSize);
    }
  }

  private static void serializeOrderOfField(KeyValue kv, ByteWriter w) {
    int keySize = kv.getOrderOfField();
    int keySizeOfSize = KeyValueSizeDescriptor.getKeySizeOfSize(0, keySize);
    KeyValueSizeDescriptor.encodeKeyValueSize(0 /*numBitsForValue*/, keySizeOfSize,
                                              0 /*valueSizeOfSize*/, keySize,
                                              0 /*valueSize*/, w);
  }

  /*
   * Serializes the value into the output buffer
   */
  private static void serializeValue(KeyValue kv, int estimatedValueSize,
                                     SerializationContext context, ByteWriter w) {
    /*
     * if the op type is not none then set flag
     * in the context to indicate this
     */
    if (kv.opType != OpType.NONE.ordinal()) {
      context.setHasMutation(true);
    }

    switch (kv.getType()) {
    case ARRAY: serializeArrayValue(kv, context, w); return;
    case BINARY: {
      ByteBuffer b = kv.getBinary();
      // don't encode zero size buffer
      if (b.remaining() <= 0) return;
      int pos = b.position();
      w.put(b);
      b.position(pos);
      return;
    }
    case NULL: {
      /*
       * if create time is not valid and delete time is valid
       * then this row contains delete. Set the flag in context
       */
      if ((TimeDescriptor.isCreateTimeValid(kv) == false) &&
          (TimeDescriptor.isDeleteTimeValid(kv) == true)) {
        context.setHasDeletes(true);
      }
      return;
    }
    case BOOLEAN: return;

    case BYTE: w.put(kv.getByte()); return;

    case DECIMAL:
      BigDecimalSizeDescriptor.serialize(kv.getDecimal(), w);
      return;

    case DOUBLE:
    case LONG: w.putLong(kv.getPrimValue()); return;

    case FLOAT:
    case INT:
      w.putInt((int) (kv.getPrimValue() & 0xffffffffL)); return;

    case MAP: serializeMap(kv, context, w); return;
    case SHORT: w.putShort(kv.getShort()); return;
    case STRING:
      byte[] b = Bytes.toBytes(kv.getString());
      w.put(b);
      return;

    case INTERVAL:
    case DATE:
    case TIME:
    case TIMESTAMP:
      serializeVarLong(kv, estimatedValueSize, w); return;
    }
    throw new TypeException("Unknown type " + kv.getType());
  }

  private static void serializeMap(KeyValue kv,
          SerializationContext context, ByteWriter w) {
    DBDocumentImpl rec = (DBDocumentImpl) kv;
    /*
     * Put the order of field in each keyvalue now.
     */
    Map <String, KeyValue> sortedMap = new TreeMap<String, KeyValue>();
    int i = 0;
    for (Map.Entry<String, KeyValue> e : rec.map.entrySet()) {
      /*
       * if the child is root of another cf then
       * skip its serialization in current serialization
       * we need to check flag on child as well in the contex
       * because the same child can be processed by two diff
       * threads with different cf structure. Flag on the child
       * is just hint while the flag in the context is confirmation.
       */
      String k = e.getKey();
      KeyValue child = e.getValue();
      if (child.isRootOfColumnFamily() && context.isFamilyRoot(child)) {
        ++i;
        continue;
      }
      child.setOrderOfField(i);
      sortedMap.put(k, child);
      ++i;
    }

    // need to pass both key and value separately if we want to get rid of key field from value
    // implementations.
    for (Map.Entry<String, KeyValue> e : sortedMap.entrySet()) {
      KeyValue child = e.getValue();
      KeyValueSerializeHelper.serialize(e.getKey(), child, rec, child.getOrderOfField(), context, w);
    }
    w.put((byte) RowcolType.END_OF_FIELD_TYPE);
  }

  private static void serializeVarLong(KeyValue kv, int estimatedValueSize, ByteWriter w) {
    long v = kv.getPrimValue();
    switch (estimatedValueSize) {
    case 0 : return;
    case 1 : w.put((byte) (v & 0xff)); return;
    case 2 : w.putShort((short) (v & 0xffff)); return;
    case 4 : w.putInt((int) v & 0xffffffff); return;
    case 8 : w.putLong(v); return;
    default : throw new IllegalArgumentException("Invalid size in serialize " + estimatedValueSize);
    }
  }

  private static void serializeArrayValue(KeyValue kv,
          SerializationContext context, ByteWriter w) {
    /* if the array is of type absolute then access its elements using map */
    DBList list = (DBList) kv;

    if (list.IsAbsoluteIndexType()) {
      for (int index : list.map.keySet()) {
        KeyValueSerializeHelper.serialize(null, list.map.get(index), list, index, context, w);
      }
    } else {
      for (int index = 0; index < list.list.size(); ++index) {
        // Don't insert NULL values as it can be NULL if the projection reads only
        // specific fields of an array. For example, read of a[2] will leave a[0] and
        // a[1] set as NULL in the input record. If someone writes it back then serialization
        // should skip such element
        if (list.list.get(index) != null) {
          KeyValueSerializeHelper.serialize(null, list.list.get(index), list, index, context, w);
        }
      }
    }
    w.put((byte) RowcolType.END_OF_FIELD_TYPE);
  }
}
