package com.mapr.db.rowcol;

import java.nio.ByteBuffer;

import org.ojai.FieldPath;
import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.exceptions.TypeException;

import com.mapr.db.rowcol.ArrayIndexDescriptor.ArrayIndexType;
import com.mapr.db.rowcol.InsertContext.OpType;
import com.mapr.db.rowcol.SerializationContext.ProjectionState;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

/*
 * Class to deserialize the keyvalue from the rowcol encoded bytebuffer
 */
public class KeyValueDeserializeHelper {

  public static Value.Type getType(ByteBuffer input) {
    byte rowColType = input.get();
    if (rowColType == RowcolType.END_OF_FIELD_TYPE) {
      return null;
    }

    /* use only the bottom 5 bits for the type */
    Value.Type type = RowcolType.valueType(rowColType & RowcolType.TYPE_MASK);
    return type;
  }

  public static void deserializeWithoutKeyValue(SerializationContext context,
            ByteBuffer input) {
    if (context.isRoot()) {
      context.setNewRecord(false);
      RootTimeDescriptor.deserialize(input, context);
      TimeDescriptor.deserialize(null /*keyValue*/, input, context);
    }
    Value.Type type = getType(input);
    context.setType(type);
    if (type == null) {
      return;
    }
    int numBitsForValue;
    if (context.isArrayElement()) {
      numBitsForValue = 8;
    } else {
      numBitsForValue = KeyValueSizeDescriptor.valueSizeBitsForType(type);
    }
    context.setNumBitsForValue(numBitsForValue);
    context.setKeyValueSize(input);
    TimeDescriptor.deserialize(null /*keyValue*/, input, context);
    if (context.isArrayElement()) {
      ArrayIndexDescriptor.deserialize(null, input, context);
    } else {
      int[] kvOrder = KeyValueSizeDescriptor.decodeKeyValueSize(0, input);
      context.setOrderInMap(kvOrder[0]);
    }

  }

  /**
   * Deserialize the next key value from the input buffer
   * it returns null if it hits the end of field type for
   * the keyvalue
   */
  public static KeyValuePair deserialize(boolean isArrayElement, KeyValue parentKv,
                                     SerializationContext context, ByteBuffer input) {
    KeyValue kv;
    String key = null;
    FieldPath parentPath = context.currentPath();
    ProjectionState parentProjState = context.currentProjectionState();
    ProjectionState projState = parentProjState;

    do {
      byte rowColType = input.get();
      if (rowColType == RowcolType.END_OF_FIELD_TYPE) {
        return null;
      }
      /* use only the bottom 5 bits for the type */
      Value.Type type = RowcolType.valueType(rowColType & RowcolType.TYPE_MASK);

      if (type == Type.MAP) {
        kv = new DBDocumentImpl();
      } else if (type == Type.ARRAY) {
        kv = new DBList(OpType.NONE);
      } else {
        kv = context.newKeyValue();
        kv.type_ = type.getCode();
      }
      kv.setIsArrayElement(isArrayElement);
      rowColType = (byte) ((rowColType >> RowcolType.OPTYPE_SHIFT) & RowcolType.OPTYPE_MASK);
      kv.opType = (byte)(OpType.values()[rowColType].ordinal() & 0xff);

      int numBitsForValue;
      if (isArrayElement) {
        numBitsForValue = 8;
      } else {
        numBitsForValue = KeyValueSizeDescriptor.valueSizeBitsForType(type);
      }

      int []keyValueSize = KeyValueSizeDescriptor.decodeKeyValueSize(numBitsForValue, input);

      TimeDescriptor.deserialize(kv, input, context);
      if (isArrayElement) {
        ArrayIndexDescriptor.deserialize(kv, input, context);
      } else {
        deserializeOrderOfField(kv, input);
        key = Bytes.toString(input, keyValueSize[0]);
      }

      if (context.shouldPrunePaths()) {
        FieldPath p = parentPath;
        projState = parentProjState;

        if (!context.isRoot() || (parentPath == FieldPath.EMPTY)) {
          if (key != null) {
            // key will not be null if we are inside a map
            p = (parentPath == FieldPath.EMPTY) ? FieldPath.parseFrom(key) :
              parentPath.cloneWithNewChild(key);
          } else if (isArrayElement) {
          // key will be null inside an array, hence take the index from KeyValue
            p = (parentPath == FieldPath.EMPTY) ? FieldPath.parseFrom(key) :
              parentPath.cloneWithNewChild(kv.orderInMap);
          }
          if (parentProjState.isAncestor()) {
            projState = context.getProjectionState(parentProjState, p);
          }
        }

        context.setCurrentPath(p);
        context.setCurrentProjectionState(projState);
      }

      context.setNewRecord(false);
      deserializeValue(kv, input, context, keyValueSize[1]);
    } while ((projState != null) && projState.isUnrelated());

    return new KeyValuePair(key, kv);
  }


  private static void deserializeOrderOfField(KeyValue kv, ByteBuffer input) {
    int []keyValueSize = KeyValueSizeDescriptor.decodeKeyValueSize(0, input);
    kv.setOrderOfField(keyValueSize[0]);
  }

  private static void deserializeValue(KeyValue kv, ByteBuffer input,
                                       SerializationContext context, int valueSize) {
      switch (kv.getType()) {
      case ARRAY: deserializeArrayValue(kv, context, input); return;
      case BINARY: {
        byte[] bytes = new byte[valueSize];
        input.get(bytes);
        ByteBuffer b = ByteBuffer.wrap(bytes);
        kv.objValue = b;
        return;
      }
      case NULL: return;
      case BOOLEAN: kv.primValue = valueSize; return;
      case BYTE: kv.primValue = input.get(); return;

      case DECIMAL:
        kv.objValue = BigDecimalSizeDescriptor.deSerialize(input);
        return;

      case DOUBLE:
      case LONG: kv.primValue = input.getLong(); return;

      case FLOAT:
      case INT:
        kv.primValue = input.getInt(); return;

      case MAP: deserializeMap(kv, context, input); return;
      case SHORT: kv.primValue = input.getShort(); return;
      case STRING:
        String v = Bytes.toString(input, valueSize);
        kv.objValue = v;
        return;

      case INTERVAL:
      case DATE:
      case TIME:
      case TIMESTAMP:
        kv.primValue = deserializeVarLong(valueSize, input); return;
      }
      throw new TypeException("Unknown type " + kv.getType());
  }

  public static long deserializeVarLong(int valueSize, ByteBuffer input) {
    switch (valueSize) {
    case 0 : return 0;
    case 1 : return ((long) input.get()) & 0xff;
    case 2 : return (((long) input.getShort()) & 0xffff);
    case 4 : return ((input.getInt()) & 0xffffffffL);
    case 8 : return input.getLong();
    default : throw new IllegalArgumentException("Invalid size in serialize " + valueSize);
    }
  }

  private static void deserializeMap(KeyValue kv, SerializationContext context,
                                     ByteBuffer input) {
    DBDocumentImpl rec = (DBDocumentImpl) kv;
    KeyValuePair value;
    TimeAndUniq savedBaseTs        = context.getBaseTime();
    FieldPath savedCurrentPath     = null;
    ProjectionState savedProjState = null;

    if (context.shouldPrunePaths()) {
      savedCurrentPath = context.currentPath();
      savedProjState   = context.currentProjectionState();
    }
    while ((value = KeyValueDeserializeHelper.deserialize(false /*isArray*/, kv,
                                                          context, input)) != null) {
      rec.map.put(value.getKey(), value.getValue());

      if (savedCurrentPath != null) {
        context.setCurrentPath(savedCurrentPath);
        context.setCurrentProjectionState(savedProjState);
      }
    }
    context.setBaseTime(savedBaseTs);
  }

  private static void deserializeArrayValue(KeyValue kv, SerializationContext context,
                                            ByteBuffer input) {
    DBList rec = (DBList) kv;
    KeyValuePair kvPair;
    TimeAndUniq savedBaseTs = context.getBaseTime();
    FieldPath savedCurrentPath     = null;
    ProjectionState savedProjState = null;

    if (context.shouldPrunePaths()) {
      savedCurrentPath = context.currentPath();
      savedProjState   = context.currentProjectionState();
    }

    while ((kvPair = KeyValueDeserializeHelper.deserialize(true /*isArray*/, kv,
                                                           context, input)) != null) {
      KeyValue value = kvPair.getValue();

      if ((savedProjState != null) && savedProjState.isUnrelated()) {
        context.setCurrentPath(savedCurrentPath);
        context.setCurrentProjectionState(savedProjState);
        continue;
      }

      /*
       * If the value is with absolute index then it is in response to projection
       * query like a[3]. So we need to insert the value at the absolute index in
       * the array to maintain the index based access intact. This might result in
       * inserting some null values in the middle
       */
      if (value.getArrayIndexType() == ArrayIndexType.ARRAY_INDEX_TYPE_ABSOLUTE) {
        if (savedProjState != null) {
          FieldPath p = savedCurrentPath.cloneWithNewChild(value.getOrderOfField());
          if (context.getProjectionState(savedProjState, p).isUnrelated()) {
            context.setCurrentPath(savedCurrentPath);
            context.setCurrentProjectionState(savedProjState);
            continue;
          }
        }

        int index = value.getOrderOfField();
        // insert nulls in the list to match the index
        while (rec.list.size() < index) {
          rec.list.add(null);
        }
        context.setHasArrayProjection();
        // set the type back to the associative to ensure that it get serialized
        // properly if its sent back again in put request
        value.setArrayIndexType(ArrayIndexType.ARRAY_INDEX_TYPE_ASSOCIATIVE);
      }
      rec.list.add(value);

      if (savedProjState != null) {
        context.setCurrentPath(savedCurrentPath);
        context.setCurrentProjectionState(savedProjState);
      }
    }
    context.setBaseTime(savedBaseTs);
  }
}
