/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.rowcol;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Map;

import org.ojai.Document;
import org.ojai.DocumentReader;
import org.ojai.Value;
import org.ojai.exceptions.TypeException;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;
import org.ojai.util.Values;

import com.mapr.db.ojai.DBDOMDocumentReader;
import com.mapr.db.rowcol.ArrayIndexDescriptor.ArrayIndexType;
import com.mapr.db.rowcol.InsertContext.OpType;

public class KeyValue implements Value, Cloneable {

  /*
   * We will try to store as many types as possible in the primitive
   * type long value to avoid overhead of object creation. For other
   * types which can't be represented in the long value we would store
   * the object reference itself.
   */
  protected long primValue;
  protected Object objValue;
  int orderInMap;
  int rootCFid;

  protected byte type_;
  byte flags;

  protected byte opType;
  byte timeDescriptor;
  boolean rootOfFamily;
  boolean partOfNonDefaultCF;

  public static final byte IsArrayElementMask = 0x1;
  public static final byte IsArrayElementShift = 0;
  public static final byte ArrayIndexTypeMask = 0x2;
  public static final byte ArrayIndexTypeShift = 1;
  public static final byte IsRootMask = 0x4;
  public static final byte IsRootShift = 2;

  public KeyValue() {
    opType = (byte)OpType.NONE.ordinal();
  }
  public KeyValue(Type type) {
    this.type_ = type.getCode();
    opType = (byte)OpType.NONE.ordinal();
  }

  public void setIsArrayElement(boolean v) {
    if (v) {
      flags |= 1 << IsArrayElementShift;
    } else {
      flags &= ~IsArrayElementMask;
    }
  }
  public boolean isArrayElement() {
    return ((flags & IsArrayElementMask) >> IsArrayElementShift) == 1 ? true : false;
  }
  public void setArrayIndexType(ArrayIndexType type) {
    flags |= type.ordinal() << ArrayIndexTypeShift;
  }
  public ArrayIndexType getArrayIndexType() {
    return ArrayIndexType.values()[((flags & ArrayIndexTypeMask) >> ArrayIndexTypeShift)];
  }

  public void setOrderOfField(int i) {
    orderInMap = i;
  }

  public int getOrderOfField() {
    return orderInMap;
  }

  public void setRootOfColumnFamily(boolean b) {
    rootOfFamily = b;
  }

  public boolean isRootOfColumnFamily() {
    return rootOfFamily;
  }

  public void setCFRootId(int id) {
    rootCFid = id;
  }

  public int getCFRootId() {
    return rootCFid;
  }

  public void setPartOfNonDefaultColumnFamily(boolean b) {
    partOfNonDefaultCF = b;
  }

  public boolean isPartOfNonDefaultColumnFamily() {
    return partOfNonDefaultCF;
  }

  public boolean isContainerType() {
    return !getType().isScalar();
  }

  public OpType getOpType() {
    return OpType.valueOf(opType);
  }

  public byte getTimeDescriptor() {
    return timeDescriptor;
  }

  public void setRootFlags(InsertContext ctx) {
    TimeDescriptor.setUpdateTimeValid(this);

    switch (ctx.getOpType()) {
    // NONE type is called in the standard path when we are creating values
    case NONE: {
      TimeDescriptor.setCreateTimeValid(this);
      TimeDescriptor.setDeleteTimeValid(this);
      break;
    }
    case MERGE:
    case SET:
    case SET_OR_REPLACE:
    case APPEND:
    case INCREMENT: {
      TimeDescriptor.setCreateTimeValid(this);
    }
    case DELETE: {
    }
    break;
    }
  }

  public void setOpTypeAndFlags(InsertContext ctx, boolean isLastElement) {
    OpType t;
    if (ctx != null) {
      t = ctx.getOpType();
    } else {
      t = OpType.NONE;
    }

    /*
     * if the element is the last element in the path
     * then set the operation type this would be conveyed in
     * the serialized format to the server
     */
    if (isLastElement) {
      opType = (byte)t.ordinal();
    }

    TimeDescriptor.reset(this);
    switch (t) {
    // NONE type is called in the standard path when we are creating values
    case NONE:
      TimeDescriptor.setCreateTimeValid(this);
      TimeDescriptor.setUpdateTimeValid(this);
      // No need to set the delete time here as
      // root of the document in insertOrReplace path
      // would have set the delete flag
      break;
    case SET:
    case SET_OR_REPLACE:
      TimeDescriptor.setCreateTimeValid(this);
      TimeDescriptor.setUpdateTimeValid(this);
      if (isLastElement) {
        TimeDescriptor.setDeleteTimeValid(this);
      }
      break;

    case INCREMENT:
    case APPEND:
    case MERGE: {
      TimeDescriptor.setCreateTimeValid(this);
      TimeDescriptor.setUpdateTimeValid(this);
      break;
    }
    case DELETE: {
      TimeDescriptor.setUpdateTimeValid(this);
      if (isLastElement) {
        opType = (byte)OpType.DELETE.ordinal();
        TimeDescriptor.setDeleteTimeValid(this);
      }
      break;
    }
    }
  }

  public void restoreKVOrder(Map <Integer, String> idToCFNameMap) {
    switch (getType()) {
    case MAP:
      DBDocumentImpl rec = (DBDocumentImpl) this;
      rec.restoreOrder(idToCFNameMap);
      break;
    case ARRAY:
      DBList arr = (DBList) this;
      arr.restoreArrayOrder(idToCFNameMap);
      break;
    default:
      break;
    }
  }

  public void setArrayIndex(ArrayIndexType type, int index) {
    setIsArrayElement(true);
    setArrayIndexType(type);
    setOrderOfField(index);
  }

  public void setRecursiveNonDefaultColumnFamily(boolean b, int cfId) {
    setPartOfNonDefaultColumnFamily(b);
    setCFRootId(cfId);
    if (getType() == Type.MAP) {
      DBDocumentImpl rec = (DBDocumentImpl) this;
      rec.setRecursiveNonDefaultColumnFamily(b, cfId);
    } else if (getType() == Type.ARRAY) {
      DBList arr = (DBList) this;
      for (KeyValue kv : arr.list) {
        if ((kv != null) &&
            (kv.getType() == Type.MAP)) {
          DBDocumentImpl rec = (DBDocumentImpl) kv;
          rec.setRecursiveNonDefaultColumnFamily(b, cfId);
        }
      }
    }
  }

  @Override
  public Type getType() {
    return Type.valueOf(type_);
  }

  @Override
  public byte getByte() {
    switch(type_) {
    case TYPE_CODE_BYTE:
      return (byte) (primValue & 0xff);
    case TYPE_CODE_SHORT:
      return (byte) getShort();
    case TYPE_CODE_INT:
      return (byte) getInt();
    case TYPE_CODE_LONG:
      return (byte) getLong();
    case TYPE_CODE_FLOAT:
      return (byte) getFloat();
    case TYPE_CODE_DOUBLE:
      return (byte) getDouble();
    case TYPE_CODE_DECIMAL:
      return ((BigDecimal) objValue).byteValue();
    default:
      throw new TypeException("Expected a numeric type, found: " + getType());
    }
  }

  @Override
  public short getShort() {
    switch(type_) {
    case TYPE_CODE_SHORT:
      return (short) (primValue & 0xffff);
    case TYPE_CODE_BYTE:
      return getByte();
    case TYPE_CODE_INT:
      return (short) getInt();
    case TYPE_CODE_LONG:
      return (short) getLong();
    case TYPE_CODE_FLOAT:
      return (short) getFloat();
    case TYPE_CODE_DOUBLE:
      return (short) getDouble();
    case TYPE_CODE_DECIMAL:
      return ((BigDecimal) objValue).shortValue();
    default:
      throw new TypeException("Expected a numeric type, found: " + getType());
    }
  }

  @Override
  public int getInt() {
    switch(type_) {
    case TYPE_CODE_INT:
      return (int) (primValue & 0xffffffffL);
    case TYPE_CODE_BYTE:
      return getByte();
    case TYPE_CODE_SHORT:
      return getShort();
    case TYPE_CODE_LONG:
      return (int) getLong();
    case TYPE_CODE_FLOAT:
      return (int) getFloat();
    case TYPE_CODE_DOUBLE:
      return (int) getDouble();
    case TYPE_CODE_DECIMAL:
      return ((BigDecimal) objValue).intValue();
    default:
      throw new TypeException("Expected a numeric type, found: " + getType());
    }
  }

  @Override
  public long getLong() {
    switch(type_) {
    case TYPE_CODE_LONG:
      return primValue;
    case TYPE_CODE_BYTE:
      return getByte();
    case TYPE_CODE_SHORT:
      return getShort();
    case TYPE_CODE_INT:
      return getInt();
    case TYPE_CODE_FLOAT:
      return (long) getFloat();
    case TYPE_CODE_DOUBLE:
      return (long) getDouble();
    case TYPE_CODE_DECIMAL:
      return ((BigDecimal) objValue).longValue();
    default:
      throw new TypeException("Expected a numeric type, found: " + getType());
    }
  }

  @Override
  public float getFloat() {
    switch(type_) {
    case TYPE_CODE_FLOAT:
      return Float.intBitsToFloat((int) (primValue & 0xffffffffL));
    case TYPE_CODE_BYTE:
      return getByte();
    case TYPE_CODE_SHORT:
      return getShort();
    case TYPE_CODE_INT:
      return (float) getInt();
    case TYPE_CODE_LONG:
      return (float) getLong();
    case TYPE_CODE_DOUBLE:
      return (float) getDouble();
    case TYPE_CODE_DECIMAL:
      return ((BigDecimal) objValue).floatValue();
    default:
      throw new TypeException("Expected a numeric type, found: " + getType());
    }
  }

  @Override
  public double getDouble() {
    switch(type_) {
    case TYPE_CODE_DOUBLE:
      return Double.longBitsToDouble(primValue);
    case TYPE_CODE_FLOAT:
      return getFloat();
    case TYPE_CODE_BYTE:
      return getByte();
    case TYPE_CODE_SHORT:
      return getShort();
    case TYPE_CODE_INT:
      return getInt();
    case TYPE_CODE_LONG:
      return getLong();
    case TYPE_CODE_DECIMAL:
      return ((BigDecimal) objValue).doubleValue();
    default:
      throw new TypeException("Expected a numeric type, found: " + getType());
    }
  }

  @Override
  public BigDecimal getDecimal() {
    switch(type_) {
    case TYPE_CODE_DECIMAL:
      return (BigDecimal) objValue;
    case TYPE_CODE_DOUBLE:
      return new BigDecimal(getDouble());
    case TYPE_CODE_FLOAT:
      return new BigDecimal(getFloat());
    case TYPE_CODE_BYTE:
      return new BigDecimal(getByte());
    case TYPE_CODE_SHORT:
      return new BigDecimal(getShort());
    case TYPE_CODE_INT:
      return new BigDecimal(getInt());
    case TYPE_CODE_LONG:
      return new BigDecimal(getLong());
    default:
      throw new TypeException("Expected a numeric type, found: " + getType());
    }
  }

  @Override
  public boolean getBoolean() {
    checkType(Type.BOOLEAN);
    return (primValue != 0);
  }

  /**
   * An internal method to get the underlying ByteBuffer without making
   * a duplicate. The caller must take care not to alter the state by
   * marking, limiting or positioning the returned ByteBuffer.
   * @return
   */
  ByteBuffer getBinaryInternal() {
    checkType(Type.BINARY);
    return ((ByteBuffer) objValue);
  }


  @Override
  public ByteBuffer getBinary() {
    checkType(Type.BINARY);
    return ((ByteBuffer) objValue).duplicate();
  }

  @Override
  public OTimestamp getTimestamp() {
    checkType(Type.TIMESTAMP);
    // on first access create the object and cache it
    // this is to avoid unnecessary object creation
    if (objValue == null) {
      objValue = new OTimestamp(primValue);
    }
    return (OTimestamp)objValue;
  }

  @Override
  public long getTimestampAsLong() {
    checkType(Type.TIMESTAMP);
    return primValue;
  }

  @Override
  public String getString() {
    checkType(Type.STRING);
    return (String) objValue;
  }

  @Override
  public ODate getDate() {
    checkType(Type.DATE);
    // on first access create the object and cache it
    // this is to avoid unnecessary object creation
    if (objValue == null) {
      objValue = ODate.fromDaysSinceEpoch((int) primValue);
    }
    return (ODate)objValue;
  }
  @Override
  public int getDateAsInt() {
    return (int) primValue;
  }

  @Override
  public OTime getTime() {
    checkType(Type.TIME);
    // on first access create the object and cache it
    // this is to avoid unnecessary object creation
    if (objValue == null) {
      objValue = OTime.fromMillisOfDay((int) primValue);
    }
    return (OTime) objValue;
  }

  @Override
  public int getTimeAsInt() {
    return (int) primValue;
  }

  @Override
  public OInterval getInterval() {
    checkType(Type.INTERVAL);
    // on first access create the object and cache it
    // this is to avoid unnecessary object creation
    if (objValue == null) {
      OInterval t = new OInterval(primValue);
      objValue = t;
    }
    return (OInterval) objValue;
  }

  @Override
  public long getIntervalAsLong() {
    return primValue;
  }

  @Override
  public Map<String, Object> getMap() {
    checkType(Type.MAP);
    return (DBDocumentImpl)this;
  }

  public Document getRecord() {
    checkType(Type.MAP);
    return (DBDocumentImpl)this;
  }

  @Override
  public Object getObject() {
    Type type = getType();
    switch (type) {
    case BOOLEAN :
      return new Boolean(getBoolean());
    case BYTE :
      return new Byte(getByte());
    case SHORT :
      return new Short(getShort());
    case INT :
      return new Integer(getInt());
    case LONG :
      return new Long(getLong());
    case FLOAT :
      return new Float(getFloat());
    case DOUBLE :
      return new Double(getDouble());
    case TIME :
      return getTime();
    case TIMESTAMP:
      return getTimestamp();
    case DATE :
      return getDate();
    case INTERVAL :
      return getInterval();
    case BINARY :
      return getBinary();
    case DECIMAL :
    case STRING :
    case NULL :
      return objValue;
    case MAP :
    case ARRAY :
      return this;
    }
    throw new TypeException("Invalid type " + type);
  }

  @Override
  public boolean equals(Object obj) {
    Type type = getType();
    if (obj == null) {
      return type == Type.NULL;
    } else if (obj instanceof KeyValue) {
      KeyValue value = (KeyValue) obj;
      if (type != value.getType()) {
        return false;
      }
      switch (type) {
      case BOOLEAN:
      case BYTE:
      case SHORT:
      case INT:
      case LONG:
      case FLOAT:
      case DOUBLE:
      case DATE:
      case TIMESTAMP:
      case TIME:
      case INTERVAL:
        return primValue == value.primValue;
      case NULL:
        return ((objValue == null) && (value.objValue == null));
      case BINARY:
      case DECIMAL:
      case STRING:
      case MAP:
      case ARRAY:
        return objValue.equals(value.objValue);
      }
    } else if (obj instanceof String) {
      return objValue.equals(obj);
    } else if (obj instanceof Byte) {
      return obj.equals(getByte());
    } else if (obj instanceof Short) {
      return obj.equals(getShort());
    } else if (obj instanceof Boolean) {
      return obj.equals(getBoolean());
    } else if (obj instanceof Float) {
      return obj.equals(getFloat());
    } else if (obj instanceof Integer) {
      return obj.equals(getInt());
    } else if (obj instanceof Long) {
      return obj.equals(getLong());
    } else if (obj instanceof BigDecimal) {
      return obj.equals(getDecimal());
    } else if (obj instanceof Double) {
      return obj.equals(getDouble());
    } else if (obj instanceof ODate) {
      /* Internally we store Date, Time and Timestamp objects as long
       * values. Therefore, it is simpler to compare against that when
       * obj is of time Date, Time or Timestamp.
       * However, if the comparison is done with, for example, a date object
       * date as date.equals(getDate()), the comparison will not be equivalent
       * since the interval implementation in java is different. It may not
       * return same result.
       */
      long dateAsLong = ((ODate) obj).toDaysSinceEpoch();
      return dateAsLong == primValue;
    } else if (obj instanceof OTime) {
      long timeAsLong = ((OTime)obj).toTimeInMillis();
      return timeAsLong == primValue;
    } else if (obj instanceof OTimestamp) {
      long timestampAsLong = ((OTimestamp)obj).getMillis();
      return getTimestampAsLong() == timestampAsLong;
    } else if (obj instanceof OInterval) {
      return obj.equals(getInterval());
    } else if (obj instanceof ByteBuffer) {
      return obj.equals(getBinary());
    } else if (obj instanceof Map) {
      return objValue.equals(obj);
    } else if (obj instanceof List) {
      return objValue.equals(obj);
    } else if (obj instanceof Value) {
      return equals(((Value) obj).getObject());
    }

    return false;
  }

  protected void checkType(Type t) throws TypeException {
    if (type_ != t.getCode()) {
      throw new TypeException("Value is of type " + getType()
          + ", but it is accessed as type " + t);
    }
    return;
  }

  protected void setPrimValue(long primValue) {
    this.primValue = primValue;
  }

  protected long getPrimValue() {
    return primValue;
  }

  protected void setObjValue(Object objValue) {
    this.objValue = objValue;
  }

  public void setIsRoot() {
    flags |= (1 << IsRootShift);

  }
  public boolean isRoot() {
    return (((flags & IsRootMask) >> IsRootShift) == 1);

  }

  @Override
  public DocumentReader asReader() {
    return new DBDOMDocumentReader(this);
  }

  @Override
  public List<Object> getList() {
    checkType(Type.ARRAY);
    return (DBList)this;
  }

  public String toStringWithTimestamp() {
    return null;
  }

  @Override
  public String toString() {
    return Values.asJsonString(this);
  }

  @Override
  public KeyValue clone() {
    try {
     return (KeyValue)super.clone();
    } catch (CloneNotSupportedException c) {
      throw new IllegalStateException("Clone of KeyValue Object failed !!");
    }
  }

  public KeyValue shallowCopy() {
    return this.clone();
  }

  public void valueToString(StringBuilder buf) {
    Type type = getType();
    switch (type) {
    case BINARY: {
      ByteBuffer b = getBinary();
      buf.append("binary size: " + (b.limit() - b.position()));
      break;
    }
    case NULL:
      buf.append("null");
      break;
    case STRING:
      buf.append('"').append(getObject().toString()).append('"');
      break;
    case BOOLEAN:
    case BYTE:
    case DECIMAL:
    case DOUBLE:
    case LONG:
    case FLOAT:
    case INT:
    case SHORT:
    case DATE:
    case TIME:
    case TIMESTAMP:
      buf.append(getObject().toString());
      break;
    case INTERVAL:
      buf.append(getIntervalAsLong());
      break;
    case MAP: {
      DBDocumentImpl r = (DBDocumentImpl) this;
      buf.append("Type: "+ type);
      buf.append(", value:");

      if (r.idValue != null) {
        buf.append(r.idValue.toString());
      }
      for (KeyValue kv : r.map.values()) {
        buf.append("\n");
        buf.append(kv.toString());
      }
      break;
    }
    case ARRAY: {
      DBList r = (DBList) this;
      buf.append("timedescriptor:" + timeDescriptor);
      for (KeyValue kv : r.list) {
        if (kv != null) {
          buf.append("\n");
          buf.append(kv.toString());
        } else {
          buf.append("NULL_ARRAY_VALUE");
        }
      }
      break;
    }
    }
  }

  public void toString(StringBuilder buf) {
    buf.append(" td:" + timeDescriptor);
    buf.append(" orderoffield:" + orderInMap);
    buf.append(" type:" + getType());
    buf.append(" optype: " + opType);
    buf.append(" value:");
    valueToString(buf);
  }

}
