package com.mapr.db.rowcol;

import org.apache.hadoop.io.WritableComparator;
import org.ojai.Value;

import com.mapr.db.impl.IdCodec;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

/**
 * This class is used to compare Value objects. This is used M/R framework for
 * sorting rows based on id.
 *
 */
public class IdValueComparator extends WritableComparator {
  private final int DATA_LENGTH_FIELD = 4;

  /**
   * Comparator that is used by M/R framework to sort keys between different phases
   * of M/R job. Input data is serialized with four bytes length (as int) followed by actual key
   * in serialized form. Due to this, we need to skip first four bytes in comparator
   * in order to ensure that the sorting is done in lexicographic order.
   * @param : b1 : byte array with serialized key1.
   * @param : s1 : offset of key1 in b1.
   * @param : l1 : length of serialized key1.
   * @param : b2 : byte array with serialized key2.
   * @param : s2 : offset of key2 in b2.
   * @param : l2 : length of serialized key2.
   * @return integer <0, 0, or >0 to indicate if key1 is less than, same or greater than key2.
   */
  @Override
  public int compare(byte[] b1, int s1, int l1, byte[] b2, int s2, int l2) {

    if ((l1 > DATA_LENGTH_FIELD) && (l2 > DATA_LENGTH_FIELD)) {
      s1 += DATA_LENGTH_FIELD;
      s2 += DATA_LENGTH_FIELD;
      l1 -= DATA_LENGTH_FIELD;
      l2 -= DATA_LENGTH_FIELD;
    } else {
      throw new IllegalArgumentException("Key length should be greater than "+Integer.toString(DATA_LENGTH_FIELD));
    }

    return Bytes.compareTo(b1, s1, l1, b2, s2, l2);

  }

  /**
   * Comparator of two value objects.
   * @param val1 : First instance of value.
   * @param val2 : Second instance of value.
   * @return integer value 0, <0 or >0 to indicate if val1 is same as, less than or greater than
   *         val2 respectively.
   */
  public int compareTo(Value val1, Value val2) {
    byte[] prevBuf = IdCodec.encodeAsBytes(val1);
    byte[] curBuf = IdCodec.encodeAsBytes(val2);
    return Bytes.compareTo(prevBuf, 0, prevBuf.length, curBuf, 0, curBuf.length);
  }

  @Override
  public int compare(Object v1, Object v2) {
    return compareTo((Value)v1, (Value)v2);
  }
}
