/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.rowcol;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.List;

import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;

import com.mapr.db.rowcol.InsertContext.OpType;

/**
 * This is the implementation of builder implementation which should be used to create value objects
 * with timestamps. The user should not create an instance of this class. Instead, the static instance
 * of this class keyValueWithTSBuilder is to be used.
 * KeyValueBuilderImpl.keyValueWithTSBuilder.initFrom(...).
 */
public class DBValueWithTSBuilder extends DBValueBuilderBase {

  private DBValueWithTSBuilder() {}

  //static instance of KeyValueBuilderImpl
  public static DBValueWithTSBuilder keyValueWithTSBuilder = new DBValueWithTSBuilder();

  @Override
  public KeyValueWithTS newDBValue() {
    return new KeyValueWithTS();
  }

  @Override
  public  KeyValueWithTS initFrom(boolean value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.BOOLEAN);
    v.setPrimValue((value) ? 1 : 0);
    return v;
  }

  @Override
  public  KeyValueWithTS initFrom(String value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.STRING);
    v.setObjValue(value);
    return v;
  }

  @Override
  public  KeyValueWithTS initFrom(byte value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.BYTE);
    v.setPrimValue(value);
    return v;
  }

  @Override
  public  KeyValueWithTS initFrom(short value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.SHORT);
    v.setPrimValue(value);
    return v;
  }

  @Override
  public  KeyValueWithTS initFrom(int value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.INT);
    v.setPrimValue(value);
    return v;
  }

  @Override
  public  KeyValueWithTS initFrom(long value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.LONG);
    v.setPrimValue(value);
    return v;
  }

  @Override
  public  KeyValueWithTS initFrom(float value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.FLOAT);
    v.setPrimValue(Float.floatToIntBits(value));
    return v;
  }

  @Override
  public  KeyValueWithTS initFrom(double value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.DOUBLE);
    v.setPrimValue(Double.doubleToLongBits(value));
    return v;
  }
  @Override
  public  KeyValueWithTS initFrom(OTime value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.TIME);
    /* just take the portion of time which is after midnight */
    v.setPrimValue(value.toTimeInMillis());
    return v;
  }

  @Override
  public  KeyValueWithTS initFrom(ODate value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.DATE);
    v.setPrimValue(value.toDaysSinceEpoch());
    return v;
  }

  @Override
  public  KeyValueWithTS initFrom(BigDecimal value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.DECIMAL);
    v.setObjValue(value);
    return v;
  }

  // NOTE : We are ignoring the nano part of the timestamp here
  @Override
  public  KeyValueWithTS initFrom(OTimestamp value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.TIMESTAMP);
    v.setPrimValue(value.getMillis());
    return v;
  }


  @Override
  public  KeyValueWithTS initFrom(OInterval value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.INTERVAL);
    v.setPrimValue(value.getTimeInMillis());
    return v;
  }

  @Override
  public  KeyValueWithTS initFrom(ByteBuffer value) {
    KeyValueWithTS v = new KeyValueWithTS(Type.BINARY);
    v.setObjValue(value.duplicate()); // prevents modifying parameter
    return v;
  }

  @Override
  public KeyValueWithTS initFrom(List<? extends Object> list) {
    if (list instanceof DBList) {
      return ((DBList)list).shallowCopy();
    }

    DBList l = new DBList(OpType.NONE);
    for (Object o : list) {
      KeyValueWithTS child = (KeyValueWithTS)initFromObject(o);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      l.addToDBList(child);
    }
    return l;
  }

  @Override
  public KeyValue initFrom(Value value) {
    if (value instanceof KeyValueWithTS) {
      return ((KeyValueWithTS) value).shallowCopy();
    }
    return initFromObject(value.getObject());
  }

  @Override
  @SuppressWarnings({ "unchecked" })
  public KeyValue initFromObject(Object value) {
    if (value instanceof KeyValueWithTS) {
      return ((KeyValueWithTS) value).shallowCopy();
    }

    // If the passed value is null the convert it to NULL type
    if (value == null) {
      return new KeyValueWithTS(Type.NULL);
    } else {
      return initFromObjectType(value);
    }

  }

  @Override
  public KeyValue initFromNull() {
    return new  KeyValueWithTS(Type.NULL);
  }

}
