/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.rowcol;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.List;

import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;

import com.mapr.db.rowcol.InsertContext.OpType;

/**
 * This is the implementation of builder implementation which should be used to create value objects
 * without timestamps. The user should not create an instance of this class. Instead,
 * the static instance of this class valueBuilder is to be used :
 * BaseValueBuilderImpl.valueBuilder.initFrom(...).
 */
public class DBValueBuilderImpl extends DBValueBuilderBase {

  private DBValueBuilderImpl() {}

  //static instance of BaseValueBuilder
  public static DBValueBuilderImpl KeyValueBuilder = new DBValueBuilderImpl();

  @Override
  public  KeyValue initFromNull() {
    return new  KeyValue(Type.NULL);
  }

  @Override
  public  KeyValue initFrom(boolean value) {
    KeyValue v = new KeyValue(Type.BOOLEAN);
    v.setPrimValue((value) ? 1 : 0);
    return v;
  }

  @Override
  public  KeyValue initFrom(String value) {
    KeyValue v = new KeyValue(Type.STRING);
    v.setObjValue(value);
    return v;
  }

  @Override
  public  KeyValue initFrom(byte value) {
    KeyValue v = new KeyValue(Type.BYTE);
    v.setPrimValue(value);
    return v;
  }

  @Override
  public  KeyValue initFrom(short value) {
    KeyValue v = new KeyValue(Type.SHORT);
    v.setPrimValue(value);
    return v;
  }

  @Override
  public  KeyValue initFrom(int value) {
    KeyValue v = new KeyValue(Type.INT);
    v.setPrimValue(value);
    return v;
  }

  @Override
  public  KeyValue initFrom(long value) {
    KeyValue v = new KeyValue(Type.LONG);
    v.setPrimValue(value);
    return v;
  }

  @Override
  public  KeyValue initFrom(float value) {
    KeyValue v = new KeyValue(Type.FLOAT);
    v.setPrimValue(Float.floatToIntBits(value));
    return v;
  }

  @Override
  public  KeyValue initFrom(double value) {
    KeyValue v = new KeyValue(Type.DOUBLE);
    v.setPrimValue(Double.doubleToLongBits(value));
    return v;
  }
  @Override
  public  KeyValue initFrom(OTime value) {
    KeyValue v = new KeyValue(Type.TIME);
    /* just take the portion of time which is after midnight */
    v.setPrimValue(value.toTimeInMillis());
    return v;
  }

  @Override
  public  KeyValue initFrom(ODate value) {
    KeyValue v = new KeyValue(Type.DATE);
    v.setPrimValue(value.toDaysSinceEpoch());
    return v;
  }

  @Override
  public  KeyValue initFrom(BigDecimal value) {
    KeyValue v = new KeyValue(Type.DECIMAL);
    v.setObjValue(value);
    return v;
  }

  // NOTE : We are ignoring the nano part of the timestamp here
  @Override
  public  KeyValue initFrom(OTimestamp value) {
    KeyValue v = new KeyValue(Type.TIMESTAMP);
    v.setPrimValue(value.getMillis());
    return v;
  }


  @Override
  public  KeyValue initFrom(OInterval value) {
    KeyValue v = new KeyValue(Type.INTERVAL);
    v.setPrimValue(value.getTimeInMillis());
    return v;
  }

  @Override
  public  KeyValue initFrom(ByteBuffer value) {
    KeyValue v = new KeyValue(Type.BINARY);
    v.setObjValue(value.duplicate()); // prevents modifying parameter
    return v;
  }

  @Override
  public KeyValue initFrom(List<? extends Object> list) {
    if (list instanceof DBList) {
      return ((DBList)list).shallowCopy();
    }

    DBList l = new DBList(OpType.NONE);
    for (Object o : list) {
      KeyValue child = initFromObject(o);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      l.addToDBList(child);
    }
    return l;
  }

  @Override
  public KeyValue initFrom(Value value) {
    if (value instanceof KeyValue) {
      return ((KeyValue) value).shallowCopy();
    }
    return initFromObject(value.getObject());
  }

  @Override
  @SuppressWarnings({ "unchecked" })
  public KeyValue initFromObject(Object value) {
    if (value instanceof KeyValue) {
      return ((KeyValue) value).shallowCopy();
    }

    // If the passed value is null the convert it to NULL type
    if (value == null) {
      return new KeyValue(Type.NULL);
    } else {
      return initFromObjectType(value);
    }

  }

  @Override
  public KeyValue newDBValue() {
    return new KeyValue();
  }

}
