package com.mapr.db.rowcol;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Map;

import org.ojai.Document;
import org.ojai.Value;
import org.ojai.exceptions.TypeException;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;
import org.ojai.util.Fields;

import com.mapr.db.rowcol.InsertContext.OpType;


public abstract class DBValueBuilderBase implements DBValueBuilder {

  @Override
  public KeyValue initFromArray(Object value) {
    Class<?> c = value.getClass().getComponentType();
    if (c.isPrimitive()) {
      if (c == byte.class) {
        byte[] v = (byte[]) value;
        return initFromArray(v);
      }
      if (c == short.class) {
        short[] v = (short[]) value;
        return initFromArray(v);
      }
      if (c == int.class) {
        int[] v = (int[]) value;
        return initFromArray(v);
      }
      if (c == long.class) {
        long[] v = (long[]) value;
        return initFromArray(v);
      }
      if (c == float.class) {
        float[] v = (float[]) value;
        return initFromArray(v);
      }
      if (c == double.class) {
        double[] v = (double[]) value;
        return initFromArray(v);
      }
      if (c == boolean.class) {
        boolean[] v = (boolean[]) value;
        return initFromArray(v);
      }
      if (c == char.class) {
        char[] v = (char[]) value;
        return initFromArray(v);
      }
    } else {
      return initFromArray((Object [])value);
    }
    return null;
  }

  protected KeyValue initFromObjectType(Object value) {
    /* based on the type of object initialize the value */
    if (value instanceof Byte) {
      return initFrom(((Byte) value).byteValue());
    }

    if (value instanceof Boolean) {
      return initFrom(((Boolean) value).booleanValue());
    }

    if (value instanceof String) {
      return initFrom((String) value);
    }

    if (value instanceof Short) {
      return initFrom(((Short) value).shortValue());
    }

    if (value instanceof Integer) {
      return initFrom(((Integer) value).intValue());
    }

    if (value instanceof Long) {
      return initFrom(((Long) value).longValue());
    }

    if (value instanceof Float) {
      return initFrom(((Float) value).floatValue());
    }

    if (value instanceof Double) {
      return initFrom(((Double) value).doubleValue());
    }

    if (value instanceof OTime) {
      return initFrom(((OTime) value));
    }

    if (value instanceof ODate) {
      return initFrom(((ODate) value));
    }

    if (value instanceof OTimestamp) {
      return initFrom(((OTimestamp) value));
    }

    if (value instanceof BigDecimal) {
      return initFrom(((BigDecimal) value));
    }

    if (value instanceof ByteBuffer) {
      return initFrom(((ByteBuffer) value));
    }

    if (value instanceof OInterval) {
      return initFrom(((OInterval) value));
    }

    if (value instanceof Document) {
      return initFrom(((Document) value));
    }

    if (value instanceof Map) {
      return initFrom((Map<String, Object>) value);
    }

    if (value instanceof List<?>) {
      return initFrom((List<Object>) value);
    }

    // must be tried as last
    if (value instanceof Value) {
      return initFrom((Value) value);
    }

    // Its an array of primitive or object type
    if (value.getClass().isArray()) {
      return initFromArray(value);
    }

    throw new TypeException("Unsupported object type of class: " + value.getClass());
  }

  @Override
  public KeyValue initFrom(Map<String, ? extends Object> map) {
    // We can't put the DBDocument directly into the map as
    // it could be part of some other document as well and we
    // will be putting information like order of field in each
    // keyvalue which would change based on which document tree
    // this value is part of. Hence make a shallow copy of the
    // value before return it.
    if (map instanceof DBDocumentImpl) {
      return ((DBDocumentImpl) map).shallowCopy();
    }

    DBDocumentImpl doc = new DBDocumentImpl();
    for (String k : map.keySet()) {
      KeyValue child = initFromObject(map.get(k));
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      doc.set(k, child);
    }
    return doc;
  }

  @Override
  public KeyValue initFrom(Document value) {
    if (value instanceof DBDocumentImpl) {
      return ((DBDocumentImpl) value).shallowCopy();
    }

    DBDocumentImpl r = new DBDocumentImpl();
    for (Map.Entry<String, Value> e : value) {
      KeyValue child = initFromObject(e.getValue());
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      r.set(Fields.quoteFieldName(e.getKey()), child);
    }
    return r;
  }

  @Override
  public  KeyValue initFromArray(Object[] values) {
    DBList list = new DBList(OpType.NONE);
    for (int i = 0; i < values.length; ++i) {
      KeyValue child = initFromObject(values[i]);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      list.addToDBList(child);
    }
    return list;
  }

  @Override
  public KeyValue initFromArray(byte[] values) {
    DBList list = new DBList(OpType.NONE);
    for (int i = 0; i < values.length; ++i) {
      KeyValue child = initFrom(values[i]);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      list.addToDBList(child);
    }
    return list;
  }

  @Override
  public KeyValue initFromArray(short[] values) {
    DBList list = new DBList(OpType.NONE);
    for (int i = 0; i < values.length; ++i) {
      KeyValue child = initFrom(values[i]);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      list.addToDBList(child);
    }
    return list;
  }

  @Override
  public KeyValue initFromArray(int[] values) {
    DBList list = new DBList(OpType.NONE);
    for (int i = 0; i < values.length; ++i) {
      KeyValue child = initFrom(values[i]);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      list.addToDBList(child);
    }
    return list;
  }

  @Override
  public KeyValue initFromArray(long[] values) {
    DBList list = new DBList(OpType.NONE);
    for (int i = 0; i < values.length; ++i) {
      KeyValue child = initFrom(values[i]);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      list.addToDBList(child);
    }
    return list;
  }

  @Override
  public KeyValue initFromArray(float[] values) {
    DBList list = new DBList(OpType.NONE);
    for (int i = 0; i < values.length; ++i) {
      KeyValue child = initFrom(values[i]);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      list.addToDBList(child);
    }
    return list;
  }

  @Override
  public KeyValue initFromArray(double[] values) {
    DBList list = new DBList(OpType.NONE);
    for (int i = 0; i < values.length; ++i) {
      KeyValue child = initFrom(values[i]);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      list.addToDBList(child);
    }
    return list;
  }

  @Override
  public KeyValue initFromArray(boolean[] values) {
    DBList list = new DBList(OpType.NONE);
    for (int i = 0; i < values.length; ++i) {
      KeyValue child = initFrom(values[i]);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      list.addToDBList(child);
    }
    return list;
  }

  @Override
  public KeyValue initFromArray(char[] values) {
    DBList list = new DBList(OpType.NONE);
    for (int i = 0; i < values.length; ++i) {
      KeyValue child = initFrom(values[i]);
      child.setOpTypeAndFlags(null /*ctx*/, false /*isLastElement*/);
      list.addToDBList(child);
    }
    return list;
  }


}
