package com.mapr.db.rowcol;

import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.TreeMap;

import org.ojai.FieldSegment;
import org.ojai.Value;

import com.mapr.db.rowcol.ArrayIndexDescriptor.ArrayIndexType;
import com.mapr.db.rowcol.InsertContext.OpType;

public class DBList extends KeyValueWithTS implements List<Object> {

  /*
   * If the array contains absolute index -- like its created with rowmutation
   * operation on the array element a.b[5].x.y -- then optype in not NONE
   * and the index and corresponding value is kept in map. For absolute type of
   * List the functions access as Java.util.List will fail. This is fine since
   * for record mutation we don't expose the JH List interface
   *
   * If the array is created by the associative index -- like a.b = a list of elements
   * then OPTYPE is NONE and list is kept in list structure.
   */
  public DBList(OpType t) {
    super(Type.ARRAY);
    /*
     * If this list is getting created as part of none
     * op then it contains associative index which increments
     * from 0 onwards otherwise it contains the absolute index of
     * array element
     */
    if (t == OpType.NONE) {
      isAbsoluteIndexType = false;
      list = new ArrayList<KeyValue>();
      objValue = list;
    } else {
      isAbsoluteIndexType = true;
      map = new TreeMap<Integer, KeyValue>();
    }
  }

  private boolean isAbsoluteIndexType;
  public boolean IsAbsoluteIndexType() {
    return isAbsoluteIndexType;
  }

  /*
   * When the isAbsoluteIndexType is true then map is valid
   * otherwise list is valid
   */
  List <KeyValue> list;
  Map<Integer, KeyValue> map;

  /* List interface start */
  @Override
  public int size() {
    return list.size();
  }

  @Override
  public boolean isEmpty() {
    return list.isEmpty();
  }

  @Override
  public boolean contains(Object o) {
    return list.contains(KeyValueBuilder.initFromObject(o));
  }

  @Override
  public Iterator<Object> iterator() {
    final Iterator<KeyValue> itr = list.iterator();
    return new Iterator<Object>() {

      @Override
      public void remove() {
        throw new UnsupportedOperationException();
      }

      @Override
      public Object next() {
        KeyValue kv = itr.next();
        return (kv != null) ? kv.getObject() : null;
      }

      @Override
      public boolean hasNext() {
        return itr.hasNext();
      }
    };
  }

  @Override
  public Object[] toArray() {
    List <Object> objs = new ArrayList<Object>(list.size());
    // Make a copy from the MapRDBValue to native objects
    for (KeyValue kv : list) {
      objs.add(kv.getObject());
    }
    return objs.toArray();
  }

  @Override
  public <T> T[] toArray(T[] a) {
    //TODO: implement this
    return a;
  }

  @Override
  public boolean add(Object e) {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean remove(Object o) {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean containsAll(Collection<?> c) {
    for (Object o : c) {
      if (list.contains(KeyValueBuilder.initFromObject(o)) == false) {
        return false;
      }
    }
    return true;
  }

  @Override
  public boolean addAll(Collection<? extends Object> c) {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean addAll(int index, Collection<? extends Object> c) {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean removeAll(Collection<?> c) {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean retainAll(Collection<?> c) {
    throw new UnsupportedOperationException();
  }

  @Override
  public void clear() {
    throw new UnsupportedOperationException();
  }

  @Override
  public Object get(int index) {
    KeyValue kv = getKeyValueAt(index);
    if (kv == null) {
      return null;
    }
    return kv.getObject();
  }

  @Override
  public Object set(int index, Object element) {
    throw new UnsupportedOperationException();
  }

  @Override
  public void add(int index, Object element) {
    throw new UnsupportedOperationException();
  }

  @Override
  public Object remove(int index) {
    throw new UnsupportedOperationException();
  }

  @Override
  public int indexOf(Object o) {
    return list.indexOf(KeyValueBuilder.initFromObject(o));
  }

  @Override
  public int lastIndexOf(Object o) {
    return list.lastIndexOf(KeyValueBuilder.initFromObject(o));
  }

  @Override
  public ListIterator<Object> listIterator() {
    return new MapRDBListIterator();
  }

  @Override
  public ListIterator<Object> listIterator(int index) {
    return new MapRDBListIterator(index);
  }

  public Iterator<Object> getSparseListIterator() {
    return new SparseListIterator();
  }

  @Override
  public List<Object> subList(int fromIndex, int toIndex) {
    List <Object> l = new ArrayList<Object>();
    for (KeyValue v : list.subList(fromIndex, toIndex)) {
      l.add(v.getObject());
    }
    return l;
  }
  /**************LIST interface END
   * @param ctx ***********************/

  /*
   * Inserts the new keyvalue child into the array
   */
  public void insert(int index, KeyValue child, InsertContext ctx) {
    child.setIsArrayElement(true);
    if (IsAbsoluteIndexType()) {
      map.put(index, child);
    } else {
      int size = list.size();
      if (index >= size) {
        while (index > size++) {
          KeyValue nullChild = KeyValueBuilder.initFromNull();
          nullChild.setIsArrayElement(true);
          nullChild.setOpTypeAndFlags(ctx, true);
          list.add(nullChild);
        }
        list.add(child);
      } else {
        list.set(index, child);
      }
    }
  }

  public void insertValueWithFlags(int index, KeyValue child) {
    InsertContext ctx = new InsertContext();
    setRootFlags(ctx);
    child.setOpTypeAndFlags(ctx, true);
    insert(index, child, ctx);
  }

  public void removeFromList(int index) {
    /*
     * remove should be called only through the JH record interface
     * Which will create an array index of type absolute
     */
    assert(IsAbsoluteIndexType() == false) :
      "remove operation must be called only on the associative type of array";

    if (index >= list.size()) {
      return;
    }

    // the key doesn't exist in the map
    list.remove(index);
  }

  /*
   * This function is called to insert an element in the docuent tree.
   * It can be called from the JH Record API as part of set / delete etc operations.
   * In that case it will be called with the ctx.optype = NONE
   * It can also be called from the record mutation operation.
   * In this call the ctx.optype will contain the actual operation like
   * SET, SET_OR_REPLACE, DELETE etc
   */
  public void createOrInsert(Iterator<FieldSegment> iter, KeyValue inKeyValue, InsertContext ctx) {

    /* if the field is null then get the next field from the iterator */
    FieldSegment field = iter.next();
    int index = field.getIndexSegment().getIndex();
    ArrayIndexType arrayIndexType = IsAbsoluteIndexType() ?
                          ArrayIndexType.ARRAY_INDEX_TYPE_ABSOLUTE :
                          ArrayIndexType.ARRAY_INDEX_TYPE_ASSOCIATIVE;

    /* set the flag in the context that it has abs array index */
    if (IsAbsoluteIndexType()) {
      ctx.setHasAbsArrayIndex(true);
    }

    /* if operation type is none then the array must be of associative type of index */
    assert((ctx.getOpType() != OpType.NONE) || !IsAbsoluteIndexType()) :
      "if operation type is none then the array must be of associative type of index";

    /* if operation type is not none then the array must be of absolute type of index */
    assert((ctx.getOpType() == OpType.NONE) || IsAbsoluteIndexType()) :
      "if operation type is none then the array must be of associative type of index";

    /*
     * get the existing key value at the given index from map or list based on
     * the type of the array index
     */
    KeyValue oldKeyValue = getKeyValueAt(index);

    /*
     * If this is the last element in the path then just
     * overwrite the previous value for the same key with
     * new value
     */
    if (field.isLastPath()) {
      inKeyValue.setOpTypeAndFlags(ctx, true /*isLastElement*/);
      insert(index, inKeyValue, ctx);
      return;
    }

    if (field.isMap()) {
      /*
       * if the existing value for the same field is not
       * a map then delete the existing value and write new
       */
      DBDocumentImpl newRecord;
      if ((oldKeyValue == null) || (oldKeyValue.getType() != Type.MAP)) {
        newRecord = new DBDocumentImpl();
        newRecord.createOrInsert(iter, inKeyValue, ctx);
        newRecord.setOpTypeAndFlags(ctx, false /*isLastElement*/);
        insert(index, newRecord, ctx);
        return;
      }

      /*
       * If the op type is not none then check if the parent
       * element has non none type - then we are having overlapping
       * mutation which we don't support. so remove the old path
       * and insert the new path.
       *
       * Example - Initially some operation merge on a.b.c by another map
       * such that the path a.b.c.d is a valid path then in the same mutation
       * another operation is to increment a.b.c.d by 5.
       * These are conflicting operations and we will take only the last
       */
      if ((ctx.getOpType() != OpType.NONE) && (oldKeyValue.getOpType() != OpType.NONE)) {
        newRecord = new DBDocumentImpl();
        newRecord.createOrInsert(iter, inKeyValue, ctx);
        newRecord.setOpTypeAndFlags(ctx, false /*isLastElement*/);
        insert(index, newRecord, ctx);
        return;
      }

      // Inserting into an existing child of map type
      newRecord = (DBDocumentImpl) oldKeyValue;
      newRecord.setOpTypeAndFlags(ctx, false /*isLastElement*/);
      newRecord.createOrInsert(iter, inKeyValue, ctx);
      return;
    }

    /*
     * next field is an array element - like a.b[5][6]
     * If the old keyvalue at the same name doesn't exist
     * or its not an array type
     * or its an array type but based on the op type
     * it doesn't have the matching associative / absolute index type
     * then remove the older element and create a new element for this key
     */
    DBList newList;
    if ((oldKeyValue == null) || (oldKeyValue.getType() != Type.ARRAY)) {
      newList = new DBList(ctx.getOpType());
      newList.createOrInsert(iter, inKeyValue, ctx);
      newList.setOpTypeAndFlags(ctx, false /*isLastElement*/);
      insert(index, newList, ctx);
      return;
    }

    /*
     * oldKeyValue exists and is of array type. Now check if the array absolute / assoc
     * index type matches for the optype
     */
    OpType ctxOpType = ctx.getOpType();
    boolean isAssocIndex = !((DBList)oldKeyValue).IsAbsoluteIndexType();

    if (((ctxOpType == OpType.NONE) && !isAssocIndex) ||
        ((ctxOpType != OpType.NONE) &&
            (isAssocIndex) || (oldKeyValue.getOpType() != OpType.NONE))) {
      newList = new DBList(ctx.getOpType());
      newList.createOrInsert(iter, inKeyValue, ctx);
      newList.setOpTypeAndFlags(ctx, false /*isLastElement*/);
      insert(index, newList, ctx);
      return;
    }

    /*
     * Inserting into an existing array type of keyvalue
     */
    newList = (DBList) oldKeyValue;
    newList.createOrInsert(iter, inKeyValue, ctx);
    newList.setOpTypeAndFlags(ctx, false /*isLastElement*/);
    insert(index, newList, ctx);
    return;
  }

  public void addToDBList(com.mapr.db.rowcol.KeyValue keyValue) {
    assert !IsAbsoluteIndexType() :
      "This should be called only during Record API";
    insert(list.size(),keyValue, null /*ctx*/);
  }

  public void addToDBListWithFlags(com.mapr.db.rowcol.KeyValue keyValue) {
    assert !IsAbsoluteIndexType() :
      "This should be called only during Record API";
    insertValueWithFlags(list.size(),keyValue);
  }

  public void delete(Iterator<FieldSegment> iter) {
    FieldSegment field = iter.next();
    if (field == null) return;

    int index = field.getIndexSegment().getIndex();
    KeyValue kv = getKeyValueAt(index);

    // if value doesn't exist in array then return null
    if (kv == null) {
      return;
    }

    // if this is the last path then return the value at this key in map
    if (field.isLastPath()) {
      removeFromList(index);
      return;
    }

    if (field.isMap()) {
      if (kv.getType() != Type.MAP) {
        return;
      }
      ((DBDocumentImpl)kv).delete(iter);
      return;
    }

    if (kv.getType() != Type.ARRAY) {
      return;
    }
    DBList l = (DBList) kv;
    l.delete(iter);
    return;
  }

  KeyValue getKeyValueAt(int index) {
    if (isAbsoluteIndexType) {
      return map.get(index);
    } else {
      return list.size() <= index ? null : list.get(index);
    }
  }

  KeyValue getKeyValueAt(Iterator<FieldSegment> iter) {
    FieldSegment field = iter.next();
    if (field == null) return null;

    int index = field.getIndexSegment().getIndex();

    KeyValue kv = getKeyValueAt(index);
    // if value doesn't exist in map then return null
    if (kv == null) {
      return null;
    }

    // if this is the last path then return the value at this index
    if (field.isLastPath()) {
      return kv;
    }

    if (field.isMap()) {
      if (kv.getType() != Type.MAP) {
        return null;
      }
      return ((DBDocumentImpl)kv).getKeyValueAt(iter);
    }
    if (kv.getType() != Type.ARRAY) {
      return null;
    }
    return ((DBList)kv).getKeyValueAt(iter);
  }

  /*
   * MapRDBListIterator is accessed only when list as associative index
   * it can't be accessed when list is created as part of mutation operation
   * and has absolute index in it.
   */
  class MapRDBListIterator implements ListIterator<Object> {
    final ListIterator<KeyValue> iter;

    public MapRDBListIterator(int index) {
      iter = list.listIterator(index);
    }

    public MapRDBListIterator() {
      iter = list.listIterator();
    }
    @Override
    public boolean hasNext() {
      return iter.hasNext();
    }

    @Override
    public Object next() {
      KeyValue kv = iter.next();
      if (kv != null) {
        return kv.getObject();
      }
      return null;
    }

    @Override
    public boolean hasPrevious() {
      return iter.hasPrevious();
    }

    @Override
    public Object previous() {
      KeyValue kv = iter.previous();
      if (kv != null) {
        return kv.getObject();
      }
      return null;
    }

    @Override
    public int nextIndex() {
      return iter.nextIndex();
    }

    @Override
    public int previousIndex() {
      return iter.previousIndex();
    }

    @Override
    public void remove() {
      throw new UnsupportedOperationException();
    }

    @Override
    public void set(Object e) {
      throw new UnsupportedOperationException();
    }

    @Override
    public void add(Object e) {
      throw new UnsupportedOperationException();
    }
  }

  /*
   * Iterator to provide sparse list version on top of the list
   * with associative index.
   */
  class SparseListIterator implements ListIterator<Object> {

    final ListIterator<KeyValue> dbListIter;
    private int cursorPrevious ; //cached previous index

    public SparseListIterator() {
      dbListIter = list.listIterator();
      cursorPrevious = -1;
    }

    @Override
    public boolean hasNext() {
      return dbListIter.nextIndex() < list.size() ;
    }

    @Override
    public Object next() {
      cursorPrevious = dbListIter.nextIndex() ;

      KeyValue kv = dbListIter.next();
      while ((kv == null) && (dbListIter.hasNext())) {
        cursorPrevious = dbListIter.nextIndex() ;
        kv = dbListIter.next();
      }
      if (kv != null) {
        return kv.getObject();
      }
      throw new NoSuchElementException();
    }

    @Override
    public boolean hasPrevious() {
      throw new UnsupportedOperationException();
    }

    @Override
    public Object previous() {
      throw new UnsupportedOperationException();
    }

    @Override
    public int nextIndex() {
      throw new UnsupportedOperationException();
    }

    @Override
    public int previousIndex() {
      return cursorPrevious;
    }

    @Override
    public void remove() {
      throw new UnsupportedOperationException();
    }

    @Override
    public void add(Object e) {
      throw new UnsupportedOperationException();
    }

    @Override
    public void set(Object e) {
      throw new UnsupportedOperationException();
    }

  }


  @Override
  public DBList shallowCopy() {
    if (!isAbsoluteIndexType) {
      DBList rec = new DBList(OpType.NONE);
      rec.list = list;
      rec.objValue = objValue;
      rec.primValue = primValue;
      return rec;
    } else {
      return (DBList)super.shallowCopy();
    }
  }

  @Override
  public String toString() {
    StringBuilder sb = new StringBuilder().append('[');
    for (Value element : list) {
      if (element != null)
        sb.append(element.toString()).append(", ");
      else
        sb.append("null").append(", ");
    }
    if (sb.length() > 2) {
      sb.setLength(sb.length()-2);
    }
    return sb.append(']').toString();
  }

  @Override
  public String toStringWithTimestamp() {

    StringBuilder sb = new StringBuilder().append('[');

    // { _timestamp : [  ],
    //   "k1" : { _timestamp : [], _value : 10 },
    //   "k2" : { _timestamp : [], _value : { "k3" : { _timestamp : [], _value : 20 } } }
    // }
    // keyname : { _timestamp : [     ], _value : { _timestamp : [ ], _value : 10 },
    for (Value element : list) {
      KeyValueWithTS kv = (KeyValueWithTS) element;
      sb.append("{")
        .append("\"_index\": ").append(ArrayIndexDescriptor.toStringWithTimestamp(kv)).append(", ")
        .append("\"_timestamp\":").append(TimeDescriptor.toStringWithTimestamp(kv)).append(", ")
        .append("\"_value\":")
        .append(kv.toStringWithTimestamp()).append("}")
        .append(", ");
    }
    if (sb.length() > 2) {
      sb.setLength(sb.length()-2);
    }
    return sb.append(']').toString();
  }

  public void restoreArrayOrder(Map <Integer, String> idToCFNameMap) {
    for (KeyValue kv : list) {
      if (kv != null) {
        if (kv.getType() == Type.MAP) {
          DBDocumentImpl rec = (DBDocumentImpl) kv;
          rec.restoreOrder(idToCFNameMap);
        }
        if (kv.getType() == Type.ARRAY) {
          DBList iarr = (DBList)kv;
          iarr.restoreArrayOrder(idToCFNameMap);
        }
      }
    }
  }
}
