/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.rowcol;

import static com.mapr.db.impl.Constants.DEFAULT_FAMILY;
import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;
import static org.ojai.DocumentConstants.ID_KEY;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import org.ojai.Document;
import org.ojai.DocumentReader;
import org.ojai.FieldPath;
import org.ojai.FieldSegment;
import org.ojai.Value;
import org.ojai.annotation.API;
import org.ojai.beans.BeanCodec;
import org.ojai.exceptions.TypeException;
import org.ojai.json.Json;
import org.ojai.json.JsonOptions;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;
import org.ojai.util.Documents;
import org.ojai.util.MapEncoder;

import com.mapr.db.ControlInfo;
import com.mapr.db.impl.IdCodec;
import com.mapr.db.ojai.DBDOMDocumentReader;
import com.mapr.db.ojai.DBDocumentReader;
import com.mapr.db.rowcol.InsertContext.OpType;
import com.mapr.db.rowcol.SerializationContext.ProjectionState;
import com.mapr.db.util.ByteBufs;

public class DBDocumentImpl extends KeyValueWithTS
    implements Document, Map <String, Object> {

  class CachedBufferInfo {
    Map <Integer, ByteBuffer> map;
    Map <FieldPath, Integer> jsonPathMap;
    Map <Integer, String> idToCFNameMap;
    boolean insertionOrder;
    public boolean decodeTimestamp;
    boolean preserveDeleteFlags;
    private String[] paths;

    public CachedBufferInfo(Map <Integer, ByteBuffer> map,
        Map <FieldPath, Integer> jsonPathMap, Map <Integer, String> idToCFNameMap,
        boolean insertionOrder, boolean decodeTimestamp, boolean preserveDeleteFlags, String[] paths) {

      this.map = new LinkedHashMap<Integer, ByteBuffer>();
      for (Integer i : map.keySet()) {

        // TODO : REMOVE THIS COPY ONCE THE BUFFER CORRUPTION
        // ISSUE IS RESOLVED. BUG 20008
        ByteBuffer src = map.get(i);
        ByteBuffer b = ByteBufs.allocatePreferred(src.limit() - src.position());
        b.put(src);
        b.rewind();
        b.order(ByteOrder.LITTLE_ENDIAN);
        this.map.put(i, b);
      }
      this.jsonPathMap = jsonPathMap;
      this.idToCFNameMap = idToCFNameMap;
      this.insertionOrder = insertionOrder;
      this.decodeTimestamp = decodeTimestamp;
      this.preserveDeleteFlags = preserveDeleteFlags;
      this.paths = paths;
    }
  }

  LinkedHashMap<String, KeyValue> map;
  CachedBufferInfo cachedBuffer;
  /*
   * Keyvalue to store the rowkey with "_id" key type
   * This is valid only at the root level. It is NULL
   * if the excludeId option is set in the table
   */
  KeyValue idValue;
  boolean excludeId;
  boolean hasDeletes;
  byte rootTimeDescriptor;


  public DBDocumentImpl() {
    super(Type.MAP);
    map = new LinkedHashMap<String, KeyValue>();
    objValue = map;
    cachedBuffer = null;
    hasDeletes = false;
  }

  void setHasDeletes(boolean v) {
    hasDeletes = v;
  }

  boolean hasDeletes() {
    return hasDeletes;
  }

  @Override
  public boolean isReadOnly() {
    return false;
  }

  public void setSerializedJson(Map <Integer, ByteBuffer> map,
      Map <FieldPath, Integer> jsonPathMap, Map <Integer, String> idToCFNameMap,
      Value id, boolean excludeId,
      boolean insertionOrder, boolean decodeTimestamp, boolean preserveDeleteFlags,
      String[] paths) {
    cachedBuffer = new CachedBufferInfo(map, jsonPathMap, idToCFNameMap,
                                        insertionOrder, decodeTimestamp, preserveDeleteFlags,
                                        paths);
    setId(id, excludeId);
  }

  public Map<FieldPath, Integer> getJsonPathMap() {
    return cachedBuffer.jsonPathMap;
  }

  public Map<Integer, ByteBuffer> getCachedBuffers() {
    return cachedBuffer.map;
  }

  @Override
  public int hashCode() {
    return map.hashCode();
  }

  public ControlInfo getControlInfo(String fieldPath) {
    return getControlInfo(FieldPath.parseFrom(fieldPath));
  }

  public ControlInfo getControlInfo(FieldPath fieldPath) {
    // XXX - Not sure what needs to go in here
    return null;
  }

  public void insertKeyValue(String key, KeyValue value, boolean isAtRoot) {
    setMapOnDemand();

    // At the root level _id field is considered as rowkey
    if (isAtRoot && key.equals(ID_KEY)) {
      setId(value);
      return;
    }

    //insertion as a leaf
    InsertContext ctx = new InsertContext();
    setRootFlags(ctx);
    value.setOpTypeAndFlags(ctx, true);
    map.put(key, value);
  }

  /**
   * insert the key pointed by the iter in the map or list. If any element
   * in the path doesn't exist then create it new.
   *
   * Passed context carries the information about the optype (none, set, delete etc)
   * for which this insertion is happening. This is used to decide the correct flags
   * and encoding in the rowcol format
   *
   */
  void createOrInsert(Iterator<FieldSegment> iter, KeyValue newKeyValue, InsertContext ctx) {
    setMapOnDemand();
    FieldSegment field;
    field = iter.next();
    String key = field.getNameSegment().getName();
    KeyValue oldKeyValue = map.get(key);

    if (ctx.IsAtRoot()) {
      // if the key name is "_id" then it needs to be a rowkey of type bytebuffer
      if (key.equals(ID_KEY)) {
        if (field.isLastPath() == false) {
          throw new TypeException("'_id' field can not be a complex type.");
        }
        setId(newKeyValue);
        return;
      }
      setRootFlags(ctx);
      ctx.SetIsAtRoot(false);
    }

    /*
     * If this is the last element in the path then just
     * overwrite the previous value for the same key with
     * new value
     */
    if (field.isLastPath()) {
      newKeyValue.setOpTypeAndFlags(ctx, true /*isLastElement*/);
      map.put(key, newKeyValue);
      return;
    }

    if (field.isMap()) {
      /*
       * if the new value for the same field is not
       * a map then delete the existing value and write new
       */
      DBDocumentImpl newDocument;
      if ((oldKeyValue == null) || (oldKeyValue.getType() != Type.MAP)) {
        newDocument = new DBDocumentImpl();
        newDocument.createOrInsert(iter, newKeyValue, ctx);
        newDocument.setOpTypeAndFlags(ctx, false /*isLastElement*/);
        map.put(key, newDocument);
        return;
      }

      /*
       * If the op type is not none then check if the parent
       * element has non none type - if its non none then we are having
       * overlapping mutation which we don't support. so remove the old path
       * and insert the new path.
       *
       * Example - Initially some operation merge on a.b.c by another map
       * such that the path a.b.c.d is a valid path then in the same mutation
       * another operation is to increment a.b.c.d by 5.
       * These are conflicting operations and we will take only the last
       */
      if ((ctx.getOpType() != OpType.NONE) && (oldKeyValue.getOpType() != OpType.NONE)) {
        newDocument = new DBDocumentImpl();
        newDocument.createOrInsert(iter, newKeyValue, ctx);
        newDocument.setOpTypeAndFlags(ctx, false /*isLastElement*/);
        map.put(key, newDocument);
        return;
      }

      // Inserting into an existing child of map type
      newDocument = (DBDocumentImpl) oldKeyValue;
      newDocument.setOpTypeAndFlags(ctx, false /*isLastElement*/);
      newDocument.createOrInsert(iter, newKeyValue, ctx);
      return;
    }

    /*
     * next field is an array element - like a.b[5]
     * If the old keyvalue at the same name doesn't exist
     * or its not an array type
     * or its an array type but based on the op type
     * it doesn't have the matching associative / absolute index type
     * then remove the older element and create a new element for this key
     */

    DBList newList;
    if ((oldKeyValue == null) || (oldKeyValue.getType() != Type.ARRAY)) {
      newList = new DBList(ctx.getOpType());
      newList.createOrInsert(iter, newKeyValue, ctx);
      newList.setOpTypeAndFlags(ctx, false /*isLastElement*/);
      map.put(key, newList);
      return;
    }

    /*
     * oldKeyValue exists and is of array type. Now check if the array absolute / assoc
     * index type matches for the optype
     */
    OpType ctxOpType = ctx.getOpType();
    boolean isAssocIndex = !((DBList)oldKeyValue).IsAbsoluteIndexType();

    if (((ctxOpType == OpType.NONE) && !isAssocIndex) ||
        ((ctxOpType != OpType.NONE) &&
            (isAssocIndex) || (oldKeyValue.getOpType() != OpType.NONE))) {
      newList = new DBList(ctx.getOpType());
      newList.createOrInsert(iter, newKeyValue, ctx);
      newList.setOpTypeAndFlags(ctx, false /*isLastElement*/);
      map.put(key, newList);
      return;
    }

    /*
     * Inserting into an existing array type of keyvalue
     */
    newList = (DBList) oldKeyValue;
    newList.createOrInsert(iter, newKeyValue, ctx);
    newList.setOpTypeAndFlags(ctx, false /*isLastElement*/);
    return;
  }

  /*
   * This function is called via the JH ecord interface for setting
   * individual fields of the record. For this path of inserting into
   * the document we create a new insertion context which has optype
   * as NONE.
   */
  private DBDocumentImpl setCommon(FieldPath fieldPath, KeyValue value) {

    Value.Type t = value.getType();
    if ((t == Type.DECIMAL) || (t == Type.INTERVAL)) {
      throw new UnsupportedOperationException("BigDecimal and Interval type not supported");
    }
    setMapOnDemand();
    Iterator<FieldSegment> iter = fieldPath.iterator();
    createOrInsert(iter, value, new InsertContext());
    return this;
  }

  private void checkValueForNull(FieldPath field, Value v) {
    if (v == null) {
      throw new NoSuchElementException("Field '" + field + "' not found in the record.");
    }
  }

  @Override
  public DBDocumentImpl empty() {
    map.clear();
    idValue = null;
    cachedBuffer = null;
    return this;
  }

  @Override
  public DBDocumentImpl set(String field, String value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, String value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, boolean value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, boolean value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, byte value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, byte value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, short value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, short value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, int value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, int value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, long value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, long value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, float value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, float value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, double value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, double value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, OTime value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, OTime value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, ODate value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, ODate value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, BigDecimal value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(FieldPath field, BigDecimal value) {
    throw new UnsupportedOperationException("BigDecimal type not supported");

  }

  @Override
  public DBDocumentImpl set(String field, byte[] value) {
    return setCommon(FieldPath.parseFrom(field),
                     KeyValueBuilder.initFrom(ByteBuffer.wrap(value)));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, byte[] value) {
    return setCommon(field, KeyValueBuilder.initFrom(ByteBuffer.wrap(value)));
  }

  @Override
  public DBDocumentImpl set(String field, byte[] value, int off, int len) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(ByteBuffer.wrap(value, off, len)));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, byte[] value, int off, int len) {
    return setCommon(field, KeyValueBuilder.initFrom(ByteBuffer.wrap(value, off, len)));
  }

  @Override
  public DBDocumentImpl set(String field, ByteBuffer value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, ByteBuffer value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, Map<String, ? extends Object> value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, Map<String, ? extends Object> value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, Document value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, Document value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, Value value) {
    return setCommon(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }
  @Override
  public DBDocumentImpl set(FieldPath field, Value value) {
    return setCommon(field, KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(String field, List<? extends Object> value) {
    return set(FieldPath.parseFrom(field), KeyValueBuilder.initFrom(value));
  }

  @Override
  public DBDocumentImpl set(FieldPath field, List<? extends Object> value) {
    if (value instanceof DBList) {
      return setCommon(field, (DBList) value);
    }
    /* create the list with optype none */
    DBList list = new DBList(OpType.NONE);
    for (Object o : value) {
      list.addToDBList(KeyValueBuilder.initFromObject(o));
    }
    return setCommon(field, list);
  }

  @Override
  public Document set(String fieldPath, OTimestamp value) {
    return set(FieldPath.parseFrom(fieldPath), value);
  }

  @Override
  public Document set(FieldPath fieldPath, OTimestamp value) {
    return setCommon(fieldPath, KeyValueBuilder.initFrom(value));
  }

  @Override
  public Document set(String fieldPath, OInterval value) {
    return set(FieldPath.parseFrom(fieldPath), value);
  }

  @Override
  public Document set(FieldPath fieldPath, OInterval value) {
    throw new UnsupportedOperationException("Interval type not supported");
  }

  @Override
  public Document setArray(String fieldPath, byte[] values) {
    return setArray(FieldPath.parseFrom(fieldPath), values);
  }

  @Override
  public Document setArray(FieldPath fieldPath, byte[] values) {
    return setCommon(fieldPath, KeyValueBuilder.initFromArray(values));
  }

  @Override
  public Document setArray(String fieldPath, boolean[] values) {
    return setArray(FieldPath.parseFrom(fieldPath), values);
  }

  @Override
  public Document setArray(FieldPath fieldPath, boolean[] values) {
    return setCommon(fieldPath, KeyValueBuilder.initFromArray(values));
  }

  @Override
  public Document setArray(String fieldPath, short[] values) {
    return setArray(FieldPath.parseFrom(fieldPath), values);
  }

  @Override
  public Document setArray(FieldPath fieldPath, short[] values) {
    return setCommon(fieldPath, KeyValueBuilder.initFromArray(values));
  }

  @Override
  public Document setArray(String fieldPath, int[] values) {
    return setArray(FieldPath.parseFrom(fieldPath), values);
  }

  @Override
  public Document setArray(FieldPath fieldPath, int[] values) {
    return setCommon(fieldPath, KeyValueBuilder.initFromArray(values));
  }

  @Override
  public Document setArray(String fieldPath, long[] values) {
    return setArray(FieldPath.parseFrom(fieldPath), values);
  }

  @Override
  public Document setArray(FieldPath fieldPath, long[] values) {
    return setCommon(fieldPath, KeyValueBuilder.initFromArray(values));
  }

  @Override
  public Document setArray(String fieldPath, float[] values) {
    return setArray(FieldPath.parseFrom(fieldPath), values);
  }

  @Override
  public Document setArray(FieldPath fieldPath, float[] values) {
    return setCommon(fieldPath, KeyValueBuilder.initFromArray(values));
  }

  @Override
  public Document setArray(String fieldPath, double[] values) {
    return setArray(FieldPath.parseFrom(fieldPath), values);
  }

  @Override
  public Document setArray(FieldPath fieldPath, double[] values) {
    return setCommon(fieldPath, KeyValueBuilder.initFromArray(values));
   }

  Document setCommonFromObjectArray(FieldPath field, Object[] values) {
    return setCommon(field, KeyValueBuilder.initFromArray(values));
  }

  @Override
  public Document setArray(String fieldPath, String[] values) {
    return setArray(FieldPath.parseFrom(fieldPath), values);
  }

  @Override
  public Document setArray(FieldPath fieldPath, String[] values) {
    return setCommonFromObjectArray(fieldPath, values);
  }

  @Override
  public Document setArray(String fieldPath, Object... values) {
    return setArray(FieldPath.parseFrom(fieldPath), values);
  }

  @Override
  public Document setArray(FieldPath fieldPath, Object... values) {
    return setCommonFromObjectArray(fieldPath, values);
  }

  @Override
  public Document setNull(String fieldPath) {
    return setNull(FieldPath.parseFrom(fieldPath));
  }

  @Override
  public Document setNull(FieldPath fieldPath) {
    return setCommon(fieldPath, new KeyValue(Type.NULL));
  }

  /********************************************************/
  /*                      GET METHODS                     */
  /********************************************************/

  /*
   * Private method to return KeyValue object at a given path
   * returns null if the path doesn't exist in the document
   */
  KeyValue getKeyValueAt(Iterator<FieldSegment> iter) {

    setMapOnDemand();
    FieldSegment field = iter.next();
    if (field == null) return null;

    String key = field.getNameSegment().getName();

    // if the excludeId is not set and request is to
    // get "_id" field then return that value here
    if ((idValue != null) && key.equals(ID_KEY) && !excludeId) {
      return idValue;
    }

    if (key.equals("")) {
      return this;
    }

    KeyValue kv = map.get(key);
    if (kv == null) {
      return null;
    }

    // if this is the last path then return the value at this key in map
    if (field.isLastPath()) {
      return kv;
    }

    // this is intermediate path so based on the type return the hierarchical value
    if (field.isMap()) {
      if (kv.getType() != Type.MAP) {
        return null;
      }
      return ((DBDocumentImpl)kv).getKeyValueAt(iter);
    }

    if (kv.getType() != Type.ARRAY) {
      return null;
    }
    return ((DBList)kv).getKeyValueAt(iter);
  }

  public KeyValue getKeyValue(String path) {
    return getKeyValue(FieldPath.parseFrom(path));
  }

  public KeyValue getKeyValue(FieldPath path) {
    return getKeyValueAt(path.iterator());
  }

  @Override
  public Value getValue(String fieldPath) {
    return getValue(FieldPath.parseFrom(fieldPath));
  }

  @Override
  public Value getValue(FieldPath fieldPath) {
    return getKeyValue(fieldPath);
  }

  @Override
  public String getString(String field) {
    return getString(FieldPath.parseFrom(field));
  }

  @Override
  public String getString(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getString();
    }
    return null;
  }

  @Override
  public boolean getBoolean(String field) {
    return getBoolean(FieldPath.parseFrom(field));
  }

  @Override
  public boolean getBoolean(FieldPath field) {
    KeyValue v = getKeyValue(field);
    checkValueForNull(field, v);
    return v.getBoolean();
  }

  @Override
  public Boolean getBooleanObj(String field) {
    return getBooleanObj(FieldPath.parseFrom(field));
  }

  @Override
  public Boolean getBooleanObj(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getBoolean();
    }
    return null;
  }

  @Override
  public byte getByte(String field) {
    return getByte(FieldPath.parseFrom(field));
  }

  @Override
  public byte getByte(FieldPath field) {
    KeyValue v = getKeyValue(field);
    checkValueForNull(field, v);
    return v.getByte();
  }

  @Override
  public Byte getByteObj(String field) {
    return getByteObj(FieldPath.parseFrom(field));
  }

  @Override
  public Byte getByteObj(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getByte();
    }
    return null;
  }

  @Override
  public short getShort(String field) {
    return getShort(FieldPath.parseFrom(field));
  }

  @Override
  public short getShort(FieldPath field) {
    KeyValue v = getKeyValue(field);
    checkValueForNull(field, v);
    return v.getShort();
  }

  @Override
  public Short getShortObj(String field) {
    return getShortObj(FieldPath.parseFrom(field));
  }

  @Override
  public Short getShortObj(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getShort();
    }
    return null;
  }

  @Override
  public int getInt(String field) {
    return getInt(FieldPath.parseFrom(field));
  }

  @Override
  public int getInt(FieldPath field) {
    KeyValue v = getKeyValue(field);
    checkValueForNull(field, v);
    return v.getInt();
  }

  @Override
  public Integer getIntObj(String field) {
    return getIntObj(FieldPath.parseFrom(field));
  }

  @Override
  public Integer getIntObj(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getInt();
    }
    return null;
  }

  @Override
  public long getLong(String field) {
    return getLong(FieldPath.parseFrom(field));
  }

  @Override
  public long getLong(FieldPath field) {
    KeyValue v = getKeyValue(field);
    checkValueForNull(field, v);
    return v.getLong();
  }

  @Override
  public Long getLongObj(String field) {
    return getLongObj(FieldPath.parseFrom(field));
  }

  @Override
  public Long getLongObj(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getLong();
    }
    return null;
  }

  @Override
  public float getFloat(String field) {
    return getFloat(FieldPath.parseFrom(field));
  }

  @Override
  public float getFloat(FieldPath field) {
    KeyValue v = getKeyValue(field);
    checkValueForNull(field, v);
    return v.getFloat();
  }

  @Override
  public Float getFloatObj(String field) {
    return getFloatObj(FieldPath.parseFrom(field));
  }

  @Override
  public Float getFloatObj(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getFloat();
    }
    return null;
  }

  @Override
  public double getDouble(String field) {
    return getDouble(FieldPath.parseFrom(field));
  }

  @Override
  public double getDouble(FieldPath field) {
    KeyValue v = getKeyValue(field);
    checkValueForNull(field, v);
    return v.getDouble();
  }

  @Override
  public Double getDoubleObj(String field) {
    return getDoubleObj(FieldPath.parseFrom(field));
  }

  @Override
  public Double getDoubleObj(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getDouble();
    }
    return null;
  }

  @Override
  public OTime getTime(String field) {
    return getTime(FieldPath.parseFrom(field));
  }
  @Override
  public OTime getTime(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getTime();
    }
    return null;
  }

  @Override
  public ODate getDate(String field) {
    return getDate(FieldPath.parseFrom(field));
  }

  @Override
  public ODate getDate(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getDate();
    }
    return null;
  }

  @Override
  public BigDecimal getDecimal(String field) {
    return getDecimal(FieldPath.parseFrom(field));
  }

  @Override
  public BigDecimal getDecimal(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getDecimal();
    }
    return null;
  }

  @Override
  public ByteBuffer getBinary(String field) {
    return getBinary(FieldPath.parseFrom(field));
  }

  @Override
  public ByteBuffer getBinary(FieldPath field) {
    KeyValue v = getKeyValue(field);
    if (v != null) {
      return v.getBinary();
    }
    return null;
  }

  @Override
  public OTimestamp getTimestamp(String fieldPath) {
    return getTimestamp(FieldPath.parseFrom(fieldPath));
  }

  @Override
  public OTimestamp getTimestamp(FieldPath fieldPath) {
    KeyValue v = getKeyValue(fieldPath);
    if (v != null) {
      return v.getTimestamp();
    }
    return null;
  }

  @Override
  public OInterval getInterval(String fieldPath) {
    return getInterval(FieldPath.parseFrom(fieldPath));
  }

  @Override
  public OInterval getInterval(FieldPath fieldPath) {
    KeyValue v = getKeyValue(fieldPath);
    if (v != null) {
      return v.getInterval();
    }
    return null;
  }

  @Override
  public Map<String, Object> getMap(String fieldPath) {
    return getMap(FieldPath.parseFrom(fieldPath));
  }

  @Override
  public Map<String, Object> getMap(FieldPath fieldPath) {
    KeyValue v = getKeyValue(fieldPath);
    if (v != null) {
      return v.getMap();
    }
    return null;
  }

  @Override
  public List<Object> getList(String fieldPath) {
    return getList(FieldPath.parseFrom(fieldPath));
  }

  @Override
  public List<Object> getList(FieldPath fieldPath) {
    KeyValue v = getKeyValue(fieldPath);
    if (v != null) {
      return v.getList();
    }
    return null;
  }

  /************************************************/
  /*          DELETE METHODS                      */
  /************************************************/

  /*
   * deletes an element from the record
   */
  void delete(Iterator<FieldSegment> iter) {

    setMapOnDemand();

    FieldSegment field = iter.next();
    if (field == null) return;

    String key = field.getNameSegment().getName();

    // Remove the rowkey id delete on _id is called.
    if (field.isLastPath() && key.equals(ID_KEY)) {
      // we can't return from here as this can be at non root
      // level as well. If its at non root level then we can
      // make idValue null and remove it actually from map as
      // well.
      idValue = null;
    }

    KeyValue kv = map.get(key);
    if (kv == null) {
      return;
    }

    // if this is the last path then return the value at this key in map
    if (field.isLastPath()) {
      cachedBuffer = null;
      map.remove(key);
      return;
    }

    // this is intermediate path so based on the type return the hierarchical value
    if (field.isMap()) {
      if (kv.getType() != Type.MAP) {
        return;
      }
      ((DBDocumentImpl)kv).delete(iter);
      return;
    }

    if (kv.getType() != Type.ARRAY) {
      return;
    }
    ((DBList)kv).delete(iter);
    return;
  }

  @Override
  public Document delete(String field) {
    delete(FieldPath.parseFrom(field));
    return this;
  }

  @Override
  public Document delete(FieldPath path) {
    delete(path.iterator());
    return this;
  }

  /***************************************************/
  /*      MAP interface methods                      */
  /***************************************************/

  /* iterator over the Document object */
  class DBDocumentIterator implements Iterator<java.util.Map.Entry<String, Value>> {

    Iterator<String> keyIter;
    KeyValue idKeyValue;
    boolean returnedIdKeyValue;
    DBDocumentIterator() {
      setMapOnDemand();
      keyIter = map.keySet().iterator();
      idKeyValue = idValue;
      if (idValue != null && !excludeId) {
        returnedIdKeyValue = false;
      } else {
        returnedIdKeyValue = true;
      }
    }

    @Override
    public boolean hasNext() {
      /* if the _id field is there then return it first */
      if (!returnedIdKeyValue) {
        return true;
      }
      return keyIter.hasNext();
    }

    @Override
    public java.util.Map.Entry<String, Value> next() {
      /* if the _id field is there then return it first */
      if (!returnedIdKeyValue) {
        java.util.Map.Entry<String, Value> e =
            new AbstractMap.SimpleImmutableEntry<String, Value>(ID_KEY, idKeyValue);
        returnedIdKeyValue = true;
        return e;
      }
      String key = keyIter.next();
      KeyValue kv = map.get(key);
      return new AbstractMap.SimpleImmutableEntry<String, Value>(key, kv);
    }

    @Override
    public void remove() {
      throw new UnsupportedOperationException();
    }

  }

  @Override
  public Iterator<java.util.Map.Entry<String, Value>> iterator() {
    return new DBDocumentIterator();
  }

  @Override
  public void clear() {
    throw new UnsupportedOperationException();
  }

  @Override
  public boolean containsKey(Object key) {
    setMapOnDemand();
    if ((idValue != null && !excludeId) && key.equals(ID_KEY)) {
      return true;
    }
    return map.containsKey(key);
  }

  @Override
  public boolean containsValue(Object value) {
    setMapOnDemand();
    KeyValue v = KeyValueBuilder.initFromObject(value);
    if (idValue != null && !excludeId) {
      if (v.equals(value)) {
        return true;
      }
    }
    return map.containsValue(v);
  }

  @Override
  public Set<java.util.Map.Entry<String, Object>> entrySet() {
    setMapOnDemand();
    /* make a copy of the string and the real object and return that */
    LinkedHashSet<Map.Entry<String, Object>> s = new LinkedHashSet<Map.Entry<String,Object>>();
    if (idValue != null && !excludeId) {
      Map.Entry<String, Object> newEntry =
          new AbstractMap.SimpleImmutableEntry<String, Object>(ID_KEY, idValue.getObject());
      s.add(newEntry);
    }

    for (String k : map.keySet()) {

      Map.Entry<String, Object> newEntry =
          new AbstractMap.SimpleImmutableEntry<String, Object>(k, map.get(k).getObject());
      s.add(newEntry);
    }
    return s;
  }

  @Override
  public Object get(Object key) {
    setMapOnDemand();
    if ((idValue != null) && !excludeId && key.equals(ID_KEY)) {
      return idValue.getObject();
    }

    KeyValue kv = map.get(key);
    if (kv != null) {
      return kv.getObject();
    }
    return null;
  }

  @Override
  public boolean isEmpty() {
    setMapOnDemand();
    if (idValue != null && !excludeId) {
      return false;
    }
    return map.isEmpty();
  }

  @Override
  public Set<String> keySet() {
    setMapOnDemand();
    HashSet<String> s = new HashSet<String>();
    if (idValue != null && !excludeId) {
      s.add(ID_KEY);
    }
    s.addAll(map.keySet());
    return s;
  }

  @Override
  public Object put(String key, Object value) {
    throw new UnsupportedOperationException();
  }

  @Override
  public void putAll(Map<? extends String, ? extends Object> m) {
    throw new UnsupportedOperationException();
  }

  @Override
  public Object remove(Object key) {
    throw new UnsupportedOperationException();
  }

  @Override
  public int size() {
    setMapOnDemand();
    int size = map.size();
    if (idValue != null && !excludeId) {
      size++;
    }
    return size;
  }

  @Override
  public Collection<Object> values() {
    setMapOnDemand();
    /* make a copy and return the real value object */
    ArrayList<Object> list = new ArrayList<Object>();
    if (idValue != null && !excludeId) {
      list.add(idValue.getObject());
    }

    for (KeyValue v : map.values()) {
      list.add(v.getObject());
    }
    return list;
  }

  @Override
  public DocumentReader asReader() {
    if (cachedBuffer == null) {
      return new DBDOMDocumentReader(this, idValue);
    } else {
      return new DBDocumentReader(cachedBuffer.map, cachedBuffer.jsonPathMap,
          idValue, excludeId);
    }
  }

  /* This function assumes that the record object is created with cachedBuffers and
   * we need to convert it into DOM. It will parse through cached buffers to create
   * the DOM.
   */
  public void getDOMFromCachedBuffer() {
    if (cachedBuffer != null) {
      CachedBufferInfo c = cachedBuffer;
      cachedBuffer = null;
      RowcolCodec.decodeInternal(c.map,
          c.jsonPathMap, c.idToCFNameMap,
          c.insertionOrder, c.decodeTimestamp, c.preserveDeleteFlags, this, c.paths);
    }
  }

  public boolean getNeedDOMStruct() {
    return (cachedBuffer != null);
  }

  @Override
  public DocumentReader asReader(String fieldPath) {
    return asReader(FieldPath.parseFrom(fieldPath));
  }

  @Override
  public DocumentReader asReader(FieldPath fieldPath) {
    KeyValue value = getKeyValue(fieldPath);
    return new DBDOMDocumentReader(value);
  }

  @Override
  public DBDocumentImpl shallowCopy() {
    setMapOnDemand();
    //use the document as-is
    if (this.cachedBuffer != null) {
      return this;
    }
    DBDocumentImpl rec = new DBDocumentImpl();
    rec.map = map;
    rec.objValue = objValue;
    rec.primValue = primValue;
    rec.orderInMap = orderInMap;
    rec.rootCFid = rootCFid;
    rec.partOfNonDefaultCF = partOfNonDefaultCF;
    return rec;
  }

  /*
   * Return the index of this path's final element in its parent doc.
   * If doc has 5 children=m,a,n,d,y then for "n" it will return 2.
   * NOTE: This is used only internally for unit testing the output order.
   */
  public int getIndex(String path) {
    FieldPath field= FieldPath.parseFrom(path);
    Iterator<FieldSegment> iter = field.iterator();
    return getIndex(iter);
  }

  /* NOTE: This is only for debug/test purposes.
   * It cannot be used when the insertion order requrirement is disabled on the
   * server or client.
   */
  private int getIndex(Iterator<FieldSegment> iter) {
    setMapOnDemand();
    FieldSegment fs = iter.next();
    String key = fs.getNameSegment().getName();

    if (fs.isLastPath()) {
      int i = 0;
      for (String k : map.keySet()) {
        if (key.equals(k)) {
          return i;
        }
        i++;
      }
    }

    if (fs.isMap()) {
      KeyValue kv = map.get(key);

      if (kv == null) {
        return -1;
      }

      return ((DBDocumentImpl)kv).getIndex(iter);
    }

    return -1;
  }

  // Called duing deserialization to set the non-default cfId to be used for
  // restoring cross-CF insertion ordering.
  @Override
  public void setRecursiveNonDefaultColumnFamily(boolean b, int cfId) {
    Iterator<KeyValue> it = this.map.values().iterator();
    while (it.hasNext()) {
      KeyValue kv = it.next();
      kv.setPartOfNonDefaultColumnFamily(b);
      kv.setCFRootId(cfId);
      kv.setRecursiveNonDefaultColumnFamily(b, cfId);
   }
  }

  /*
   * Changes the doc to have the fields in the insertion order, rather than
   * the sorted order of field names.
   */
  DBDocumentImpl restoreOrder(Map <Integer, String> idToCFNameMap) {
    setMapOnDemand();
    if (map.size() == 0) {
      return this;
    }

    int i = 0;

    Map<String, List<KeyValuePair>> perCFKVs = null;
    List<KeyValuePair> cfRootKVs = null;
    Map<Integer, String> orderKeyMap   = null;
    Map<String, Integer> cfNameIdxMap  = new LinkedHashMap<String, Integer>();
    Map<Integer, String> defaultKeyMap = new LinkedHashMap<Integer, String>();
    boolean defaultAdded = false;

    for(Map.Entry<String, KeyValue> e : map.entrySet()) {
      KeyValue kv = e.getValue();
      String k = e.getKey();
      if (!kv.isPartOfNonDefaultColumnFamily()) {
        // System.out.println("k/v " + kv.getOrderOfField() + " " + kv.getKey()
        // + " " + i + " " + defaultAdded + " " +  kv.getCFRootId());
        if (!defaultAdded) {
          cfNameIdxMap.put(DEFAULT_FAMILY, -1);
          defaultAdded = true;
        }
        i++;
        defaultKeyMap.put(kv.getOrderOfField(), k);
      } else {
        int cfId = kv.getCFRootId();
        String cfName = idToCFNameMap.get(cfId);

        // CF's are sorted by cf name.
        if (idToCFNameMap != null) {
          if (cfRootKVs == null) {
            cfRootKVs = new ArrayList<KeyValuePair>();
          }

          if (perCFKVs == null) {
            perCFKVs = new HashMap<String, List<KeyValuePair>>();
          }

          // System.out.println(" root id=" + cfId + " " + cfName +
          //               "@" + cfRootKVs.size()  + " " + kv.getKey() + " " +
          //                kv.getOrderOfField());

          List<KeyValuePair> addTo = perCFKVs.get(cfName);
          if (addTo == null) {
            perCFKVs.put(cfName, new ArrayList<KeyValuePair>());
            addTo = perCFKVs.get(cfName);
          }

          cfNameIdxMap.put(cfName, cfRootKVs.size());
          KeyValuePair p = new KeyValuePair(k,kv);
          cfRootKVs.add(p);
          addTo.add(p);
        }
      }
    }

    // Add in order of CF name, multi CF case only
    if (perCFKVs != null) {
      i = 0;
      SortedSet<String> cfOrder = new TreeSet<String>(cfNameIdxMap.keySet());
      orderKeyMap = new LinkedHashMap<Integer, String>();
      //System.out.println("sorted cfs " + cfNameIdxMap.size() + " " +
      //             perCFKVs.size() + " : ");
      if (cfOrder.size() == 1) {
        // If there is only one non-default CF exists at this level, then add
        // only those order/keys
        for(KeyValuePair kvp : cfRootKVs) {
          orderKeyMap.put(kvp.getValue().getOrderOfField(), kvp.getKey());
        }
      } else {
        for (String s : cfOrder) {
          int idx = cfNameIdxMap.get(s);
          // System.out.print(" (" + s + " " + idx +  " " + i + ") - ");
          if (idx != -1) {
            List<KeyValuePair> takeFrom = perCFKVs.get(s);
            int maxOrder = -1;
            for(KeyValuePair kvp : takeFrom) {
              String kvKey = kvp.getKey();
              KeyValue kv = kvp.getValue();
              if (maxOrder == -1)
                maxOrder = kv.getOrderOfField();
              else if (maxOrder > kv.getOrderOfField())
                maxOrder = kv.getOrderOfField();

              orderKeyMap.put(i + (kv.getOrderOfField() + 1), kvKey);
            }
            i += (maxOrder + 1 + takeFrom.size());
          } else {
            // Add all of default's kv's in order
            int maxOrder = -1;
            for (Integer order : defaultKeyMap.keySet()) {
              if (maxOrder == -1)
                maxOrder = order;
              else if (maxOrder > order)
                 maxOrder = order;
              orderKeyMap.put(i + (order + 1), defaultKeyMap.get(order));
            }
            i += (maxOrder + 1 + defaultKeyMap.size());
            // System.out.println(" i @ " + i + ".");
          }
        }
      }
    } else {
      // ensure no duplicate order numbers
      assert(defaultKeyMap.size() == i);
      orderKeyMap = defaultKeyMap;
    }

    SortedSet<Integer> insOrder = new TreeSet<Integer>(orderKeyMap.keySet());

    // Restore order in the original map in the field order
    for (Integer fieldOrder : insOrder) {
      String key = orderKeyMap.get(fieldOrder);
      KeyValue kv = map.get(key);

      // Field order should stay same if not multi-CF
      if ((perCFKVs == null) && !kv.isRootOfColumnFamily()) {
        assert(kv.getOrderOfField() == fieldOrder);
      }
      // System.out.println("ki/vi " + fieldOrder + " " + kv.getKey());
      map.remove(key);
      map.put(key, kv);
    }

    // restore any child maps recursively
    for (KeyValue kv : map.values()) {
      kv.restoreKVOrder(idToCFNameMap);
    }

    return this;
  }

  /*
   * Serializes the data for MapRDB record into the rowcol format
   * The passed ByteWriter should be in little endian mode
   */
  public void serializeToRowCol(ByteWriter w, SerializationContext ctx) {
    ctx.setHasDeletes(false);
    RootTimeDescriptor.serialize(this, w, ctx);
    TimeDescriptor.serialize(this, w);
    /*
     * Put the order of field in each keyvalue now.
     */
    Map <String, KeyValue> sortedMap = new TreeMap<String, KeyValue>();
    int i = 0;

    for (Map.Entry<String, KeyValue> e : map.entrySet()) {
      KeyValue kv = e.getValue();
      /*
       * skip the kv if its root of another cf
       * it will be encoded in that cf
       */
      if (kv.isRootOfColumnFamily() && ctx.isFamilyRoot(kv)) {
        ++i;
        continue;
      }
      kv.setOrderOfField(i);
      sortedMap.put(e.getKey(), kv);
      ++i;
    }

    for (Map.Entry<String, KeyValue> e : sortedMap.entrySet()) {
      KeyValue kv = e.getValue();
      KeyValueSerializeHelper.serialize(e.getKey(), kv, this, kv.getOrderOfField(), ctx, w);
    }
    /* if the record has deletes flag set now then re-write the roottime desc */
    if (ctx.hasDeletes()) {
      setHasDeletes(true);
      RootTimeDescriptor.rewriteHasDeleteFlag(this, w, ctx);
    }
    w.put((byte) RowcolType.END_OF_FIELD_TYPE);
  }

  /*
   * Deserializes the record from the rowcol format
   */
  public void deserializeFromRowCol(ByteBuffer input, SerializationContext ctx) {
    setIsRoot();
    RootTimeDescriptor.deserialize(input, ctx);
    TimeDescriptor.deserialize(this, input, ctx);
    KeyValuePair kv;
    TimeAndUniq savedBaseTs = ctx.getBaseTime();
    FieldPath savedCurrentPath = ctx.currentPath();
    ProjectionState savedProjState = ctx.currentProjectionState();
    ctx.setNewRecord(true);
    while ((kv = KeyValueDeserializeHelper.deserialize(false /*isArray*/,
                                                       this /*parentkv*/,
                                                       ctx, input)) != null) {
      map.put(kv.getKey(), kv.getValue());
      ctx.setCurrentPath(savedCurrentPath);
      ctx.setCurrentProjectionState(savedProjState);
      ctx.setNewRecord(true);
    }
    ctx.setBaseTime(savedBaseTs);
  }

  @Override
  public <T> T toJavaBean(Class<T> beanClass) {
    return BeanCodec.encode(asReader(), beanClass);
  }

  @Override
  public Document setId(String _id) {
    setId(KeyValueBuilder.initFrom(_id));
    return this;
  }

  @Override
  public Document setId(ByteBuffer _id) {
    setId(KeyValueBuilder.initFrom(_id));
    return this;
  }

  @Override
  public Document setId(Value id) {
    setId(id, false /*excludeId*/);
    return this;
  }

  /**
   * Set the row key value for this record. The record
   * takes the ownership of the passed ByteBuffer.
   *
   * @param id
   * @param exclude
   */
  void setId(Value id, boolean exclude) {
    if (id != null) {
      switch (id.getType()) {
      case BINARY:
      case STRING:
        break;
      default:
        throw new TypeException("Only " + Type.BINARY + " AND " + Type.STRING
            + " types are currently support for '_id' field.");
      }
      idValue = KeyValueBuilder.initFrom(id);
    }
    /* once the excludeId is set don't reset it */
    if (exclude == true) {
      excludeId = exclude;
    }
  }

  @Override
  public Value getId() {
    return idValue;
  }

  @Override
  public String getIdString() {
    return (idValue != null) ? idValue.getString() : null;
  }

  @Override
  public ByteBuffer getIdBinary() {
    return (idValue != null) ? idValue.getBinary() : null;
  }

  /**
   * @return the String representation of the id of this document.
   */
  @API.Internal
  public String getIdAsString() {
    return (idValue != null) ? IdCodec.asString(idValue) : null;
  }

  @API.Internal
  public void setMapOnDemand() {
    if (getNeedDOMStruct()) {
      getDOMFromCachedBuffer();
    }
  }

  @Override
  public String toString() {
    return asJsonString();
  }

  @Override
  public String asJsonString() {
    return asJsonString(JsonOptions.DEFAULT);
  }

  @Override
  public String asJsonString(JsonOptions options) {
    setMapOnDemand();
    return Json.toJsonString(this, options);
  }

  @Override
  public Map<String, Object> asMap() {
    return MapEncoder.encode(asReader());
  }

  @Override
  public String toStringWithTimestamp() {

    setMapOnDemand();
    StringBuilder sb = new StringBuilder().append('{');
    if (isRoot()) {
      sb.append("\"_timestamp\":").append(TimeDescriptor.toStringWithTimestamp(this));
    }

    boolean addedChild = false;
    for (Map.Entry<String, Value> element : this) {
      if (!(element.getValue() instanceof KeyValueWithTS)) {
        throw new IllegalStateException("toStriwithTimestamp encountered value object without timestamps");
      }
      if (isRoot() && addedChild == false) {
        sb.append(", ");
      }
      KeyValueWithTS kv = (KeyValueWithTS) element.getValue();
      sb.append("\"").append(element.getKey()).append("\"").append(':')
        .append("{\"_timestamp\":").append(TimeDescriptor.toStringWithTimestamp(kv))
        .append(", ")
        .append("\"_value\":")
        .append(kv.toStringWithTimestamp()).append("}, ");
      addedChild = true;
    }
    if (addedChild) {
      sb.setLength(sb.length()-2);
    }
    return sb.append('}').toString();
  }

  //disabling the setter method.
  //TODO : This is only being invoked from recordMutation serializer
  //       in MapReduce. Once we fix that, it should be removed.
  public void setCachedBuffer(ByteBuffer buffer) {
    cachedBuffer = null;
  }

  /* converts a db record with cached bytebuffers into string with timestamp */
  public String getStringWithTs(Map<FieldPath, Integer> tablePathToIdMap) {
    SerializedFamilyInfo[] doc1Data =
        RowcolCodec.encode(this, tablePathToIdMap, false /*isBulkLoad*/, true /*useEncoded*/);

    StringBuilder sb = new StringBuilder();
    for (SerializedFamilyInfo info : doc1Data) {
      if (info.getAction() != SerializationAction.NO_ACTION) {
        ByteBuffer bb = info.getByteBuffer();
        if (bb.remaining() > 0) {
          DBDocumentImpl familyDoc =
              (DBDocumentImpl) RowcolCodec.decode(bb, null,
                  true /*excludeId*/,
                  true /*decodeTimestamp*/, true /*preserveDeleteFlags*/);
          String path = null;
          for (Map.Entry<FieldPath, Integer>  e : tablePathToIdMap.entrySet()) {
            if (e.getValue() == info.getFamilyId()) {
              path = e.getKey().toString();
              break;
            }
          }
          sb.append("\"_familypath\":\"" + path + "\",");
          sb.append("\"_value\":" + familyDoc.toStringWithTimestamp());
        }
      }
    }
    return sb.toString();
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == null ) {
      return false;
    } else if (this == obj) {
      return true;
    }

    setMapOnDemand();
    if (obj instanceof DBDocumentImpl) {
      ((DBDocumentImpl) obj).setMapOnDemand();
    }

    if (obj instanceof Document) {
      return Documents.equals(this, (Document)obj);
    } else if (obj instanceof Map) {
      return obj.equals(this);
    }
    return false;
  }

}
