package com.mapr.db.rowcol;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import com.mapr.db.util.ByteBufs;

/**
 * A flexible size byte buffer. This class exposes methods similar to ByteBuffer
 * but it can grow in size. Internally it will store data in the direct byte buffer.
 * If the size of current direct byte buffer is not sufficient, then it will allocate
 * a larger byte buffer.
 */
public class ByteWriter  {
  static final int minBufferSize = 1024;
  static final int maxBufferSize = 16777216;

  // Used to cache previous allocation size - but can cause heap/direct
  // out of memory issues if it grows too much.
  static int lastBufferSize = minBufferSize;

  ByteBuffer buf;

  public ByteWriter() {
    if (lastBufferSize > maxBufferSize)
      lastBufferSize = maxBufferSize;

    buf = ByteBufs.allocatePreferred(lastBufferSize);
    buf.order(ByteOrder.LITTLE_ENDIAN);
  }

  public ByteBuffer getByteBuffer() {
    int savedPos = buf.position();
    int savedLimit = buf.limit();

    buf.flip();
    ByteBuffer newbuf = buf.slice();
    newbuf.order(ByteOrder.LITTLE_ENDIAN);

    buf.position(savedPos);
    buf.limit(savedLimit);

    // Reduce the buffer size if we are using less than the cached buffer size
    if ((buf.remaining() < (lastBufferSize / 2)) &&
        (lastBufferSize > minBufferSize)) {
      lastBufferSize = minBufferSize;
    }

    if (lastBufferSize < minBufferSize)
      lastBufferSize = minBufferSize;

    return newbuf;
  }

  void grow(int size) {
    if ((buf.capacity() - buf.position()) < size) {
      int newSize = buf.capacity();
      while (newSize < (buf.capacity() + size)) {
        newSize = newSize * 2;
      }

      ByteBuffer newBuffer = ByteBufs.allocatePreferred(newSize);
      newBuffer.order(ByteOrder.LITTLE_ENDIAN);
      buf.flip();
      newBuffer.put(buf);
      newBuffer.position(buf.limit());
      buf = newBuffer;
      lastBufferSize = newSize;
    }
  }

  ByteWriter put(ByteBuffer value) {
    int rem = value.remaining();
    grow(rem);
    assert((buf.capacity() - buf.position()) >= rem);
    assert(rem == value.remaining());
    // Bug 21703 - debug info to understand the issue
    try {
      buf.put(value);
    } catch (IllegalArgumentException iae) {
      System.out.println("BUG 21703 INFO - " + buf + " " + rem + " " + value);
    } 
    return this;
  }

  ByteWriter put(byte[] value) {
    grow(value.length);
    buf.put(value);
    return this;
  }

  ByteWriter put(byte value) {
    grow(1);
    buf.put(value);
    return this;
  }

  public int position() {
    return buf.position();
  }

  public void putShort(short value) {
    grow(2);
    buf.putShort(value);
  }

  public void putInt(int value) {
    grow(4);
    buf.putInt(value);
  }

  // Writes an int value at the given offset in the buffer
  // Then it sets back the position to the current position
  public void putAtOffset(int offset, int value) {
    int curOff = buf.position();
    buf.position(offset);
    buf.putInt(value);
    buf.position(curOff);
  }


  // Writes byte value at the given offset in the buffer
  // Then it sets back the position to the current position
  public void putByteAtOffset(int offset, byte value) {
    int curOff = buf.position();
    buf.position(offset);
    buf.put(value);
    buf.position(curOff);
  }

  public void put(byte[] value, int off, int len) {
    grow(len - off);
    buf.put(value, off, len);
  }

  public void putLong(long value) {
    grow(8);
    buf.putLong(value);
  }

  public void order(ByteOrder o) {
    buf.order(o);
  }
}
