package com.mapr.db.rowcol;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.nio.ByteBuffer;

public class BigDecimalSizeDescriptor {

  // maps the size (possible value 0, 1, 2, 4) to 2 bit quantity
  static byte encodeSize(int size) {
    if (size <= 2) {
      return (byte) size;
    }
    return 4;
  }
  static int decodeSize(int size) {
    if (size >= 3) {
      return 4;
    }
    return size;
  }

  /*
   * returns the serialized size of the big decimal including meta data
   * required to store the size of precision and scale, and raw value
   */
  public static int getSerializedSize(BigDecimal d) {

    /*
     * Size descriptor - First 2 bits are size of precision
     *                   0x0 = 0 byte, 0x1 = 1 byte 0x2 = 2 byte, 0x3 = 4 byte
     *                 - Next 2 bits are size of scale - values same as above
     *                 - next 4 bits are size of raw bytes
     *                 - 0 - 14 are real size of bytes and if its more than 14 byte
     *                   then next 4 bytes contains size of the byte array followed by the real byte array
     */
    int precision = d.precision();
    int scale = d.scale();
    byte[] unscaledValue = d.unscaledValue().toByteArray();

    int size = 1;
    int precisionSize = KeyValueSizeDescriptor.varIntSize(precision);
    int scaleSize = KeyValueSizeDescriptor.varIntSize(scale);

    if (unscaledValue.length >= 15) {
      // we will store the size of the byte array as separate 4 byte value
      size += 4;
    }
    size += precisionSize;
    size += scaleSize;
    size += unscaledValue.length;
    return size;
  }

  /*
   * returns the serialized size of the big decimal including meta data
   * required to store the size of precision and scale, and raw value
   */
  public static void serialize(BigDecimal d, ByteWriter w) {
    int precision = d.precision();
    int scale = d.scale();
    byte[] unscaledValue = d.unscaledValue().toByteArray();
    byte sizeDesc = 0;
    int precisionSize = KeyValueSizeDescriptor.varIntSize(precision);
    int scaleSize = KeyValueSizeDescriptor.varIntSize(scale);

    sizeDesc = encodeSize(precisionSize);
    sizeDesc |= encodeSize(scaleSize) << 2;
    if (unscaledValue.length < 15) {
      sizeDesc |= unscaledValue.length << 4;
      w.put(sizeDesc);
    } else {
      sizeDesc |= 15 << 4;
      w.put(sizeDesc);
      // we will store the size of the byte array as separate 4 byte value
      w.putInt(unscaledValue.length);
    }
    KeyValueSizeDescriptor.encodeVarInt(precision, precisionSize, w);
    KeyValueSizeDescriptor.encodeVarInt(scale, scaleSize, w);
    w.put(unscaledValue);
    return;
  }

  public static BigDecimal deSerialize(ByteBuffer input) {
    byte sizeDesc = input.get();
    int unscaledValueSize = getBigDecimalUnscaledValueSize(sizeDesc, input);
    int precision = getBigDecimalPrecision(sizeDesc, input);
    int scale = getBigDecimalScale(sizeDesc, input);
    byte [] unscaledValue = getBigDecimalUnscaledValue(unscaledValueSize, input);
    return new BigDecimal(new BigInteger(unscaledValue), scale, new MathContext(precision));
  }

  public static int getBigDecimalPrecision(byte sizeDesc, ByteBuffer input) {
    int precisionSize = decodeSize(sizeDesc & 0x3);
    return KeyValueSizeDescriptor.readVarInt(precisionSize, input);
  }

  public static int getBigDecimalScale(byte sizeDesc, ByteBuffer input) {
    int scaleSize = decodeSize((sizeDesc >> 2) & 0x3);
    return KeyValueSizeDescriptor.readVarInt(scaleSize, input);
  }

  public static int getBigDecimalUnscaledValueSize(byte sizeDesc, ByteBuffer input) {
    int unscaledValueSize = ((sizeDesc & 0xff) >> 4);
    if (unscaledValueSize >= 15) {
      unscaledValueSize = input.getInt();
    }
    return unscaledValueSize;
  }

  public static byte [] getBigDecimalUnscaledValue(int valueSize, ByteBuffer input) {
    byte [] unscaledValue = new byte[valueSize];
    input.get(unscaledValue);
    return unscaledValue;
  }


}
