package com.mapr.db.rowcol;

import java.nio.ByteBuffer;

/* class to deal with the serialization deserialization of array index */
class ArrayIndexDescriptor {
/*
 * ArrayIndexDescriptor : Describes the index of an array. There are
 * two types of indexes possible for an Array.
 *
 * 1. ArrayIndexTypeAbsolute - This is sent by the client when it wants
 * to perform mutation on specific array element. For example increment
 * a[1] by 100. Here client shall send the index 1 as absolute index.
 * This type stays only for the update from client to server. On the
 * server the index is converted to Associative index type always.
 *
 * 2. ArrayIndexTypeAssociative - This is the format in which server stores
 * the array index. Associative index contains two parts:
 *
 *  A) Timestamp B) Uniq. - Timestamp is gives the relative ordering between
 * the array elements which are appeneded at different times. Advantage of
 * using the associative index is that it makes append op on array without
 * need to read anything back from the server. Timestamp stored itself contains
 * basetimestamp an optional timestamp uniq which is used only when there
 * are multiple puts on the same rowkey in the same millisecond.

 * Uniq is an variable array of bytes it is used to decide the order between
 * multiple array elements added at the same timestamp. Uniq is mem cmparable.
 *
 * Encoding of array index:
 *
 * First byte contains the descriptor for array. It has following format
 *
 * Bit 0 : 0 - ArrayIndexTypeAbsolute 1 - ArrayIndexTypeAssociative
 *
 * If ArrayIndexTypeAbsolute then the real index would be stored as 4 bit
 * value in bit 1 - bit 5.
 *
 * If the ArrayIndexTypeAssociative then
 * Bit 1-4 decides the size and sign of timestamp
 * 0x0 - time is same as base row ts
 * 0x1 - time is +ve diff of base rowtime and diff is 1 byte long
 * 0x2 - time is +ve diff of base rowtime and diff is 2 byte long
 * 0x3 - time is +ve diff of base rowtime and diff is 4 byte long
 * 0x4 - time is absolute and is  6 byte long
 * 0x5 - time is -ve diff of base rowtime and diff is 4 byte long
 * 0x6 - time is -ve diff of base rowtime and diff is 2 byte long
 * 0x7 - time is -ve diff of base rowtime and diff is 1 byte long
 *
 * bit 5-6 : size of timestamp uniq
 * 0x0 - uniq is same as base ts uniq
 * 0x1 - uniq is abs and 1 byte long
 * 0x2 - uniq is abs and 2 byte long
 * 0x3 - uniq is abs and 4 byte long
 *
 * bit 6-7 : size of array uniq
 * 0x0 - uniq is of 1 byte size
 * 0x1 - uniq is of 2 byte size
 * 0x2 - uniq is of 3 byte size
 * 0x3 - uniq size is external and is in next 1 byte size of size
 */

  enum ArrayIndexType {
    ARRAY_INDEX_TYPE_ABSOLUTE,
    ARRAY_INDEX_TYPE_ASSOCIATIVE
  };

  static final int INDEX_TYPE_SHIFT = 0;
  static final int INDEX_TYPE_MASK = 0x1 << INDEX_TYPE_SHIFT;
  static final int INDEX_TIMESTAMP_SIZE_SHIFT = 1;
  static final int INDEX_TIMESTAMP_SIZE_MASK = 0x7 << INDEX_TIMESTAMP_SIZE_SHIFT;
  static final int INDEX_TIMESTAMP_UNIQ_SIZE_SHIFT = 4;
  static final int INDEX_TIMESTAMP_UNIQ_SIZE_MASK= 0x3 << INDEX_TIMESTAMP_UNIQ_SIZE_SHIFT;
  static final int INDEX_UNIQ_SIZE_SHIFT = 6;
  static final int INDEX_UNIQ_SIZE_MASK = 0x3 << INDEX_UNIQ_SIZE_SHIFT;

  /*
   * Serialize the index and index meta data into the output buffer
   * type - type of index (absolute or associative)
   * index  - for absolute type its the real index that we want to put
   *        - for relative type this is the index of element in the parent array
   * totalIndex - size of the parent array
   * output - output byte buffer
   */
  static void serialize(ArrayIndexType type, int index,
      int totalIndex, ByteWriter w) {
    if (type == ArrayIndexType.ARRAY_INDEX_TYPE_ABSOLUTE) {
      byte desc = (byte) type.ordinal();
      /*
       * for absolute type of index always put the index in 4 byte fixed size
       * This makes the size calculation easy and should not cause lot of overhead
       * as its used only to address specific fields at the time of rowmutation
       * operation. This type of index is not stored on disk in this format.
       * Size of 4 bytes is mapped to serialize size of 3.
       */
      int size = 3;
      desc |= (size << INDEX_TIMESTAMP_SIZE_SHIFT);
      w.put(desc);
      w.putInt(index);
      return;
    }

    // For associative index
    // first bit is 1 for assoc type
    byte desc = (byte) type.ordinal();


    /*
     * for jave client the timestamp for array is same as
     * base time hence don't need to set anything for the time size bits
     * and time uniq bits
     */


    // Based on the total index decide the size of uniq
    byte [] uniq;
    byte encodedUniqSize = 0;
    if (totalIndex <= 0xff) {
      encodedUniqSize = 0;
      uniq = new byte[1];
      uniq[0] = (byte) index;
    } else if (totalIndex <= 0xffff) {
      encodedUniqSize = 1;
      uniq = new byte[2];
      uniq[0] = (byte) ((index >> 8) & 0xff);
      uniq[1] = (byte) (index & 0xff);
    } else if (totalIndex <= 0xffffff) {
      encodedUniqSize = 2;
      uniq = new byte[3];
      uniq[0] = (byte) ((index >> 16) & 0xff);
      uniq[1] = (byte) ((index >> 8) & 0xff);
      uniq[2] = (byte) (index & 0xff);
    } else {
      encodedUniqSize = 3;
      uniq = new byte[4];
      uniq[0] = (byte) ((index >> 24) & 0xff);
      uniq[1] = (byte) ((index >> 16) & 0xff);
      uniq[2] = (byte) ((index >> 8) & 0xff);
      uniq[3] = (byte) (index & 0xff);
    }

    desc |= encodedUniqSize << INDEX_UNIQ_SIZE_SHIFT;
    w.put(desc);

    // If we are storing 4 byte uniq then store size of size and actual size
    if (encodedUniqSize == 3) {
      w.put((byte) 1);
      w.put((byte) 4);
    }
    w.put (uniq);
  }

  /*
   * Deserialize the index and index meta data from the input buffer
   */

  static void deserialize(KeyValue kv, ByteBuffer r, SerializationContext ctx) {
    byte desc = r.get();

    /* check the first bit to get the type of the index */
    ArrayIndexType type = ArrayIndexType.values()[desc & INDEX_TYPE_MASK];
    ctx.setIndexType(type);
    if (type == ArrayIndexType.ARRAY_INDEX_TYPE_ABSOLUTE) {
      int index = r.getInt();
      if (kv != null) {
        kv.setArrayIndex(type, index);
      }
      ctx.setArrayIndex(index);
      return;
    }

    if (ctx.getDecodeTimestamp()) {
      if (!(kv instanceof KeyValueWithTS)) {
        throw new IllegalStateException("ArrayIndexDescriptior.deserialize() : keyvalue object without timestamp");
      }
      ((KeyValueWithTS)kv).arrayIndex = new TimeAndUniq();
    }

    /*
     * Associative index - get the timestamp and uniq
     * For java client we don't use the relative or
     * absolute timestamp type hence ignore the second bit in the desc
     */

    // Time stamp size is in bit 2,3,4 of the desc
    // mapping of time stamp size
    // 0 -> 0, 1 -> 1, 2->2, 3 ->4 and >=4 = 8 bytes
    short timeSizeInfo = (short) ((desc & INDEX_TIMESTAMP_SIZE_MASK) >> INDEX_TIMESTAMP_SIZE_SHIFT);
    short uniqSizeInfo = (short) ((desc & INDEX_TIMESTAMP_UNIQ_SIZE_MASK) >> INDEX_TIMESTAMP_UNIQ_SIZE_SHIFT);

    TimeAndUniq arrayIndex = null;
    if ((kv != null)) {
      if (ctx.getDecodeTimestamp()) {
        arrayIndex = ((KeyValueWithTS)kv).arrayIndex;
      }
    }

    TimeDescriptor.readTimeAndUniq(r, timeSizeInfo, uniqSizeInfo, arrayIndex, ctx);

    int uniqSize = (desc & INDEX_UNIQ_SIZE_MASK) >> INDEX_UNIQ_SIZE_SHIFT;
    byte[] arrayIndexUniq = null;
    switch (uniqSize) {
    case 0: {
      arrayIndexUniq = new byte[1];
      break;
    }
    case 1: {
      arrayIndexUniq = new byte[2];
      break;
    }
    case 2: {
      arrayIndexUniq = new byte[3];
      break;
    }
    case 3: {
      byte sizeOfSize = r.get();
      assert(sizeOfSize == 1);
      byte size = r.get();
      assert(size == 4);
      arrayIndexUniq = new byte[size];
      break;
    }
    }
    r.get(arrayIndexUniq);
    if (ctx.getDecodeTimestamp()) {
      ((KeyValueWithTS)kv).arrayIndexUniq = arrayIndexUniq;
    }
  }

  public static String toStringWithTimestamp(KeyValueWithTS kv) {
    StringBuilder b = new StringBuilder();
    b.append("[");
    TimeAndUniq t = kv.arrayIndex;
    String s = String.format("%d.%d", t.time(), t.uniq());
    b.append(s).append(", \"0x");
    for (int i = 0; i < kv.arrayIndexUniq.length; ++i) {
      b.append(String.format("%02x", kv.arrayIndexUniq[i]));
    }
    b.append("\"");
    b.append("]");
    return b.toString();
  }

  public static int compareIndexTimeAndUniq(
      TimeAndUniq index1, byte[] index1Uniq,
      TimeAndUniq index2, byte[] index2Uniq) {

    if (index1.time() != index2.time()) {
      return index1.time() > index2.time() ? +1 : -1;
    }

    if (index1.uniq() != index2.uniq()) {
      return index1.uniq() > index2.uniq() ? 1 : -1;
    }

    return ByteBuffer.wrap(index1Uniq).compareTo(ByteBuffer.wrap(index2Uniq));
  }
}
