/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.ojai;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

import org.ojai.Document;
import org.ojai.DocumentListener;
import org.ojai.DocumentReader;
import org.ojai.DocumentStream;
import org.ojai.FieldPath;
import org.ojai.exceptions.StreamInUseException;
import com.mapr.db.exceptions.DBException;
import com.mapr.db.exceptions.ExceptionHandler;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.impl.MapRDBTableImplHelper;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.RowcolCodec;
import com.mapr.fs.MapRResultScanner;
import com.mapr.fs.jni.MapRResult;

// Provides an iterator over mulitple Document's for operations like scan.
// Implements Iterable<Document> but only one call is allowed to iterator().
public class DBDocumentStream implements DocumentStream {

  MapRResultScanner scanner_;
  boolean excludeId_;
  MapRDBTableImpl table_;
  private volatile boolean iteratorOpened_;
  private volatile boolean closed_;
  private volatile boolean docStreamIteratorOpened_;

  // used for removal of conditions fields that were "projected" - bug 18186
  Set<FieldPath> conditionPaths_;
  String[] projPaths_;

  public DBDocumentStream(MapRResultScanner scanner, boolean excludeId,
                          MapRDBTableImpl table) {
    scanner_ = scanner;
    excludeId_ = excludeId;
    table_ = table;
    iteratorOpened_ = false;
    conditionPaths_ = null;
    projPaths_ = null;
    closed_ = false;
    docStreamIteratorOpened_ = false;
  }

  public DBDocumentStream(MapRResultScanner scanner, boolean excludeId,
                          MapRDBTableImpl table, Set<FieldPath> condPaths,
                          String...projPaths) {
    scanner_ = scanner;
    excludeId_ = excludeId;
    table_ = table;
    iteratorOpened_ = false;
    conditionPaths_ = condPaths;
    projPaths_ = projPaths;
    closed_ = false;
    docStreamIteratorOpened_ = false;
  }

  @Override
  public void streamTo(DocumentListener l) {
    Exception failure = null;
    try {
      for (Document doc : this) {
        if (!l.documentArrived(doc)) {
          break;
        }
      }
    } catch (Exception e) {
      failure = e;
    } finally {
      try {
        close();
      } catch (Exception e) {
        if (failure == null) {
          failure = e;
        }
      }
    }

    if (failure == null) {
      l.eos();
    } else {
      l.failed(failure);
    }
  }

  @Override
  public synchronized void close() {
    if (!closed_) {
      scanner_.close();
      closed_ = true;
    }
  }

  /**
   * Grab the next row's worth of values. The scanner will return a Document.
   *
   * @return Document object if there is another row, null if the scanner is
   *         exhausted
   * @throws DBException e
   */
  private Document next() throws IOException {
    MapRResult res = null;
    try {
      res = scanner_.nextRow();
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "findAll.next()");
    }

    ByteBuffer keyBuf = res.getKey();

    if (keyBuf == null) {
      return null;
    }

    Map<Integer, ByteBuffer> cfbufs = res.getJsonByteBufs();
    if (cfbufs == null) {
      return null;
    }

    boolean shouldPrunePaths = (conditionPaths_ != null) &&
                               (conditionPaths_.size() != 0);

    Document doc = RowcolCodec.decode(cfbufs, table_.sortedByPath(),
                              table_.idToCFNameMap(), keyBuf, excludeId_,
                              true /*cacheBuffer*/,
                              table_.isKeepInsertionOrder(),
                              table_.decodeTimestamp(),
                              false /*preserveDeleteFlags*/,
                              shouldPrunePaths ? projPaths_ : null);

    if (shouldPrunePaths) {

      // If we need to remove some of the paths from returned document, we 
      // always decode the internal buffers. This is because the 
      // DBDocumentReader currently does not remove the non-projected paths
      ((DBDocumentImpl)doc).getDOMFromCachedBuffer();

      // If it has no other projected field, skip the record
      if (doc.size() == 0) {
         return next();
      }
    }

    return doc;
  }

  public void makeIteratorNotOpen() {
    docStreamIteratorOpened_ = false;
  }

  private void checkDocStreamIteratorOpened() {
    if (docStreamIteratorOpened_) {
      throw new StreamInUseException("An iterator has already been opened on " +
                                     "this document stream.");
    }
  }

  private synchronized void checkDocStreamClosed() {
    if (closed_) {
      throw new IllegalStateException("DocumentStream already closed.");
    }
  }

  @Override
  public Iterator<Document> iterator() {
    checkDocStreamIteratorOpened();
    checkDocStreamClosed();
    docStreamIteratorOpened_ = true;

    return new Iterator<Document>() {
      // The next Record, possibly pre-read
      Document next = null;
      boolean  done = false;

      // return true if there is another item pending, false if there isn't.
      // this method is where the actual advancing takes place, but you need
      // to call next() to consume it. hasNext() will only advance if there
      // isn't a pending next().
      @Override
      public boolean hasNext() {
        // Once next() got null, do not try to call it again.
        if (done) {
          return false;
        }
        checkDocStreamClosed();

        if (next == null) {
          try {
            next = DBDocumentStream.this.next();
            // Once done, no other calls will go to backend
            if (next == null) {
              done = true;
              try {
                close();
              } catch (Exception e) {
                // Ignore error in this best effort closure
              }
            }
            return next != null;
          } catch (IOException e) {
            throw ExceptionHandler.handle(e, "findNext()");
          }
        }

        return true;
      }

      // get the pending next item and advance the iterator. returns null if
      // there is no next item.
      @Override
      public Document next() {
        // since hasNext() does the real advancing, we call this to determine
        // if there is a next before proceeding.
        if (!hasNext()) {
          throw new NoSuchElementException("next() called after hasNext() " +
                                           "returned false.");
        }

        // if we get to here, then hasNext() has given us an item to return.
        // we want to return the item and then null out the next pointer, so
        // we use a temporary variable.
        Document temp = next;
        next = null;
        return temp;
      }

      @Override
      public void remove() {
        throw new UnsupportedOperationException();
      }
    };
  }

  @Override
  public Iterable<DocumentReader> documentReaders() {
    checkStateForIteration();
    iteratorOpened_ = true;
    return new DBDocumentReaderIterable(this);
  }

  private void checkStateForIteration() {
    if (iteratorOpened_) {
      throw new StreamInUseException("An iterator has already been opened on this document stream.");
    }
  }
}
