/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.ojai;

import static com.mapr.db.impl.Constants.MILLISECONDS_IN_A_DAY;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Stack;

import org.ojai.DocumentConstants;
import org.ojai.DocumentReader;
import org.ojai.FieldPath;
import org.ojai.FieldSegment;
import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.annotation.API;
import org.ojai.exceptions.DecodingException;
import org.ojai.exceptions.TypeException;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;
import org.ojai.util.Types;
import org.w3c.dom.ranges.RangeException;

import com.mapr.db.ControlInfo;
import com.mapr.db.rowcol.BigDecimalSizeDescriptor;
import com.mapr.db.rowcol.ControlInfoImpl;
import com.mapr.db.rowcol.KeyValue;
import com.mapr.db.rowcol.KeyValueDeserializeHelper;
import com.mapr.db.rowcol.SerializationContext;
import com.mapr.db.util.MCFTree;
import com.mapr.db.util.MCFTree.Node;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

// Provides an iterator over the contents within a DBDocument.
@API.Internal
public class DBDocumentReader implements DBDocumentReaderBase {
  @SuppressWarnings("unused")
  private Map<FieldPath, Integer> jsonPathMap;
  private Map<Integer, ByteBuffer> dataMap;
  private Iterator<Entry<Integer,ByteBuffer>> dataMapIterator;
  private Entry<Integer,ByteBuffer> dataMapEntry;

  private boolean emitId;
  private KeyValue idValue;

  private ByteBuffer rowBuf = null;
  private ByteBuffer defaultRowBuf = null;

  private long currentLongValue = 0;
  private Object currentObjValue = null;
  private int decimalScale ;
  private int decimalPrecision;
  private byte [] decimalUnscaledValue;

  private EventType currentEvent;
  private EventType nextEvent = null;
  private EventType processedEvent = null;
  private EventType lastEvent = null;
  private int currentCFId ;
  private int mapLevel;
  private Stack<ContainerContext> containerStack;
  private ContainerContext currentContainer;
  private Stack<MCFContext> stackOfContext;

  private SerializationContext context;
  private int[] keyValueSize;
  private String fieldName;
  private MCFTree treeMap;
  private boolean doInit = true;
  private Node currentNode = null;

  private boolean doneWithAll = false;
  private String DEFAULT_PATH = "";
  private boolean replaceTopMapFieldName = false;


  public DBDocumentReader(Map<Integer, ByteBuffer> data,
      Map<FieldPath, Integer> pathMap, KeyValue _id, boolean excludeId) {
    jsonPathMap = pathMap;
    dataMap = data;
    dataMapIterator = dataMap.entrySet().iterator();
    idValue = _id;
    emitId = !excludeId;
    decimalPrecision = 0;
    decimalScale = 0;
    decimalUnscaledValue = null;

    runInit();
    initMCFTree();
    currentNode = treeMap.search(0, DEFAULT_PATH);
    initializeCurrentNode();

  }

  private ByteBuffer checkAndRetrieveBuffer(ByteBuffer buff) {
    if (buff == null) return null;
    ByteBuffer b = buff.duplicate().order(buff.order());
    if (b.order() != ByteOrder.LITTLE_ENDIAN) {
      throw new DecodingException("Byte order of serialized buffer must be little endian.");
    }
    return b;
  }

  /* based on currentCFId, set the first node we need to visit. */
  private void initializeCurrentNode() {
    /* find the jsonPath for currentCFId */
    for (Map.Entry<FieldPath, Integer> jsonPaths : jsonPathMap.entrySet()) {
      if (currentCFId == jsonPaths.getValue()) {
        String firstKey = jsonPaths.getKey().iterator().next().getNameSegment().getName();
        if (!firstKey.isEmpty()) {
          currentNode = currentNode.findChild(firstKey);
          replaceTopMapFieldName = true;
        }
        rowBuf = currentNode.getBuffer();
        return;
      }
    }
  }


  private void runInit() {
    if (dataMap != null) {
      currentCFId = dataMapIterator.next().getKey();
    }
    mapLevel = 1;
    context = new SerializationContext();
    context.setNewRecord(true);
    containerStack = new Stack<ContainerContext>();
    /* A document is a map, by default */
    currentEvent = EventType.START_MAP;
    currentContainer = containerStack.push(new ContainerContext(Type.MAP));
    doInit = false;
    stackOfContext = new Stack<MCFContext>();
  }

  private void initMCFTree() {
    //create MCFTree
    treeMap = new MCFTree();
    for (Map.Entry<FieldPath, Integer> jsonPaths : jsonPathMap.entrySet()) {

      FieldPath p = jsonPaths.getKey();
      if (p.asPathString().isEmpty()) {
        //add buffer if it exists, if not then the root has no buffer.
        //ByteBuffer buff = dataMap.get(jsonPaths.getValue());
        ByteBuffer buff = checkAndRetrieveBuffer(dataMap.get(jsonPaths.getValue()));
        rowBuf = buff;
        treeMap.addBuffer(mapLevel, p.asPathString(), buff);
        continue;
      }
      //if the buffer associated with the fieldPath p is empty, then we don't need
      //to include this in the tree
      ByteBuffer cfBuffer = checkAndRetrieveBuffer(dataMap.get(jsonPaths.getValue()));
      if (cfBuffer == null) {
        continue;
      }

      Iterator<FieldSegment> iter = p.iterator();
      String parent = DEFAULT_PATH;

      int i = 0;
      while (iter.hasNext()) {
        String currentKey = iter.next().getNameSegment().getName();

        try {
          treeMap.insertNode(mapLevel+i+1, currentKey, parent, null);
          i++;
          parent = currentKey;
        } catch (Exception e) {
          System.err.println("Error processing JSON paths : "+e.getMessage());
        }
      }
      treeMap.addBuffer(mapLevel+i, parent, cfBuffer);
    }
  }



  private Node childrenToVisit() {
    /* If we are at map level that is greater than the level associated with
     * the current node, we don't need to continue DFS visit.
     */
    if (currentNode.getLevel() < (mapLevel-1)) {
      return null;
    }
    Map<String, Node> children = currentNode.getChildren();
    if (children != null) {
      for (Map.Entry<String, Node> c:children.entrySet()) {
        Node n = c.getValue();
        if (!n.isVisited()) {
          return n;
        }
      }
    }
    return null;
  }

  /**
   * If there is a buffer associated with a child node, we make a switch
   * We will save the serialization context of the parent in the stack and
   * flip the buffer. Next we will process the token from this buffer.
   * We also create a new context for child buffer.
   */
  private void moveToChildNode(EventType lEvent, boolean pushSerializationContext, boolean getType) {
    SerializationContext newContext = new SerializationContext();
    newContext.setNewRecord(true);
    /*
     * If we need to save a token while context switching, we use non-null lEvent.
     */
    if (pushSerializationContext) {
      stackOfContext.push(new MCFContext(context, lEvent));
    }
    context = newContext;
    rowBuf = currentNode.getBuffer();
    if (getType) {
      getNextType();
    }
  }

  private void moveToParentNode() {
    currentNode.setVisited();
    currentNode = treeMap.search(currentNode.getLevel()-1, currentNode.getParent());
  }

  private void fixFieldNameInStack() {
    if (currentEvent == EventType.START_MAP) {
      //update fieldName in the last element of stack.
      ContainerContext lastContext = containerStack.pop();
      if (!lastContext.isMap()) {
        throw new IllegalStateException("Expected map context");
      }
      containerStack.push(new ContainerContext(Type.MAP, fieldName));
    }
  }

  private void dfsVisit() {
    if (doneWithAll) {
      return;
    }
    if (rowBuf.hasRemaining()) {
      //get next token
      getNextType();

      /*
       * If the next parsed event is START_MAP then we will emit this event.
       * However, we need to see if there is a child of current node which
       * matches with the field name associated with START_MAP. If there is,
       * we need to move to this child node. If there is a buffer associated
       * with this node, we need to switch and process it before we proceed
       * further in current buffer.
       */
      if (currentEvent == EventType.START_MAP) {
        /* check if fieldName matches with a child Node */
        /* In this case, we have the fieldname present in parent */
        /* CF for which there is a child CF.  We don't need to save this */
        /* state.                                                        */
        if (!inMap()) {
          return;
        }
        Node childNode = currentNode.findChild(getFieldName());
        if (childNode != null) {
          currentNode = childNode;
          if (currentNode.hasBuffer()) {
            /* save current context and create new context for new buffer */
            moveToChildNode(null, true, false);
            fieldName = childNode.getFieldName();
          }
          /* if no buffer then move to this node */
          /* just move to this child Node */
          return;
        }
      } else if (currentEvent == EventType.END_MAP) {
        /* if there is children to be visited
         * then if the child has a buffer, we save END_MAP token in stack
         * and switch to this buffer. Otherwise we just move down.
         */

        Node nextNode = childrenToVisit() ;
        if (nextNode != null) {
          //visit child node.
          //keep going down the tree until we find a node with a buffer.
          //There should be at least one node with a buffer, guaranteed.
          currentNode = nextNode;
          containerStack.push(new ContainerContext(Type.MAP, fieldName));
          context.setIsArrayElement(false);
          if (currentNode.hasBuffer()) {
            //children has buffer. Switch and proceed.
            //save context
            moveToChildNode(currentEvent, true, true);
            fieldName = currentNode.getFieldName();
            return;
          }
          else {
            //children has no buffer
            currentNode.setVisited();
            stackOfContext.push(new MCFContext(context, currentEvent));
            currentEvent = EventType.START_MAP;
            mapLevel++;
            fieldName = currentNode.getFieldName();
          }

        } else {
          mapLevel--;

          /*
           * If we are at a level > currentNode level , we have to emit END_MAP and continue
           * parsing the current buffer.
           */
          if ((currentNode.getLevel() > mapLevel) && (currentNode.isSubtreeVisited())) {
            //if all children of current node already visited, then set it visited and go up to parent
            moveToParentNode();
            /* If there is children to visit, then visit them first */
            /* else finish visiting this node.                      */
            Node c = childrenToVisit();
            if (c != null) {
              /* switch to child */
              /* We saved END_MAP...so fixing containerContext for the document */
              mapLevel++;
              containerStack.push(new ContainerContext(Type.MAP, fieldName));
              currentNode = c;
              if (currentNode.hasBuffer()) {
                moveToChildNode(null, false, true);
              } else {
                c = childrenToVisit();
                currentEvent = EventType.START_MAP;
              }
              fieldName = currentNode.getFieldName();
              fixFieldNameInStack();
              return;
            } else {
              if (currentNode.hasBuffer()) {
                MCFContext m = stackOfContext.pop();
                context = m.context;
                rowBuf = currentNode.getBuffer();

                if (!currentNode.isRoot()) {
                  nextEvent = m.lastEvent;
                  mapLevel--;
                }

              } //end  if (currentNode.hasBuffer())
            }
          }
        }
      }

      /* if we are dealing with dummy root node for non default
       * CF, replace the fieldname.
       */
      if (replaceTopMapFieldName) {
        fieldName = currentNode.getFieldName();
        fixFieldNameInStack();
        replaceTopMapFieldName = false;
      }
    } else {

        /**
         * Here we are traversing through the tree. If the next node does not buffer, then
         * we will just emit START_MAP with the fieldName and move on.
         */
        Node child = childrenToVisit();
        if (child != null) {
          currentNode = child;
          if (child.hasBuffer()) {
            moveToChildNode(null, false, true);
          } else {
            child = childrenToVisit();
            currentEvent = EventType.START_MAP;
          }
          fieldName = currentNode.getFieldName();
          return;
        } else {
          moveToParentNode();

          if (currentNode != null) {
            if (currentNode.hasBuffer()) {
              MCFContext m = stackOfContext.pop();
              context = m.context;
              rowBuf = currentNode.getBuffer();
              /* if the subtree associated with current Node is visited then
               * emit the saved token.
               */
              if (currentNode.isSubtreeVisited()) {
                currentEvent = m.lastEvent;
                return;
              }
            }
          } else {
            doneWithAll = true;
          }
          dfsVisit();
        }

    }
  }

  /** Function to parse the next token from a serialized buffer containing
   *  a JSON document/sub-document.
   */
  private void getNextType() {
    updateCurrentContainer();

    //deserialize time descriptor
    KeyValueDeserializeHelper.deserializeWithoutKeyValue(context, rowBuf);

    //deserialize value type
    Value.Type t = context.getType();

    //value type null means END_MAP or END_ARRAY.
    if (t == null) {
      ContainerContext lastContainer = containerStack.pop();
      if (lastContainer.isMap()) {
        currentEvent = EventType.END_MAP;
        fieldName = lastContainer.getFieldName();
      } else {
        currentEvent = EventType.END_ARRAY;
      }
      if (!containerStack.isEmpty()) {
        context.setIsArrayElement(!containerStack.peek().isMap());
      }
      updateCurrentContainer();
      return;
    }

    //if value type is not null, process it and get fieldname.
    //at the end current event has the token.
    keyValueSize = context.getKeyValueSize();
    currentEvent = Types.getEventTypeForType(t);
    if (inMap()) {
      fieldName = Bytes.toString(rowBuf, keyValueSize[0]);
    } else {
      if (context.isAbsoluteIndex()) {
        currentContainer.setIndex(context.getArrayIndex());
      } else {
        currentContainer.incrementIndex();
      }
    }

    if (t == Type.MAP) {
      context.setIsArrayElement(false);
      mapLevel++;

      containerStack.push(new ContainerContext(Type.MAP, fieldName));
    } else if (t == Type.ARRAY) {
      context.setIsArrayElement(true);
      containerStack.push(new ContainerContext(Type.ARRAY));
    }

  }


  private void cacheBigDecimalComponents() {
    byte sizeDesc = rowBuf.get();
    int unscaledValueSize = BigDecimalSizeDescriptor.getBigDecimalUnscaledValueSize(sizeDesc, rowBuf);
    decimalPrecision = BigDecimalSizeDescriptor.getBigDecimalPrecision(sizeDesc, rowBuf);
    decimalScale = BigDecimalSizeDescriptor.getBigDecimalScale(sizeDesc, rowBuf);
    decimalUnscaledValue = BigDecimalSizeDescriptor.getBigDecimalUnscaledValue(unscaledValueSize, rowBuf);
  }

  private void cacheCurrentValue(EventType eventType) {
    processedEvent = eventType;
    switch(eventType) {
    case BOOLEAN:
      currentLongValue = keyValueSize[1];
      break;
    case STRING:
      currentObjValue = Bytes.toString(rowBuf, keyValueSize[1]);
      break;
    case BYTE:
      currentLongValue = rowBuf.get();
      break;
    case SHORT:
      currentLongValue = rowBuf.getShort();
      break;
    case FLOAT:
    case INT:
      currentLongValue = rowBuf.getInt();
      break;
    case DOUBLE:
    case LONG:
      currentLongValue = rowBuf.getLong();
      break;
    case DECIMAL:
      cacheBigDecimalComponents();
      currentObjValue = null;
      break;
    case DATE:
    case TIME:
    case TIMESTAMP:
    case INTERVAL:
      currentLongValue = KeyValueDeserializeHelper.deserializeVarLong(keyValueSize[1], rowBuf);
      break;
    case BINARY:
      byte[] bytes = new byte[keyValueSize[1]];
      rowBuf.get(bytes);
      ByteBuffer b = ByteBuffer.wrap(bytes);
      currentObjValue = b;
      break;
    default:
      // ARRAY, MAP and NULL need not be cached
      break;
    }
  }

  private boolean hasNextBuffer() {
    if (dataMapIterator.hasNext()) {
      rowBuf = dataMapIterator.next().getValue();
      return true;
    }
    return false;
  }

  /**
   * Moves the {@code RecordReader} to the next node and returns the node type
   * as {@code EventType}.
   *
   * @return the {@code EventType} for the current node or {@code null} if the
   *         reader is past the end of the record
   */
  @Override
  public DocumentReader.EventType next() {
    if (rowBuf == null) {
      if (!hasNextBuffer()) {
        return null;
      }
    }

    EventType et;
    if (currentEvent == null) {
      if (emitId) {
        // if _id is included, we will emit it before
        // we start with the rest of the document
        fieldName = DocumentConstants.ID_KEY;
        et = Types.getEventTypeForType(idValue.getType());
        processedEvent = et;
        currentObjValue = idValue.getObject();
        emitId = false;
        return et;
      }
      if (nextEvent != null) {
        currentEvent = nextEvent;
        nextEvent = null;
      } else {
        dfsVisit();
      }

    }

    et = currentEvent;
    currentEvent = null;

    if (et != null) {
      cacheCurrentValue(et);
    }

    return et;
  }

  private void updateCurrentContainer() {
    currentContainer = containerStack.isEmpty() ? null : containerStack.peek();
  }

  private void checkEventType(EventType eventType) throws TypeException {
    if (processedEvent != eventType) {
      throw new TypeException("Event type mismatch");
    }
  }

  /**
   * @return the name of the current field
   * @throws TypeException if the current {@code EventType} is not
   *         {@code FIELD_NAME}
   */
  @Override
  public String getFieldName() {
    if (!inMap()) {
      throw new IllegalStateException("Not traversing a map!");
    }
    return fieldName;
  }

  /**
   * @return the {@code byte} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code BYTE}
   */
  @Override
  public byte getByte() {
    checkEventType(EventType.BYTE);
    return (byte)currentLongValue;
  }

  /**
   * @return the {@code byte} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code SHORT}
   */
  @Override
  public short getShort() {
    checkEventType(EventType.SHORT);
    return (short) currentLongValue;
  }

  /**
   * @return the {@code long} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code LONG}
   */
  @Override
  public long getLong() {
    checkEventType(EventType.LONG);
    return currentLongValue;
  }

  /**
   * @return the {@code int} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code INT}
   */
  @Override
  public int getInt() {
    checkEventType(EventType.INT);
    return (int) currentLongValue;
  }

  /**
   * @return the {@code float} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code FLOAT}
   */
  @Override
  public float getFloat() {
    checkEventType(EventType.FLOAT);
    return Float.intBitsToFloat((int) (currentLongValue & 0xffffffffL));
  }

  /**
   * @return the {@code double} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code DOUBLE}
   */
  @Override
  public double getDouble() {
    checkEventType(EventType.DOUBLE);
    return Double.longBitsToDouble(currentLongValue);
  }

  /**
   * @return the {@code BigDecimal} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code DECIMAL}
   */
  @Override
  public BigDecimal getDecimal() {
    checkEventType(EventType.DECIMAL);
    if (currentObjValue == null) {
      currentObjValue = new BigDecimal(
          new BigInteger(decimalUnscaledValue), decimalScale, new MathContext(decimalPrecision));
    }
    return (BigDecimal)currentObjValue;
  }

  /**
   * Returns the <i>precision</i> of current {@code DECIMAL} node.
   * (The precision is the number of digits in the unscaled value.)
   *
   * <p>The precision of a zero value is 1.
   * @return the precision of current {@code DECIMAL} node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code DECIMAL}
   */
  @Override
  public int getDecimalPrecision() {
    checkEventType(EventType.DECIMAL);
    return decimalPrecision;
  }

  /**
   * Returns the <i>scale</i> of the current {@code DECIMAL} node. If the returned
   * value is zero or positive, the scale is the number of digits to the right
   * of the decimal point.  If negative, the unscaled value of the number is
   * multiplied by ten to the power of the negation of the scale.  For example,
   * a scale of {@code -3} means the unscaled value is multiplied by 1000.
   *
   * @return the scale of current {@code DECIMAL} node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code DECIMAL}
   */
  @Override
  public int getDecimalScale() {
    checkEventType(EventType.DECIMAL);
    return decimalScale;
  }

  /**
   * Returns an {@code int} whose value is the <i>unscaled value</i> of this
   * {@code DECIMAL}.
   *
   * @return the unscaled value of current {@code DECIMAL} node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code DECIMAL}
   * @throws RangeException if the precision of the decimal number is greater
   *         than 9
   */
  @Override
  public int getDecimalValueAsInt() {
    BigDecimal decimal = getDecimal();
    if (decimal != null) {
      return decimal.intValueExact();
    }
    return 0;
  }

  /**
   * Returns a {@code long} whose value is the <i>unscaled value</i> of this
   * {@code DECIMAL}.
   *
   * @return the unscaled value of current {@code DECIMAL} node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code DECIMAL}
   * @throws RangeException if the precision of the decimal number is greater
   *         than 18
   */
  @Override
  public long getDecimalValueAsLong() {
    BigDecimal decimal = getDecimal();
    if (decimal != null) {
      return decimal.longValueExact();
    }
    return 0;
  }

  /**
   * Returns a {@code ByteBuffer} containing the two's complement representation
   * of the current {@code DECIMAL} node. The byte array will be in <i>big-endian
   * </i> byte order: the most significant byte is in the zeroth element. The
   * array will contain the minimum number of bytes required to represent this
   * {@code DECIMAL}, including one sign bit.
   *
   * @return a byte array containing the two's complement representation of
   *         current {@code DECIMAL} node
   */
  @Override
  public ByteBuffer getDecimalValueAsBytes() {
    BigDecimal decimal = getDecimal();
    if (decimal != null) {
      BigInteger decimalInteger = decimal.unscaledValue();
      byte[] bytearray = decimalInteger.toByteArray();
      return ByteBuffer.wrap(bytearray);
    }
    return null;
  }

  /**
   * @return the {@code boolean} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code BOOLEAN}
   */
  @Override
  public boolean getBoolean() {
    checkEventType(EventType.BOOLEAN);
    return currentLongValue == 0 ? false : true;
  }

  /**
   * @return the {@code String} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code STRING}
   */
  @Override
  public String getString() {
    checkEventType(EventType.STRING);
    return (String) currentObjValue;
  }

  /**
   * @return the {@code Timestamp} value of the current node as a {@code long}
   *         representing the number of milliseconds since epoch
   * @throws TypeException if the current {@code EventType} is not
   *         {@code TIMESTAMP}
   */
  @Override
  public long getTimestampLong() {
    checkEventType(EventType.TIMESTAMP);
    return currentLongValue;
  }

  /**
   * @return the {@code Timestamp} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code TIMESTAMP}
   */
  @Override
  public OTimestamp getTimestamp() {
    checkEventType(EventType.TIMESTAMP);
    return new OTimestamp(currentLongValue);
  }

  /**
   * @return the {@code Date} value of the current node as an {@code int}
   *         representing the number of DAYS since epoch
   * @throws TypeException if the current {@code EventType} is not
   *         {@code DATE}
   */
  @Override
  public int getDateInt() {
    checkEventType(EventType.DATE);
    return (int) currentLongValue;
  }

  /**
   * @return the {@code Date} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code DATE}
   */
  @Override
  public ODate getDate() {
    checkEventType(EventType.DATE);
    return ODate.fromDaysSinceEpoch((int) currentLongValue);
  }

  /**
   * @return the {@code Time} value of the current node as an {@code int}
   *         representing the number of milliseconds since midnight
   * @throws TypeException if the current {@code EventType} is not
   *         {@code TIME}
   */
  @Override
  public int getTimeInt() {
    checkEventType(EventType.TIME);
    return (int) (currentLongValue % MILLISECONDS_IN_A_DAY);
  }

  /**
   * @return the {@code Time} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code TIME}
   */
  @Override
  public OTime getTime() {
    checkEventType(EventType.TIME);
    return OTime.fromMillisOfDay((int) currentLongValue);
  }

  /**
   * @return the {@code Interval} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code INTERVAL}
   */
  @Override
  public OInterval getInterval() {
    checkEventType(EventType.INTERVAL);
    return new OInterval(currentLongValue);
  }

  @Override
  public int getIntervalDays() {
    checkEventType(EventType.INTERVAL);
    return (int) (currentLongValue / MILLISECONDS_IN_A_DAY);
  }

  @Override
  public long getIntervalMillis() {
    checkEventType(EventType.INTERVAL);
    return currentLongValue;
  }

  /**
   * @return the {@code ByteBuffer} value of the current node
   * @throws TypeException if the current {@code EventType} is not
   *         {@code BINARY}
   */
  @Override
  public ByteBuffer getBinary() {
    checkEventType(EventType.BINARY);
    return (ByteBuffer) currentObjValue;
  }

  @Override
  public ControlInfo getControlInfo() {
    if ((processedEvent == EventType.END_MAP) || (processedEvent == EventType.END_ARRAY)) {
      throw new UnsupportedOperationException("Can not return timeDescriptor for eventType END_MAP or END_ARRAY");
    }
    return new ControlInfoImpl(context.getTimeDescriptor());
  }

  @Override
  public Value getId() {
    return idValue;
  }

  @Override
  public boolean inMap() {
    return currentContainer == null
        || currentContainer.isMap();
  }

  @Override
  public int getArrayIndex() {
    return currentContainer.getIndex();
  }

}
