/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.ojai;


import static com.mapr.db.impl.Constants.MILLISECONDS_IN_A_DAY;
import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;
import static org.ojai.Value.Type.ARRAY;
import static org.ojai.Value.Type.MAP;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.Map;
import java.util.Stack;

import org.ojai.Document;
import org.ojai.DocumentBuilder;
import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;
import org.ojai.util.Decimals;

import com.google.common.base.Preconditions;
import com.mapr.db.MapRDB;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.DBList;
import com.mapr.db.rowcol.InsertContext;
import com.mapr.db.rowcol.InsertContext.OpType;
import com.mapr.db.rowcol.KeyValue;

// Creates the streaming rowcol format which is sent to the jni and server layers
public class DBDocumentBuilder implements DocumentBuilder {

  private DBDocumentImpl dbRecord;
  private DBList curList;
  private boolean isClosed;
  private Stack<ContainerContext> ctxStack;
  private ContainerContext currentContext;

  public DBDocumentBuilder() {
    dbRecord = new DBDocumentImpl();
    InsertContext ctx = new InsertContext();
    dbRecord.setRootFlags(ctx);
    curList = null;
    ctxStack = new Stack<ContainerContext>();

    isClosed = false;
    currentContext = ContainerContext.NULL;
  }

  public boolean inMap() {
    return currentContext.getType() == null
        || currentContext.getType() == MAP;
  }

  @Override
  public DBDocumentBuilder put(String field, String value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, int value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, ODate value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, boolean value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, byte value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, short value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, long value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, float value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, double value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, BigDecimal value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, byte[] value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFromArray(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, byte[] value, int off, int len) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(ByteBuffer.wrap(value, off, len)), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, ByteBuffer value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, OInterval value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder putNewMap(String field) {
    preparePut();
    DBDocumentImpl newRecord = (DBDocumentImpl)MapRDB.newDocument();
    dbRecord.insertKeyValue(field, newRecord, ctxStack.size() == 1);
    dbRecord = newRecord;
    currentContext = ctxStack.push(new ContainerContext(MAP, field, newRecord));
    return this;
  }

  @Override
  public DBDocumentBuilder putNewArray(String field) {
    preparePut();
    curList = new DBList(OpType.NONE);
    currentContext = ctxStack.push(new ContainerContext(ARRAY, field, curList));
    return this;
  }

  @Override
  public DBDocumentBuilder putNull(String field) {
    preparePut();
    dbRecord.insertKeyValue(field, new KeyValue(Type.NULL), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, Value value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, Document value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  /* Advanced Map Methods */
  @Override
  public DBDocumentBuilder putDecimal(String field, int unscaledValue, int scale) {
    preparePut();
    return put(field, Decimals.convertIntToDecimal(unscaledValue, scale));
  }

  @Override
  public DBDocumentBuilder putDecimal(String field, long unscaledValue, int scale) {
    preparePut();
    return put(field, Decimals.convertLongToDecimal(unscaledValue, scale));
  }

  @Override
  public DBDocumentBuilder putDecimal(String field, long decimalValue) {
    preparePut();
    return put(field, new BigDecimal(decimalValue));
  }

  @Override
  public DBDocumentBuilder putDecimal(String field, double decimalValue) {
    preparePut();
    return put(field, new BigDecimal(decimalValue));
  }

  @Override
  public DBDocumentBuilder putDecimal(String field, byte[] unscaledValue, int scale) {
    return put(field, Decimals.convertByteToBigDecimal(unscaledValue, scale));
  }

  @Override
  public DBDocumentBuilder put(String field, OTime value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder put(String field, OTimestamp value) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(value), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder putDate(String field, int days) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(ODate.fromDaysSinceEpoch(days)), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder putTime(String field, int millis) {
    if (millis > MILLISECONDS_IN_A_DAY) {
      throw new IllegalArgumentException("Long value exceeds "
          + Long.toString(MILLISECONDS_IN_A_DAY) + " " + Long.toString(millis));
    }
    return put(field, OTime.fromMillisOfDay(millis));
  }

  @Override
  public DBDocumentBuilder putTimestamp(String field, long timeMillis) {
    return put(field, new OTimestamp(timeMillis));
  }

  @Override
  public DBDocumentBuilder putInterval(String field, long durationInMs) {
    preparePut();
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(new OInterval(durationInMs)), ctxStack.size() == 1);
    return this;
  }

  @Override
  public DBDocumentBuilder putInterval(String field, int months, int days, int milliseconds) {
    preparePut();
    long total_milliseconds = milliseconds + (days + (long) months * 30) * MILLISECONDS_IN_A_DAY;
    dbRecord.insertKeyValue(field, KeyValueBuilder.initFrom(new OInterval(total_milliseconds)), ctxStack.size() == 1);
    return this;
  }

  private void preparePut() {
    checkContext(MAP);
  }

  private void prepareAdd() {
    checkContext(ARRAY);
    if (currentContext.getType() == ARRAY) {
      currentContext.incrementIndex();
    }
  }

  private void checkContext(Type type) {
    if (isClosed) {
      throw new IllegalStateException("Writer is closed for put and add");
    }
    Preconditions.checkState(currentContext.getType() == type,
        "Mismatch in writeContext. Expected %s but found %s",
        type, currentContext.getType());
  }

  private void addElementToList(KeyValue child) {
    prepareAdd();
    child.setOpTypeAndFlags(null, false);
    curList.addToDBListWithFlags(child);
  }

  /* Array Methods */

  @Override
  public DocumentBuilder setArrayIndex(int index) {
    checkContext(ARRAY);
    int lastIndex = currentContext.getIndex();
    if (index <= lastIndex) {
      throw new IllegalArgumentException(String.format(
          "Specified index %d is not larger than the last written index %d",
          index, lastIndex));
    }
    int nullCount = index - lastIndex;
    for (int i = 1; i < nullCount; i++) {
      addNull();
    }
    return this;
  }

  @Override
  public DBDocumentBuilder add(boolean value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(String value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(byte value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(short value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(int value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(long value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(float value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(double value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(BigDecimal value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(OTime value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(ODate value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(OTimestamp value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(OInterval value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(byte[] value) {
    addElementToList(KeyValueBuilder.initFrom(ByteBuffer.wrap(value)));
    return this;
  }

  @Override
  public DBDocumentBuilder add(byte[] value, int off, int len) {
    addElementToList(KeyValueBuilder.initFrom(ByteBuffer.wrap(value, off, len)));
    return this;
  }

  @Override
  public DBDocumentBuilder add(ByteBuffer value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder addNull() {
    addElementToList(KeyValueBuilder.initFromNull());
    return this;
  }

  @Override
  public DBDocumentBuilder add(Value value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  @Override
  public DBDocumentBuilder add(Document value) {
    addElementToList(KeyValueBuilder.initFrom(value));
    return this;
  }

  /* Advanced Array Methods */
  @Override
  public DBDocumentBuilder addNewArray() {
    prepareAdd();
    curList = new DBList(OpType.NONE);
    currentContext = ctxStack.push(new ContainerContext(ARRAY, null, curList));
    return this;
  }

  @Override
  public DBDocumentBuilder addNewMap() {
    if (currentContext.getType() == MAP) {
      throw new IllegalStateException("Context mismatch : addNewMap() can not be called in a Map");
    }

    if (currentContext.getType() != null) {
      prepareAdd();
      dbRecord = new DBDocumentImpl();
      InsertContext ctx = new InsertContext();
      dbRecord.setRootFlags(ctx);
    }

    currentContext = ctxStack.push(new ContainerContext(MAP, null, dbRecord));
    return this;
  }

  @Override
  public DBDocumentBuilder addDecimal(long decimalValue) {
    return add(new BigDecimal(decimalValue));
  }

  @Override
  public DBDocumentBuilder addDecimal(double decimalValue) {
    return add(new BigDecimal(decimalValue));
  }

  @Override
  public DBDocumentBuilder addDecimal(int unscaledValue, int scale) {
    return add(Decimals.convertIntToDecimal(unscaledValue, scale));
  }

  @Override
  public DBDocumentBuilder addDecimal(long unscaledValue, int scale) {
    return add(Decimals.convertLongToDecimal(unscaledValue, scale));
  }

  @Override
  public DBDocumentBuilder addDecimal(byte[] unscaledValue, int scale) {
    return add(Decimals.convertByteToBigDecimal(unscaledValue, scale));
  }

  @Override
  public DBDocumentBuilder addDate(int days) {
    return add(ODate.fromDaysSinceEpoch(days));
  }

  @Override
  public DBDocumentBuilder addTime(int millis) {
    if (millis > MILLISECONDS_IN_A_DAY) {
      throw new IllegalArgumentException("Long value exceeds "
          + Long.toString(MILLISECONDS_IN_A_DAY) + " " + Long.toString(millis));
    }
    return add(OTime.fromMillisOfDay(millis));
  }

  @Override
  public DBDocumentBuilder addTimestamp(long timeMillis) {
    return add(new OTimestamp(timeMillis));
  }

  @Override
  public DBDocumentBuilder addInterval(long durationInMs) {
    return add(new OInterval(durationInMs));
  }

  /* Lifecycle methods */
  @Override
  public DBDocumentBuilder endArray() {
    checkContext(ARRAY);
    if (curList == null) {
      throw new IllegalStateException("The array was not started");
    }

    //pop curList
    ContainerContext nextStage = ctxStack.pop();
    currentContext = ctxStack.peek();
    if (currentContext.getType() == ARRAY) {
      DBList l = curList;
      curList = (DBList)currentContext.getKv();
      curList.addToDBListWithFlags(l);
    } else {
      dbRecord = (DBDocumentImpl)currentContext.getKv();
      dbRecord.insertKeyValue(nextStage.getFieldName(), curList, ctxStack.size() == 1 /*isRoot*/);
    }
    return this;
  }

  @Override
  public DBDocumentBuilder endMap() {
    preparePut();
    ctxStack.pop();
    if (!ctxStack.empty()) {
      currentContext = ctxStack.peek();
      if (currentContext.getType() == MAP) {
        dbRecord = (DBDocumentImpl)currentContext.getKv();
      } else {
        curList = (DBList)currentContext.getKv();
        curList.addToDBListWithFlags(dbRecord);
      }
    } else {
      isClosed = true;
    }
    return this;
  }

  @Override
  public Document getDocument() {
    if (!isClosed) {
      throw new IllegalStateException("Record is not written completely");
    }
    return dbRecord;
  }

  @Override
  public DBDocumentBuilder put(String field, Map<String, Object> value) {
    return put(field, KeyValueBuilder.initFrom(value));
  }

}
