/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.ojai;

import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Map.Entry;
import java.util.Stack;

import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.annotation.API;
import org.ojai.exceptions.TypeException;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;
import org.ojai.util.Types;

import com.mapr.db.ControlInfo;
import com.mapr.db.rowcol.ControlInfoImpl;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.DBList;
import com.mapr.db.rowcol.KeyValue;
import com.mapr.db.rowcol.KeyValueWithTS;

@API.Internal
public class DBDOMDocumentReader implements DBDocumentReaderBase {

  //define data structures
  private Stack<IteratorWithType> stateStack = null;
  private IteratorWithType currentItr = null;
  private EventType nextEvent = null;
  private EventType currentEvent = null;
  private KeyValue dbValue;
  private String key = null;
  private KeyValue rootKeyValue = null;
  private KeyValue idValue = null;

  public DBDOMDocumentReader(KeyValue value) {
    this(value, null);
  }

  public DBDOMDocumentReader(KeyValue value, KeyValue idValue) {
    stateStack = new Stack<IteratorWithType>();
    /* analyze the type of value object and initialize events and stacks */
    dbValue = value;
    Type type = value.getType();
    nextEvent = Types.getEventTypeForType(type);
    if (!type.isScalar()) {
      stateStack.push(new IteratorWithType(value));
    }
    rootKeyValue = value;
    this.idValue = idValue;
  }

  /* Main method for traversing the DFS structure.
   * We use a stack to keep track of iterators at each level of
   * the graph model. Each call to iterator next() can give us either a map
   * or non-map. If it's a map, we need to return field name first followed
   * by the START_MAP token. If it's non-map and non-array, then we have
   * FIELD_NAME followed by value type. In case of array, we return FIELD_NAME
   * and START_ARRAY. Note that an array is returned as a single JsonValue
   * from iterator. Therefore, we should not traverse the iterator (using next())
   * once we have the array. We need to process array elements sequentially. Since
   * we have to determine two types of tokens when we process a node, we have two
   * variables et and nextEt.
   */
  @SuppressWarnings("unchecked")
  private void processNextNode() {
    if (stateStack.empty()) {
      /* We are done with the document */
      nextEvent = null;
      return;
    }

    currentItr = stateStack.peek();
    if (currentItr.hasNext()) {
      Object o = currentItr.next();
      if (inMap()) {
        dbValue = ((Entry<String, KeyValueWithTS>) o).getValue();
        key = ((Entry<String, KeyValueWithTS>)o).getKey();
      } else { //inside array
        dbValue = KeyValueBuilder.initFromObject(o);
      }
      nextEvent = Types.getEventTypeForType(dbValue.getType());
      if (!dbValue.getType().isScalar()) {
        stateStack.push(new IteratorWithType(key, dbValue));
      }
    } else {
      IteratorWithType iter = stateStack.pop();
      dbValue = iter.getValue();
      key = iter.getKey();
      nextEvent = (iter.getType() == Type.MAP) ? EventType.END_MAP : EventType.END_ARRAY;
      currentItr = stateStack.isEmpty() ? null : stateStack.peek();
    }
  }

  @Override
  public EventType next() {
    currentEvent = null;
    if (nextEvent != null) {
      currentEvent = nextEvent;
      nextEvent = null;
    } else {
      processNextNode();
      currentEvent = nextEvent;
      nextEvent = null;
    }
    return currentEvent;
  }

  private void checkEventType(EventType event) throws TypeException {
    if (currentEvent != event) {
      throw new TypeException(String.format(
          "Event type mismatch. The operation requires %s, but found %s",
          event, currentEvent));
    }
  }

  @Override
  public boolean inMap() {
    return currentItr == null
        || currentItr.getType() == Type.MAP;
  }

  @Override
  public int getArrayIndex() {
    if (inMap()) {
      throw new IllegalStateException("Not traversing an array!");
    }
    return currentItr.previousIndex();
  }

  @Override
  public String getFieldName() {
    if (!inMap()) {
      throw new IllegalStateException("Not traversing a map!");
    }
    return key;
  }

  @Override
  public byte getByte() {
    checkEventType(EventType.BYTE);
    return dbValue.getByte();
  }

  @Override
  public short getShort() {
    checkEventType(EventType.SHORT);
    return dbValue.getShort();
  }

  @Override
  public int getInt() {
    checkEventType(EventType.INT);
    return dbValue.getInt();
  }

  @Override
  public long getLong() {
    checkEventType(EventType.LONG);
    return dbValue.getLong();
  }

  @Override
  public float getFloat() {
    checkEventType(EventType.FLOAT);
    return dbValue.getFloat();
  }

  @Override
  public double getDouble() {
    checkEventType(EventType.DOUBLE);
    return dbValue.getDouble();
  }

  @Override
  public BigDecimal getDecimal() {
    checkEventType(EventType.DECIMAL);
    return dbValue.getDecimal();
  }

  @Override
  public int getDecimalPrecision() {
    BigDecimal d = getDecimal();
    if (d != null) {
      return d.precision();
    }
    return 0;
  }

  @Override
  public int getDecimalScale() {
    BigDecimal d = getDecimal();
    if (d != null) {
      return d.scale();
    }
    return 0;
  }

  @Override
  public int getDecimalValueAsInt() {
    BigDecimal d = getDecimal();
    if (d != null) {
      return d.intValueExact();
    }
    return 0;
  }

  @Override
  public long getDecimalValueAsLong() {
    BigDecimal d = getDecimal();
    if (d != null) {
      return d.longValueExact();
    }
    return 0;
  }

  @Override
  public ByteBuffer getDecimalValueAsBytes() {
    BigDecimal decimal = getDecimal();
    if (decimal != null) {
      BigInteger decimalInteger = decimal.unscaledValue();
      byte[] bytearray = decimalInteger.toByteArray();
      return ByteBuffer.wrap(bytearray);
    }
    return null;
  }

  @Override
  public boolean getBoolean() {
    checkEventType(EventType.BOOLEAN);
    return dbValue.getBoolean();
  }

  @Override
  public String getString() {
    checkEventType(EventType.STRING);
    return dbValue.getString();
  }

  @Override
  public long getTimestampLong() {
    checkEventType(EventType.TIMESTAMP);
    return dbValue.getTimestampAsLong();
  }

  @Override
  public OTimestamp getTimestamp() {
    checkEventType(EventType.TIMESTAMP);
    return dbValue.getTimestamp();
  }

  @Override
  public int getDateInt() {
    checkEventType(EventType.DATE);
    return dbValue.getDateAsInt();
  }

  @Override
  public ODate getDate() {
    checkEventType(EventType.DATE);
    return dbValue.getDate();
  }

  @Override
  public int getTimeInt() {
    checkEventType(EventType.TIME);
    return dbValue.getTimeAsInt();
  }

  @Override
  public OTime getTime() {
    checkEventType(EventType.TIME);
    return dbValue.getTime();
  }

  @Override
  public OInterval getInterval() {
    checkEventType(EventType.INTERVAL);
    return dbValue.getInterval();
  }

  @Override
  public int getIntervalDays() {
    return getInterval().getDays();
  }

  @Override
  public long getIntervalMillis() {
    return getInterval().getTimeInMillis();
  }

  @Override
  public ByteBuffer getBinary() {
    checkEventType(EventType.BINARY);
    return dbValue.getBinary();
  }

  @Override
  public ControlInfo getControlInfo() {
    if ((currentEvent == EventType.END_MAP) || (currentEvent == EventType.END_ARRAY)) {
      throw new UnsupportedOperationException("Can not return timeDescriptor for eventType END_MAP or END_ARRAY");
    }

    byte timeDesc = (dbValue != null) ? dbValue.getTimeDescriptor() : rootKeyValue.getTimeDescriptor();
    return new ControlInfoImpl(timeDesc);

  }

  @Override
  public Value getId() {
    return idValue;
  }

  private class IteratorWithType implements ListIterator<Object> {
    final Iterator<?> i;
    final KeyValue value;
    final String key;

    IteratorWithType(KeyValue value) {
      this(null, value);
    }

    IteratorWithType(String key, KeyValue value) {
      this.key = key;
      this.value = value;
      this.i = (value.getType() == Type.MAP)
          ? ((DBDocumentImpl) value).iterator()
          : ((DBList) value).getSparseListIterator();
    }

    public String getKey() {
      return key;
    }

    public KeyValue getValue() {
      return value;
    }

    @Override
    public boolean hasNext() {
      return i.hasNext();
    }

    @Override
    public Object next() {
      return i.next();
    }

    @Override
    public void remove() {
      throw new UnsupportedOperationException();
    }

    @Override
    public String toString() {
      return (getType() == Type.ARRAY ? "ListIterator@" : "MapIterator@")
          + hashCode();
    }

    Type getType() {
      return value.getType();
    }

    @Override
    public boolean hasPrevious() {
      checkList();
      return ((ListIterator<?>) i).hasPrevious();
    }

    @Override
    public Object previous() {
      checkList();
      return ((ListIterator<?>) i).previous();
    }

    @Override
    public int nextIndex() {
      checkList();
      return ((ListIterator<?>) i).nextIndex();
    }

    @Override
    public int previousIndex() {
      checkList();
      return ((ListIterator<?>) i).previousIndex();
    }

    @Override
    public void set(Object e) {
      throw new UnsupportedOperationException();
    }

    @Override
    public void add(Object e) {
      throw new UnsupportedOperationException();
    }

    private void checkList() {
      if (getType() != Type.ARRAY) {
        throw new UnsupportedOperationException();
      }
    }
  }

}
