/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import java.util.Arrays;

import org.ojai.annotation.API;
import org.ojai.store.QueryCondition;

import com.mapr.db.TabletInfo;
import com.mapr.db.impl.ConditionNode.RowkeyRange;
import com.mapr.fs.jni.MapRConstants;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

@API.Internal
public class TabletInfoImpl implements TabletInfo, Comparable<TabletInfoImpl> {

  final private ConditionImpl condition;
  final private String[] locations;
  final private long estimatedSize;
  final private long estimatedNumRows;
  final private RowkeyRange rowKeyRange;

  public TabletInfoImpl(final ConditionImpl condition, final String[] locations,
      final long estimatedSize, final long estimatedNumRows) {
    this.condition = condition;
    this.locations = locations;
    this.estimatedSize = estimatedSize;
    this.estimatedNumRows = estimatedNumRows;
    this.rowKeyRange = condition.getRowkeyRanges().get(0);
  }

  @Override
  public QueryCondition getCondition() {
    return condition;
  }

  @Override
  public String[] getLocations() {
    return locations;
  }

  @Override
  public long getEstimatedSize() {
    return estimatedSize;
  }

  @Override
  public long getEstimatedNumRows() {
    return estimatedNumRows;
  }

  public byte[] getStartRow() {
    return rowKeyRange.startRow;
  }

  public byte[] getStopRow() {
    return rowKeyRange.stopRow;
  }

  @API.Internal
  public RowkeyRange getRowKeyRange() {
    return rowKeyRange;
  }

  /**
   * Return true if the given row falls in this region.
   */
  public boolean containsRow(byte[] row) {
    return Bytes.compareTo(row, rowKeyRange.startRow) >= 0 &&
      (Bytes.compareTo(row, rowKeyRange.stopRow) < 0 ||
       Bytes.equals(rowKeyRange.stopRow, MapRConstants.EMPTY_BYTE_ARRAY));
  }

  @Override
  public String toString() {
    return "TabletInfoImpl [condition=" + condition.getRowkeyRanges() + ", locations=" + Arrays.toString(locations)
        + ", estimatedSize=" + estimatedSize + ", estimatedNumRows=" + estimatedNumRows + "]";
  }

  @Override
  public int compareTo(TabletInfoImpl o) {
    if (o == null) {
      return 1;
    }

    // Compare start keys.
    int result = Bytes.compareTo(this.rowKeyRange.startRow, o.rowKeyRange.startRow);
    if (result != 0) {
      return result;
    }
    // Compare end keys.
    result = Bytes.compareTo(this.rowKeyRange.stopRow, o.rowKeyRange.stopRow);
    if (result != 0) {
      if (this.rowKeyRange.startRow.length != 0
          && this.rowKeyRange.stopRow.length == 0) {
        return 1; // this is last tablet
      } else if (o.rowKeyRange.startRow.length != 0
          && o.rowKeyRange.stopRow.length == 0) {
        return -1; // o is the last tablet
      }
      return result;
    }

    return 0;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((rowKeyRange == null) ? 0 : rowKeyRange.hashCode());
    result = prime * result + Arrays.hashCode(locations);
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    TabletInfoImpl other = (TabletInfoImpl) obj;
    if (rowKeyRange == null) {
      if (other.rowKeyRange != null)
        return false;
    } else if (!rowKeyRange.equals(other.rowKeyRange))
      return false;
    if (!Arrays.equals(locations, other.locations))
      return false;
    return true;
  }

}
