/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import static com.mapr.db.impl.Constants.DEFAULT_INSERTION_ORDER;

import java.util.List;

import org.apache.hadoop.fs.Path;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.mapr.db.FamilyDescriptor;
import com.mapr.db.TableDescriptor;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.TableAces;
import com.mapr.fs.proto.Dbserver.TableAttr;
import com.mapr.fs.proto.Dbserver.TableAttr.Builder;
import com.mapr.fs.proto.Marlincommon.MarlinTableAttr;
import com.mapr.fs.tables.TableProperties;

public class TableDescriptorImpl implements TableDescriptor {

  private Path path_;

  /**
   * Mutable table attributes.
   */
  private final TableAttr.Builder tableAttr_;

  /**
   * Mutable table ACEs.
   */
  private final TableAces.Builder tableAces_;

  /**
   * List of FamilyDescriptor of this table.
   */
  private List<FamilyDescriptor> families_;

  public TableDescriptorImpl() {
    this(null, null, new TableProperties(
      TableAttr.getDefaultInstance(), TableAces.getDefaultInstance(), null, false), DEFAULT_INSERTION_ORDER);
  }

  public TableDescriptorImpl(boolean insertionOrder) {
    this(null, null, new TableProperties(
      TableAttr.getDefaultInstance(), TableAces.getDefaultInstance(), null, false), insertionOrder);
  }

  public TableDescriptorImpl(Path tablePath) {
    this(tablePath, null, new TableProperties(
      TableAttr.getDefaultInstance(), TableAces.getDefaultInstance(), null, false), DEFAULT_INSERTION_ORDER);
  }

  TableDescriptorImpl(TableDescriptorImpl other) {
    path_ = other.path_;
    tableAttr_ = other.tableAttr_.clone();
    tableAces_ = other.tableAces_.clone();
    families_ = other.getFamilies();
  }

  TableDescriptorImpl(Path tablePath,
                      List<ColumnFamilyAttr> cfList, TableProperties tableProps,
                      boolean insertionOrder) {
    path_ = tablePath;
    tableAttr_ = tableProps.getAttr().toBuilder();
    tableAces_ = tableProps.getAces().toBuilder();
    families_ = Lists.newArrayList();
    if (cfList != null) {
      for (ColumnFamilyAttr cfAttr : cfList) {
        families_.add(new FamilyDescriptorImpl(cfAttr));
      }
    }
    tableAttr_.setJson(true);
    setInsertionOrder(insertionOrder);
  }

  @Override
  public Path getPath() {
    return path_;
  }

  @Override
  public TableDescriptorImpl setPath(String tablePath) {
    return setPath(new Path(tablePath));
  }

  @Override
  public TableDescriptorImpl setPath(Path tablePath) {
    path_ = tablePath;
    return this;
  }

  @Override
  public boolean isInsertionOrder() {
    return tableAttr_.hasInsertionOrder()
        && tableAttr_.getInsertionOrder();
  }

  @Override
  public TableDescriptorImpl setInsertionOrder(boolean insertionOrder) {
    tableAttr_.setInsertionOrder(insertionOrder);
    return this;
  }

  @Override
  public boolean isBulkLoad() {
    return tableAttr_.hasBulkLoad()
        && tableAttr_.getBulkLoad();
  }

  @Override
  public TableDescriptorImpl setBulkLoad(boolean bulkLoad) {
    tableAttr_.setBulkLoad(bulkLoad);
    return this;
  }

  @Override
  public boolean isAutoSplit() {
    return tableAttr_.hasAutoSplit()
        && tableAttr_.getAutoSplit();
  }

  @Override
  public TableDescriptorImpl setAutoSplit(boolean autoSplit) {
    tableAttr_.setAutoSplit(autoSplit);
    return this;
  }

  @Override
  public long getSplitSize() {
    return tableAttr_.getRegionSizeMB();
  }

  @Override
  public TableDescriptorImpl setSplitSize(long splitSizeMB) {
    tableAttr_.setRegionSizeMB(splitSizeMB);
    return this;
  }

  @Override
  public FamilyDescriptor getFamily(String familyName) {
    for (FamilyDescriptor family : families_) {
      if (family.getName().equals(familyName)) {
        return family.clone();
      }
    }
    return null;
  }

  @Override
  public int getNumFamilies() {
    return families_.size();
  }

  @Override
  public List<FamilyDescriptor> getFamilies() {
    List<FamilyDescriptor> families = Lists.newArrayList();
    for (FamilyDescriptor family : families_) {
      families.add(family.clone());
    }
    return families;
  }

  @Override
  public TableDescriptorImpl setFamilies(List<FamilyDescriptor> families) {
    Preconditions.checkNotNull(families, "families can not be null");
    //TODO: validate for duplicate family name and path
    //      and ensure 'default' is the first column family
    families_ = Lists.newArrayList(families);
    return this;
  }

  @Override
  public TableDescriptorImpl addFamily(FamilyDescriptor family) {
    if (families_ == null) {
      families_ = Lists.newArrayList();
    }
    for (FamilyDescriptor familyDescriptor : families_) {
      if (family.getName().equals(familyDescriptor.getName())
          || family.getJsonFieldPath().equals(familyDescriptor.getJsonFieldPath())) {
        throw new IllegalArgumentException("A family with same name or json path already exist.\n" + familyDescriptor);
      }
    }
    //TODO: validate for duplicate family name and path
    families_.add(family);
    return this;
  }

  /*&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&*
   *                Internal APIs, not part of the interface           *
   *&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&*/

  /**
   * @return number of seconds after which deletes are purged from the tables
   */
  public long getDeleteTTL() {
    return tableAttr_.getDeleteTTL();
  }

  /**
   * Set the number of seconds after which deletes will be purged from the table.
   *
   * @param ttlSec TTL value in seconds
   * @return
   */
  public TableDescriptor setDeleteTTL(long ttlSec) {
    tableAttr_.setDeleteTTL(ttlSec);
    return this;
  }

  /**
   * @return size above which values are not cached in the memIndex
   */
  public int getMaxValueSizeInMemIndex() {
    return tableAttr_.getMaxValueSzInMemIndex();
  }

  /**
   * Set the size above which values are not cached in the memIndex
   *
   * @param valueSz Size in bytes
   * @return
   */
  public TableDescriptorImpl setMaxValueSizeInMemIndex(int valueSz) {
    tableAttr_.setMaxValueSzInMemIndex(valueSz);
    return this;
  }

  /**
   * @return {@code true} if client compression is enabled for the table.
   */
  public boolean getClientCompression() {
    return tableAttr_.getClientCompression();
  }

  /**
   * Enable/disable client compression for this table.
   *
   * @return {@code this} for chain invocation
   */
  public TableDescriptorImpl setClientCompression(boolean val) {
    tableAttr_.setClientCompression(val);
    return this;
  }

  /**
   * @return {@code true} if this table is a Marlin stream
   */
  public boolean isStream() {
    return tableAttr_.hasIsMarlinTable()
        && tableAttr_.getIsMarlinTable();
  }

  /**
   * Create this table as Marlin stream. Only applicable when creating the table.
   *
   * @return {@code this} for chain invocation
   */
  public TableDescriptorImpl setStream() {
    tableAttr_.setIsMarlinTable(true);
    return this;
  }

  /**
   * @return {@code true} if this stream allows auto-create of topics.
   */
  public boolean isStreamAutoCreate() {
    if (tableAttr_.hasMarlinAttr())
      return tableAttr_.getMarlinAttr().getAutoCreateTopics();
    else
      return false;
  }

  /**
   * Enable/Disable auto-create of topics. Only applicable on streams.
   *
   * @param autoCreate {@code true} if auto-create of topics is allowed.
   * @return {@code this} for chain invocation
   */
  public TableDescriptorImpl setStreamAutoCreate(boolean autoCreate) {
    MarlinTableAttr mattr;
    mattr = MarlinTableAttr.newBuilder(tableAttr_.getMarlinAttr())
                           .setAutoCreateTopics(autoCreate)
                           .build();

    tableAttr_.setMarlinAttr(mattr);
    return this;
  }

  /**
   * @return default number of partitions for topics in the stream.
   */
  public int getStreamDefaultPartitions() {
    if (tableAttr_.hasMarlinAttr())
      return tableAttr_.getMarlinAttr().getDefaultNumFeedsPerTopic();
    else
      return 0;
  }

  /**
   * Set default number of partitions for topics in the stream.
   *
   * @param numPartitions number of partitions
   * @return {@code this} for chain invocation
   */
  public TableDescriptorImpl setStreamDefaultPartitions(int numPartitions) {
    MarlinTableAttr mattr;
    mattr = MarlinTableAttr.newBuilder(tableAttr_.getMarlinAttr())
                           .setDefaultNumFeedsPerTopic(numPartitions)
                           .build();

    tableAttr_.setMarlinAttr(mattr);
    return this;
  }

  /**
   * @return TableAttr for this TableDescriptor
   */
  public Builder getTableAttr() {
    return tableAttr_.clone();
  }

  /**
   * @return TableAces for this TableDescriptor
   */
  public TableAces.Builder getTableAces() {
    return tableAces_.clone();
  }

  @Override
  public String toString() {
    return "{\"Path\": \"" + getPath()
        + "\", \"InsertionOrder\": " + isInsertionOrder()
        + ", \"BulkLoad\": " + isBulkLoad()
        + ", \"AutoSplit\": " + isAutoSplit()
        + ", \"SplitSize\": " + getSplitSize()
        + ", \"Families\": " + getFamilies() + "}";
  }

  @Override
  public TableDescriptorImpl clone() {
    return new TableDescriptorImpl(this);
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((families_ == null) ? 0 : families_.hashCode());
    result = prime * result + ((path_ == null) ? 0 : path_.hashCode());
    result = prime * result + (isAutoSplit() ? 1 : 0);
    result = prime * result + (isBulkLoad() ? 1 : 0);
    result = prime * result + getMaxValueSizeInMemIndex();
    result = prime * result + (isInsertionOrder() ? 1 : 0);
    result = prime * result + (int)getSplitSize();
    result = prime * result + (int)getDeleteTTL();
    result = prime * result + (getClientCompression() ? 1 : 0);
    result = prime * result + (isStream() ? 1 : 0);
    if (isStream()) {
      result = prime * result + (isStreamAutoCreate() ? 1 : 0);
      result = prime * result + getStreamDefaultPartitions();
    }
    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    TableDescriptorImpl other = (TableDescriptorImpl) obj;
    if (families_ == null) {
      if (other.families_ != null)
        return false;
    } else if (!families_.equals(other.families_))
      return false;
    if (path_ == null) {
      if (other.path_ != null)
        return false;
    } else if (!path_.equals(other.path_)) {
      return false;
    }
    if (isAutoSplit() != other.isAutoSplit()) {
      return false;
    }
    if (isBulkLoad() != other.isBulkLoad()) {
      return false;
    }
    if (getMaxValueSizeInMemIndex() != other.getMaxValueSizeInMemIndex()) {
      return false;
    }
    if (isInsertionOrder() != other.isInsertionOrder()) {
      return false;
    }
    if (getDeleteTTL() != other.getDeleteTTL()) {
      return false;
    }
    if (getSplitSize() != other.getSplitSize()) {
      return false;
    }

    if (getClientCompression() != other.getClientCompression()) {
      return false;
    }
    if (isStream() != other.isStream()) {
      return false;
    }
    if (isStream()) {
      if (isStreamAutoCreate() != other.isStreamAutoCreate())
        return false;
      if (getStreamDefaultPartitions() != other.getStreamDefaultPartitions())
        return false;
    }
    return true;
  }

}
