/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import static com.mapr.db.impl.Constants.CF_COLUMN_SEPARATOR;
import static com.mapr.db.impl.Constants.DEFAULT_FAMILY;
import static com.mapr.db.impl.Constants.DUMMY_FIELDPATH_V;
import static com.mapr.db.impl.Constants.DUMMY_FIELD_V;
import static com.mapr.db.impl.Constants.FIELD_SEPARATOR;
import static com.mapr.db.impl.Constants.FIELD_SEPARATOR_CHAR;
import static com.mapr.fs.jni.MapRConstants.EMPTY_END_ROW;
import static com.mapr.fs.jni.MapRConstants.EMPTY_START_ROW;
import static com.mapr.fs.jni.MapRConstants.MAPRFS_PREFIX;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.ojai.Document;
import static org.ojai.DocumentConstants.ID_KEY;
import org.ojai.FieldPath;
import org.ojai.FieldSegment;
import org.ojai.annotation.API;
import org.ojai.store.QueryCondition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.protobuf.ByteString;
import com.mapr.db.Table.TableOption;
import com.mapr.db.exceptions.DBException;
import com.mapr.db.exceptions.ExceptionHandler;
import com.mapr.db.exceptions.TableNotFoundException;
import com.mapr.db.impl.ConditionNode.RowkeyRange;
import com.mapr.db.impl.MapRDBTableImpl.TablePrivateOption;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.RowcolCodec;
import com.mapr.db.rowcol.SerializationAction;
import com.mapr.db.rowcol.SerializedFamilyInfo;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.MapRHTable;
import com.mapr.fs.jni.MapRConstants.PutConstants;
import com.mapr.fs.jni.MapRConstants.RowConstants;
import com.mapr.fs.jni.MapRGet;
import com.mapr.fs.jni.MapRPut;
import com.mapr.fs.jni.MapRResult;
import com.mapr.fs.jni.MapRRowConstraint;
import com.mapr.fs.jni.MapRScan;
import com.mapr.fs.proto.Common.FileCompressionType;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.Qualifier;
import com.mapr.fs.proto.Dbserver.RowConstraint;
import com.mapr.fs.proto.Dbserver.SchemaFamily;
import com.mapr.fs.proto.Dbserver.SchemaFamily.Builder;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

@API.Internal
public class MapRDBTableImplHelper {
  private static Logger logger = LoggerFactory.getLogger(MapRDBTableImplHelper.class);

  public static SchemaFamily getDefaultSchemaFamily(String cfName, Integer ttl) {
    Builder builder = SchemaFamily.newBuilder()
        .setName(cfName)
        .setInMemory(false)
        .setMinVersions(0)
        .setMaxVersions(1)
        .setTtl(ttl)
        .setCompression(FileCompressionType.FCT_LZ4);

    return builder.build();
  }

  public static ColumnFamilyAttr getColumnFamilyAttr(String cfName, String cfPath, Integer ttl) {
    if (cfName.equals(DEFAULT_FAMILY)) {
      return ColumnFamilyAttr.newBuilder()
          .setSchFamily(getDefaultSchemaFamily(cfName, ttl))
          .build();
    } else {
      return ColumnFamilyAttr.newBuilder()
          .setSchFamily(getDefaultSchemaFamily(cfName, ttl))
          .setJsonFamilyPath(cfPath)
          .build();
    }
  }

  // The only reason default family will not be found is if table doesnt exist
  public static int getDefaultFamilyId(MapRHTable maprTable) throws DBException {
    try {
      return maprTable.getFamilyId(DEFAULT_FAMILY);
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "getDefaultFamilyId()");
    }
  }

  //Computes the JSON path from 'cf:qualifier' tuple. The qualifier
  //should be specified *without* the DUMMY_FIELD_V prefix for non-default CF
  //qualifiers.
  public static FieldPath cfQualifierToJsonPath(String cf, String qual, List<ColumnFamilyAttr> cfAttrs)
      throws NullPointerException,IllegalArgumentException {
  if (cf == null) {
    throw new IllegalArgumentException("ColumnFamily should be non-null");
  }

    FieldPath jsonPath = null;
    if (cf.equals(Constants.DEFAULT_FAMILY)) {
      return (qual != null ? FieldPath.parseFrom(qual) : FieldPath.EMPTY);
    }

    for (ColumnFamilyAttr colFam : cfAttrs) {
      if (colFam.getSchFamily().getName().equals(cf)) {
        jsonPath = FieldPath.parseFrom(colFam.getJsonFamilyPath());
        break;
      }
    }

    if (qual != null) {
      FieldPath qualPath = FieldPath.parseFrom(qual);
      FieldSegment childSeg = qualPath.segmentAfterAncestor(FieldPath.parseFrom(DUMMY_FIELD_V));
      if (childSeg.equals(FieldPath.EMPTY.getRootSegment())){ //For cases where path = cf_jsonpath
        return jsonPath;
      }
      return jsonPath.cloneWithNewChild(childSeg);
    }
    return jsonPath;
  }

  // Compute the corresponding 'cf:qualifier' tuple for given jsonpath
  public static String jsonPathToCfQualifier(String jsonPath, List<ColumnFamilyAttr> cfAttrs) {
    String cfQual = null;

    Map<FieldPath, Integer> pathIdMap = new TreeMap<FieldPath, Integer>();
    Map<Integer, String> cfIdMap = new TreeMap<Integer, String>();

    for (ColumnFamilyAttr cf: cfAttrs) {
      pathIdMap.put(FieldPath.parseFrom(cf.getJsonFamilyPath()), cf.getSchFamily().getId());
      cfIdMap.put(cf.getSchFamily().getId(), cf.getSchFamily().getName());
    }

    Map<Integer, FieldPath> cfMapping = getOneCFQualifier(FieldPath.parseFrom(jsonPath),
                                                          pathIdMap,
                                                          true /*exclude embedded CFs*/);

    //There should be only one element in the map
    for (Entry<Integer, FieldPath> e : cfMapping.entrySet()) {
      String cfName = cfIdMap.get(e.getKey());
      FieldPath qual = e.getValue();
      if (qual != null && !qual.equals(FieldPath.EMPTY)) {
        cfQual = cfName + CF_COLUMN_SEPARATOR + e.getValue().asPathString();
      } else {
        cfQual = cfName;
      }
      break;
    }
    return cfQual;
  }

  // Return the one projections in cfid:column form and optionally include
  //embedded families
  public static
  Map<Integer, FieldPath> getOneCFQualifier(FieldPath field,
                                            Map<FieldPath, Integer> pathIdMap,
                                            boolean excludeEmbeddedCfs)
  {
    /*
     * Start with assuming "default" column family.
     */
    Integer familyId = pathIdMap.get(FieldPath.EMPTY);
    if (familyId == null) {
      throw new IllegalStateException("Unable to find family id for the default column family.");
    }

    /*
     * Now see if the field path is under a
     * column family's FieldPath.
     */
    FieldPath lastParent = null;
    Integer defaultCFId = familyId;
    for (Entry<FieldPath, Integer> kv : pathIdMap.entrySet()) {
      FieldPath familyPath = kv.getKey();
      if (familyPath.equals(FieldPath.EMPTY)) continue;

      FieldSegment childFP = field.segmentAfterAncestor(familyPath);
      if (childFP != null) {
        if (lastParent == null || lastParent.isAtOrAbove(familyPath)) {
          lastParent = familyPath;
          familyId = kv.getValue();
        }
      }
    }

    TreeMap<Integer, FieldPath> cfColMap = new TreeMap<Integer, FieldPath>();
    FieldPath col = field;
    if (lastParent == null) { //default cf
      if (!col.equals(FieldPath.EMPTY)) {
        cfColMap.put(familyId, col);
      } else {
        cfColMap.put(familyId, null);  //root of the document
      }
    } else {
      FieldSegment colSeg = field.segmentAfterAncestor(lastParent);

      if (colSeg.isNamed() &&
          colSeg.getNameSegment().getName().equals(FieldPath.EMPTY.asPathString())) { //Root of the CF
        col = null; //include entire CF
      } else {
        if (colSeg.isIndexed()) { //CF jsonpath is assigned an array
          col = FieldPath.parseFrom(DUMMY_FIELD_V + colSeg.asPathString(false /*quoted*/));
        } else {
          col = FieldPath.parseFrom(DUMMY_FIELD_V).cloneWithNewChild(colSeg);
        }
      }
      cfColMap.put(familyId, col);
    }

    if (!excludeEmbeddedCfs) {
      for (Entry<FieldPath, Integer> kv : pathIdMap.entrySet()) {
        FieldPath cfPath = kv.getKey();
        Integer cfId = kv.getValue();
        if (cfPath.isAtOrBelow(field) && cfId != familyId) {
          cfColMap.put(kv.getValue(), null);
        }
      }
    }
    return cfColMap;
  }

  //Accept a list of JSON fieldpaths to return a Map of Column Families and respective Collection
  //of Columns
  public static Map<Integer, List<String>> getMultipleCFQualifiers(Map<FieldPath, Integer> pathCFIdMap,
                                                                   boolean excludeEmbeddedFamilies,
                                                                   String...paths) {
    if (paths == null) {
     return null;
    }

    Map<Integer, List<String>> finalCFColMap = new TreeMap<Integer, List<String>>();
    for (String fp : paths) {
      Map<Integer, FieldPath> cfColMap = getOneCFQualifier(FieldPath.parseFrom(fp), pathCFIdMap, excludeEmbeddedFamilies);
      for (Entry<Integer, FieldPath> cfEntry : cfColMap.entrySet()) {
        Integer famId = cfEntry.getKey();
        FieldPath col = cfEntry.getValue();
        //look for existing list or create a new one
        List<String> colList = finalCFColMap.get(famId);
        if (colList == null) {
          colList = new ArrayList<String>();
          finalCFColMap.put(famId, colList);
        } else if (colList.isEmpty()) {
          //Entire CF included
          continue;
        }
        if (col != null) {
          colList.add(col.asPathString());
        } else {
          //Clear if entire CF is included
          colList.clear();
        }
      }
    }
    return finalCFColMap;
  }

  // paths includes condition related ones also.
  // After server fix for bug 18186 this can be removed.
  public static MapRRowConstraint toRowConstraint(
      MapRDBTableImpl table, String...paths) throws DBException {
    MapRRowConstraint rc = new MapRRowConstraint();
    Map<FieldPath, Integer> pathCFidMap = table.idPathMap();

    rc.maxVersions = 1;
    rc.minStamp = RowConstants.DEFAULT_MIN_STAMP;
    rc.maxStamp = RowConstants.DEFAULT_MAX_STAMP;
    rc.readAllCfs = table.readAllCfs();

    if (table.getDeletes()) {
      rc.getDeletes = true;
    }

    if (paths != null) {
      if (paths.length == 1 && paths[0].equals(ID_KEY)) {
        rc.idOnlyProjection = true;
      } else {
        Map<Integer, List<String>> pathsPerFam =
            getMultipleCFQualifiers(pathCFidMap, table.excludeEmbeddedFamily(), paths);

        if (logger.isTraceEnabled()) {
          logger.trace(" proj map : '{}'", pathsPerFam);
        }

        // create field paths and families array
        rc.numFamilies = pathsPerFam.size();
        rc.numColumns = paths.length;
        rc.families = new int[rc.numFamilies];
        rc.columnsPerFamily = new int[rc.numFamilies];
        rc.columns  = new byte[rc.numColumns][];

        int famIter = 0;
        int rccount = 0;
        for (Map.Entry<Integer, List<String>> entry : pathsPerFam.entrySet()) {
          Integer cfId = entry.getKey();
          List<String> cfPath = entry.getValue();

          rc.families[famIter] = cfId;
          rc.columnsPerFamily[famIter] = cfPath.size();
          famIter++;

          for (int j = 0; j < cfPath.size(); j++) {
            rc.columns[rccount] = Bytes.toBytes(cfPath.get(j));
            rccount++;
          }
        }

        rc.idOnlyProjection = false;
      }
    }
    return rc;
  }

  public static Document doGet(MapRDBTableImpl table,
      ByteBuffer id, boolean excludeId) throws DBException {
    return doGet(table, id, null, excludeId, (String[])null);
  }

  public static Document doGet(MapRDBTableImpl table,
      ByteBuffer id, boolean excludeId, String...paths) throws DBException {
    return doGet(table, id, null, excludeId, paths);
  }

  public static Document doGet(MapRDBTableImpl table,
      ByteBuffer id, QueryCondition c, boolean excludeId) throws DBException {
    return doGet(table, id, c, excludeId, (String[])null);
  }

  // If projection has a path which is superset (ancestor) of condition's path,
  // then return true. Or is same path also true. For ex: if projPaths has 'a'
  // and aCondPath is 'a.b', return true.
  private static boolean containsSuper(List<String> projPaths, String aCondPath) {
    if (projPaths.contains(aCondPath)) {
      return true;
    }

    return false;
  }

  static class CondAndProjPaths
  {
    String[] allPaths;
    Set<FieldPath> condPaths;

    public CondAndProjPaths() {
      allPaths = null;
      condPaths = null;
    }
  };

  // Concatenates condition paths and projection fields.
  // Does not add to cond if present in projPaths
  // Also removes elements from condPaths which are in projection path already.
  public static CondAndProjPaths setPaths(QueryCondition c, String[] projPaths,
                                          CondAndProjPaths ret) {
    Set<FieldPath> condPaths = ((c != null) ? ((ConditionImpl)c).getProjections()
                                : null);

    if (condPaths == null || (condPaths.size() == 0)) {
      ret.allPaths = projPaths;
      return ret;
    }

    // If projections are not present, do not add condition's fields
    if (projPaths == null || (projPaths.length == 0)) {
      ret.allPaths = projPaths;
      return ret;
    }

    List<String> addList = new ArrayList<String>();
    int numAdded = 0;
    List<String> paths = Arrays.asList(projPaths);
    Iterator<FieldPath> it = condPaths.iterator();
    while (it.hasNext()) {
      String cp = it.next().asPathString();
      if (containsSuper(paths, cp)) {
        it.remove();
        continue;
      }

      addList.add(cp);
      numAdded++;
    }
    assert(addList.size() == numAdded);

    String[] addArray = addList.toArray(new String[addList.size()]);
    ret.allPaths = new String[addArray.length + projPaths.length];
    System.arraycopy(addArray, 0, ret.allPaths, 0, numAdded);
    System.arraycopy(projPaths, 0, ret.allPaths, numAdded, projPaths.length);
    ret.condPaths = condPaths;

    return ret;
  }

  public static Document doGet(MapRDBTableImpl table, ByteBuffer id,
      QueryCondition c, boolean excludeId, String...paths) throws DBException {
    MapRGet mrget = new MapRGet();
    mrget.result = new MapRResult();

    MapRDBTableImplHelper.CondAndProjPaths bothpaths =
        new MapRDBTableImplHelper.CondAndProjPaths();
    setPaths(c, paths, bothpaths);

    mrget.rowConstraint = MapRDBTableImplHelper.toRowConstraint(table,
                                                             bothpaths.allPaths);

    byte[] serFilt;
    if (c != null && !c.isEmpty()) {
      ByteBuffer serCond = ((ConditionImpl)c).cloneOptimized().getDescriptor(table.idPathMap()).getSerialized();
      serFilt =  new byte[serCond.capacity()];
      serCond.get(serFilt);
    } else {
      serFilt = null;
    }
    mrget.setFilter(serFilt);
    mrget.setEncodedResult(true);

    mrget.key = new byte[id.remaining()];
    int currentPosition = id.position();
    id.get(mrget.key);
    id.position(currentPosition);
    ByteBuffer keyBuf = ByteBuffer.wrap(mrget.key);

    try {
      table.maprTable().getJson(mrget);
      Map<Integer, ByteBuffer> cfbufs = mrget.getResult().getJsonByteBufs();
      if (cfbufs == null) {
        return null;
      }

      boolean shouldPrunePaths = (bothpaths.condPaths != null) &&
                                 (bothpaths.condPaths.size() != 0);

      Document record = RowcolCodec.decode(cfbufs, table.sortedByPath(),
                                               table.idToCFNameMap(),
                                               keyBuf, excludeId,
                                               true /*cachedbuffer*/,
                                               table.isKeepInsertionOrder(),
                                               table.decodeTimestamp(),
                                               false /*preserverDeleteFlags*/,
                                               shouldPrunePaths ? paths : null);

      if (shouldPrunePaths) {

       // If we need to remove some of the paths from returned document, we 
       // always decode the internal buffers. This is because the 
       // DBDocumentReader currently does not remove the non-projected paths
       ((DBDocumentImpl)record).getDOMFromCachedBuffer();

        // If it has no other projected field, skip the record
        if (record.size() == 0) {
          return null;
        }
      }

      return record;
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "doGet()");
    } finally {
      if (mrget.getArena() != 0) {
        table.maprTable().freeArena(mrget.getArena());
      }
    }
  }

  // Copies a map of jsonpath/CFid tuples into an ordered by CFId list of the
  // same tuples. Done once per table.
  public static
  List<Map.Entry<FieldPath, Integer>> sortByValueToList(Map<FieldPath, Integer> map) {
    List<Map.Entry<FieldPath, Integer>> list =
        new LinkedList<Map.Entry<FieldPath, Integer> >(map.entrySet());
    Collections.sort(list, new Comparator<Map.Entry<FieldPath, Integer>>() {
      @Override
      public int compare(Map.Entry<FieldPath, Integer> o1,
                         Map.Entry<FieldPath, Integer> o2) {
        return (o1.getValue()).compareTo( o2.getValue());
      }
     });

    return list;
  }

  // Copies a path/cfid map into another map sorted by path
  public static
  BiMap<FieldPath, Integer> sortByPath(Map<FieldPath, Integer> map) {
    Map<FieldPath, Integer> result = new TreeMap<FieldPath, Integer>(map);
    ImmutableBiMap.Builder<FieldPath, Integer> mapBuilder = ImmutableBiMap.builder();
    for (Map.Entry<FieldPath, Integer> entry : result.entrySet()) {
      mapBuilder.put(entry.getKey(), entry.getValue());
    }
    return mapBuilder.build();
  }

  // Copies a map of jsonpath/CFid tuples into a map sorted in cfid order.
  // Done once per table.
  public static
  BiMap<FieldPath, Integer> sortByValue(BiMap<FieldPath, Integer> map) {
    List<Map.Entry<FieldPath, Integer>> list =
        new LinkedList<Map.Entry<FieldPath, Integer> >(map.entrySet());
    Collections.sort(list, new Comparator<Map.Entry<FieldPath, Integer>>() {
      @Override
      public int compare(Map.Entry<FieldPath, Integer> o1,
                         Map.Entry<FieldPath, Integer> o2) {
        return (o1.getValue()).compareTo( o2.getValue());
      }
     });

    ImmutableBiMap.Builder<FieldPath, Integer> mapBuilder = ImmutableBiMap.builder();
    for (Map.Entry<FieldPath, Integer> entry : list) {
      mapBuilder.put(entry.getKey(), entry.getValue());
    }
    return mapBuilder.build();
  }

  // Uses a sorted by id and jsonpath for CF list to create put/del object.
  public static MapRPut toMapRPut(ByteBuffer id, int[] famIds,
                                  ByteBuffer[] bbuf, byte type) {
    MapRPut  mrput = new MapRPut();
    int      bufLen = 0;
    mrput.numFamilies = famIds.length;
    mrput.numCells = mrput.numFamilies; // one cell per family
    mrput.type = type;
    mrput.rowTimeStamp = 0;
    mrput.key = new byte[id.remaining()];
    id.get(mrput.key);

    mrput.families = new int[mrput.numFamilies];
    mrput.keyvals  = null;
    mrput.cellsPerFamily = new int[mrput.numFamilies];
    mrput.encBuffers = new ByteBuffer[mrput.numFamilies];
    mrput.encPositions = new int[mrput.numFamilies];
    mrput.encLimits = new int[mrput.numFamilies];

    int i = 0;
    int prevCfId = 0;
    for (int entry : famIds) {
      mrput.families[i] = entry;
      if (i != 0) {
        assert(mrput.families[i] > prevCfId);
        prevCfId = mrput.families[i];
      } else {
        prevCfId = mrput.families[i];
      }

      mrput.cellsPerFamily[i] = 1;
      if (bbuf == null || bbuf[i] == null) {
        mrput.encBuffers[i] =  null;
        mrput.encPositions[i] = mrput.encLimits[i] = 0;
      } else {
        mrput.encBuffers[i] = bbuf[i];
        mrput.encPositions[i] = bbuf[i].position();
        mrput.encLimits[i] = bbuf[i].limit();
        assert(bbuf[i].limit() != 0);
        bufLen += (mrput.encLimits[i] - mrput.encPositions[i]);
        mrput.encBuffers[i].rewind();
      }

      i++;
    }

    mrput.rowTotalBytes = (mrput.key.length + Bytes.SIZEOF_LONG + bufLen);

    mrput.setIsEncoded(true);

    return mrput;
  }

  // object to fill in when creating multiple maps using the same rpc call.
  static class ComboMap {
    BiMap<FieldPath, Integer> pathToId;
    BiMap<Integer, String>    idToName;
  }

  // Get the jsonpath/cfname and cfid mapping from server
  public static void getMaps(List<ColumnFamilyAttr> cfAttrs, ComboMap cm)
      throws TableNotFoundException {
    BiMap<FieldPath, Integer> ipathToId = HashBiMap.create();
    BiMap<Integer, String> iidToName = HashBiMap.create();

    for (ColumnFamilyAttr cfAttr : cfAttrs) {
      logger.trace("CF#{}, name: '{}', path: '{}'",
                   cfAttr.getSchFamily().getId(),
                   cfAttr.getSchFamily().getName(),
                   cfAttr.getJsonFamilyPath());
      ipathToId.put(FieldPath.parseFrom(cfAttr.getJsonFamilyPath()),
                    cfAttr.getSchFamily().getId());
      iidToName.put(cfAttr.getSchFamily().getId(),
                    cfAttr.getSchFamily().getName());
    }

    cm.idToName = iidToName;
    cm.pathToId = ipathToId;
  }

  // Get the jsonpath and cfid mapping from server
  public static BiMap<FieldPath, Integer>
      getCFIdPathMap(List<ColumnFamilyAttr> cfAttrs) throws TableNotFoundException {
    ImmutableBiMap.Builder<FieldPath, Integer> mapBuilder = ImmutableBiMap.builder();
    for (ColumnFamilyAttr cfAttr : cfAttrs) {
      logger.trace("CF#{}, name: '{}', path: '{}'",
          cfAttr.getSchFamily().getId(),
          cfAttr.getSchFamily().getName(),
          cfAttr.getJsonFamilyPath());
      mapBuilder.put(FieldPath.parseFrom(cfAttr.getJsonFamilyPath()), cfAttr.getSchFamily().getId());
    }

    return mapBuilder.build();
  }

  // Given a list of serialized info, return the encoded buffers and familyIds
  // based on the action types.
  public static EncodedBufFamIdInfo getEncBufsAndFamilyIds(SerializedFamilyInfo []info) {
    int numFams = 0;

    for (int i = 0; i < info.length; i++) {
      if (info[i].getAction() != SerializationAction.NO_ACTION) {
        numFams++;
      }
    }

    ByteBuffer[] encVals = new ByteBuffer[numFams];
    int[] famIds = new int[numFams];
    int idx = 0;
    for (int i = 0; i < info.length; i++) {
      if (info[i].getAction() != SerializationAction.NO_ACTION) {
        famIds[idx] = info[i].getFamilyId();
      }
      switch(info[i].getAction()) {
      case NO_ACTION:
        break;
      case SET:
        encVals[idx] = info[i].getByteBuffer();
        idx++;
        break;
      case DELETE_FAMILY:
        encVals[idx] = null;
        idx++;
        break;
      default :
        assert(false);
        break;
      }
    }
    return new EncodedBufFamIdInfo(encVals, famIds);
  }

  public static void insertOrReplace(MapRDBTableImpl table, ByteBuffer id,
      SerializedFamilyInfo []info) throws DBException {
    EncodedBufFamIdInfo ebf = getEncBufsAndFamilyIds(info);
    MapRPut mput = toMapRPut(id, ebf.familyIds, ebf.encBuffers, PutConstants.TYPE_PUT_ROW);

    try {
      if (table.getOption(TableOption.BUFFERWRITE).getBoolean()) {
        table.maprTable().put(mput);
      } else {
        table.maprTable().syncPut(mput, /* flush and block = */true);
      }
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "insertOrReplace()");
    }
  }

  public static MapRScan toMapRScan(MapRDBTableImpl table) throws DBException {
    return toMapRScan(table, null, (String[])null);
  }

  public static MapRScan toMapRScan(MapRDBTableImpl table, String...paths)
      throws DBException {
    return toMapRScan(table, null, paths);
  }

  public static MapRScan toMapRScan(MapRDBTableImpl table,
      QueryCondition c, String...paths) throws DBException {
    MapRScan maprscan = new MapRScan();
    maprscan.batch = 0;
    maprscan.caching = 0;
    maprscan.startRow = EMPTY_START_ROW;
    maprscan.stopRow = EMPTY_END_ROW;
    maprscan.filter = null;
    maprscan.shouldDecompress = table.shouldDecompress();

    // set constraints
    maprscan.rowConstraint = MapRDBTableImplHelper.toRowConstraint(table, paths);

    byte[] serFilt = null;
    if (c != null) {
      ConditionImpl cImpl = ((ConditionImpl)c).cloneOptimized();
      // currently we support a single key range
      RowkeyRange keyRange = cImpl.getRowkeyRanges().get(0);
      maprscan.startRow = keyRange.getStartRow();
      maprscan.stopRow = keyRange.getStopRow();
      if (!cImpl.isEmpty()) {
        serFilt = Bytes.getBytes(cImpl.getDescriptor(table.idPathMap()).getSerialized());
      }
      logger.trace("QueryCondition: '{}', startRow: '{}', stopRow: '{}'", cImpl,
                   Bytes.toStringBinary(maprscan.startRow),
                   Bytes.toStringBinary(maprscan.stopRow));
    }
    maprscan.setFilter(serFilt);

    return maprscan;
  }

  private static Configuration config = new Configuration();
  public static MapRFileSystem getMapRFileSystem() {
    return getMapRFileSystem(config);
  }

  public static MapRFileSystem getMapRFileSystem(Configuration config) {
    try {
      return (MapRFileSystem)
          FileSystem.get(new URI(MAPRFS_PREFIX), config);
    } catch (URISyntaxException | IOException e) {
      throw new DBException("Could not get filesystem", e);
    }
  }

  /**
   * Parse the comma-separated list of families/columns passed in with
   * "-columns" to return a list of the column families to be copied.
   * @param columnSpec
   * @return
   */
  public static List<String> getColumnFamiliesList(String columnSpec) {
    List<String> cfList = new ArrayList<String>();
    if (columnSpec != null) {
      String[] cols = columnSpec.split(",");
      String column = null;
      for (String col : cols) {
        if (col.contains(":")) {
          String[] names = col.split(":");
          column = names[0];
        } else {
          column = col;
        }
        if (!cfList.contains(column)) {
          cfList.add(column);
        }
      }
    }
    return cfList;
  }


  public static Map <Integer, List<String>>
    condFieldPathMapToCondFieldPathStrMap(Map <Integer, ? extends Set<FieldPath>> inMap) {

    if (inMap == null) return null;
    Map <Integer, List <String>> condFieldMap;
    condFieldMap = new LinkedHashMap<Integer, List<String>>();
    for (Map.Entry<Integer, ? extends Set<FieldPath>> e : inMap.entrySet()) {
      Integer famId = e.getKey();
      Set<FieldPath> fps = e.getValue();
      List <String> fpsStringList = new ArrayList<String>();
      for (FieldPath fp : fps) {
        fpsStringList.add(fp.toString());
      }
      condFieldMap.put(famId, fpsStringList);
    }
    return condFieldMap;
  }


  /*
   * Merge two sorted list of maps which map the family id => list field path in family
   * Input maps should be sorted by the familyId and list of paths should be sorted by
   * path comparator.
   * Output list is also sorted by the familyId and fields within family
   */
  public static Map<Integer, List<String>>
  mergeFieldPathList(Map<Integer, List<String>> map1, Map<Integer, List<String>> map2) {

    if (map1 == null) {
      return map2;
    }
    if (map2 == null) {
      return map1;
    }

    Map<Integer, List<String>> result = new LinkedHashMap<Integer, List<String>>();
    Iterator<Entry<Integer, List<String>>> i1 = map1.entrySet().iterator();
    Iterator<Entry<Integer, List<String>>> i2 = map2.entrySet().iterator();


    Map.Entry<Integer, List<String>> e1;
    e1 = i1.hasNext() ? i1.next() : null;

    Map.Entry<Integer, List<String>> e2;
    e2 = i2.hasNext() ? i2.next() : null;


    while (e1 != null || e2 != null) {
      if (e1 != null && e2 != null) {
        // both lists are not empty so merge them by the familyid
        int cmp = e1.getKey().compareTo(e2.getKey());
        if (cmp == 0) {
          // merge and sort the list of field paths with
          result.put(e1.getKey(), mergeSortedFieldsPaths(e1.getValue(), e2.getValue()));
          e1 = i1.hasNext() ? i1.next() : null;
          e2 = i2.hasNext() ? i2.next() : null;
        } else if (cmp < 0) {
          result.put(e1.getKey(), e1.getValue());
          e1 = i1.hasNext() ? i1.next() : null;
        } else {
          result.put(e2.getKey(), e2.getValue());
          e2 = i2.hasNext() ? i2.next() : null;
        }
      } else if (e1 != null){
        // put remaining elements from list1
        result.put(e1.getKey(), e1.getValue());
        while (i1.hasNext()) {
          e1 = i1.next();
          result.put(e1.getKey(), e1.getValue());
        }
        e1 = null;
      } else {
        // Put remaining elements from list2
        result.put(e2.getKey(), e2.getValue());
        while (i2.hasNext()) {
          e2 = i2.next();
          result.put(e2.getKey(), e2.getValue());
        }
        e2 = null;
      }
    }
    return result;
  }

  private static List<String> mergeSortedFieldsPaths(
      List<String> l1, List<String> l2) {
    if (l1 == null) return l2;
    if (l2 == null) return l1;

    List <String> result = new ArrayList<String>();
    Iterator <String> i1 = l1.iterator();
    Iterator <String> i2 = l2.iterator();

    String s1 = null;
    String s2 = null;

    while(i1.hasNext() || (s1 != null) || i2.hasNext() || (s2 != null)) {
      if (i2.hasNext() == false  && (s2 == null)) {
        if (s1 != null) {
          result.add(s1);
          s1 = null;
        }
        while (i1.hasNext()) {
          result.add(i1.next());
        }
      } else if (i1.hasNext() == false && (s1 == null)) {
        if (s2 != null) {
          result.add(s2);
          s2 = null;
        }
        while (i2.hasNext()) {
          result.add(i2.next());
        }
      } else {
        if (s1 == null) {
          s1 = i1.next();
        }
        if (s2 == null) {
          s2 = i2.next();
        }

        FieldPath f1 = FieldPath.parseFrom(s1);
        FieldPath f2 = FieldPath.parseFrom(s2);
        int cmp = f1.compareTo(f2);
        if (cmp < 0) {
          result.add(s1);
          s1 = null;
        } else if (cmp == 0){
          result.add(s1);
          s1 = null;
          s2 = null;
        } else {
          result.add(s2);
          s2 = null;
        }
      }
    }
    return result;
  }

  /*
   * Converts the map of familyid => list of fields in the family to serialized read constraint
   */
  public static byte[] fieldPathsToSerRowConstraint(Map<Integer, List<String>> fieldsMap) {
    if (fieldsMap == null) return null;

    RowConstraint.Builder c = RowConstraint.newBuilder();
    for (Map.Entry<Integer, List<String>> e : fieldsMap.entrySet()) {
      Qualifier.Builder b = Qualifier.newBuilder();
      b.setFamily(e.getKey());

      for (String qualStr : e.getValue()) {
        b.addQualifiers(ByteString.copyFromUtf8(qualStr));
      }
      c.addQualifiers(b.build());
    }
    return c.build().toByteArray();
  }

}
