/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import static com.mapr.db.impl.ConditionNode.HASH_OF_KEYSAMPLING_FILTER;
import static com.mapr.db.impl.Constants.BUFFERWRITE_STR;
import static com.mapr.db.impl.Constants.EXCLUDEID_STR;
import static com.mapr.db.impl.Constants.JSON_INSERT_BATCH_SIZE;
import static com.mapr.db.impl.Constants.KEEPINSERTIONORDER_STR;
import static com.mapr.db.impl.Constants.NUM_ASYNC_READERS;
import static com.mapr.db.impl.Constants.ROW_EXISTS_CONDITION;
import static com.mapr.db.impl.Constants.ROW_NOT_EXISTS_CONDITION;
import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;
import static com.mapr.fs.jni.MapRConstants.EMPTY_END_ROW;
import static com.mapr.fs.jni.MapRConstants.EMPTY_START_ROW;
import static org.ojai.DocumentConstants.ID_FIELD;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.mapr.fs.tables.TableProperties;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.ojai.Document;
import org.ojai.DocumentStream;
import org.ojai.FieldPath;
import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.annotation.API;
import org.ojai.store.DocumentMutation;
import org.ojai.store.OpListener;
import org.ojai.store.QueryCondition;
import org.ojai.store.QueryCondition.Op;
import org.ojai.store.exceptions.DocumentExistsException;
import org.ojai.store.exceptions.DocumentNotFoundException;
import org.ojai.store.exceptions.FailedOp;
import org.ojai.store.exceptions.MultiOpException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.BiMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.mapr.db.Table;
import com.mapr.db.TableDescriptor;
import com.mapr.db.TabletInfo;
import com.mapr.db.exceptions.AccessDeniedException;
import com.mapr.db.exceptions.DBException;
import com.mapr.db.exceptions.DBRetryException;
import com.mapr.db.exceptions.ExceptionHandler;
import com.mapr.db.exceptions.OpNotPermittedException;
import com.mapr.db.exceptions.ReadOnlyException;
import com.mapr.db.exceptions.TableClosedException;
import com.mapr.db.exceptions.TableNotFoundException;
import com.mapr.db.impl.ConditionNode.RowkeyRange;
import com.mapr.db.ojai.DBDocumentStream;
import com.mapr.db.rowcol.KeyValue;
import com.mapr.db.rowcol.MutationImpl;
import com.mapr.db.rowcol.RowcolCodec;
import com.mapr.db.rowcol.SerializedFamilyInfo;
import com.mapr.db.util.ByteBufs;
import com.mapr.fs.MapRHTable;
import com.mapr.fs.MapRResultScanner;
import com.mapr.fs.MapRTabletScanner;
import com.mapr.fs.jni.MapRConstants.PutConstants;
import com.mapr.fs.jni.MapRPut;
import com.mapr.fs.jni.MapRScan;
import com.mapr.fs.jni.MapRUpdateAndGet;
import com.mapr.fs.proto.Dbfilters.FilterMsg;
import com.mapr.fs.proto.Dbfilters.KeySamplingFilterProto;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.SpaceUsage;
import com.mapr.fs.proto.Dbserver.TabletDesc;
import com.mapr.fs.proto.Dbserver.TabletStatResponse;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;
import com.mapr.utils.Collections;

// Implementation of the table class into which records (json docs/subdocs)
// can be added or existing ones can deleted/mutated.
@API.Internal
public class MapRDBTableImpl implements Table {
  private static Logger logger = LoggerFactory.getLogger(MapRDBTableImpl.class);
  public static final String PRESERVE_TS_STR = "maprdb.table.impl.preserve_timestamps";
  public static final String GET_DELETES_STR = "maprdb.table.impl.get_deletes";
  public static final String EXCLUDE_EMBEDDEDFAMILY_STR = "maprdb.table.impl.exclude_embeddedfamily";
  public static final String DECOMPRESS_STR = "maprdb.table.impl.decompress";
  public static final String READ_ALL_CFS_STR = "maprdb.table.impl.read_all_cfs";

  public enum TablePrivateOption {
    PRESERVE_TIMESTAMP,
    GET_DELETES,
    DECODE_TIMESTAMP,
    EXCLUDE_EMBEDDEDFAMILY,
    DECOMPRESS,
    READ_ALL_CFS
  }

  private static final long DEFAULT_BLOCK_SIZE = 8 * 1024L;

  enum BatchingType { INSERTORREPLACE, INSERT, REPLACE, DELETE }

  final MapRHTable maprTable;
  boolean closed;
  LinkedHashMap<String, KeyValue> tableOptionsMap;
  LinkedHashMap<TablePrivateOption, Object> tablePrivateOptionsMap;
  BiMap<FieldPath, Integer> unSortedIdPathMap;
  BiMap<FieldPath, Integer> idPathMap;
  BiMap<FieldPath, Integer> sortedByPathMap;
  BiMap<Integer, String> idToCFNameMap;
  List<Map.Entry<FieldPath, Integer>> sortedById;

  private final TableDescriptorImpl tableDesc;
  private final boolean insOrderInTable;

  public MapRDBTableImpl(Path tablePath, Configuration config)
      throws DBException, TableNotFoundException {
    List<ColumnFamilyAttr> cfAttrs = null;
    if (config.getInt("fs.mapr.threads", 0) == 0) {
      config.setInt("fs.mapr.threads", 64);
    }

    try {
      maprTable = new MapRHTable();
      maprTable.init(config, tablePath);
      if (!maprTable.getMapRFS().exists(tablePath)) {
        throw new TableNotFoundException(tablePath);
      }
      cfAttrs = maprTable.getMapRFS().listColumnFamily(maprTable.getTablePath(), false);
      closed = false;
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "<init>()");
    }

    tableOptionsMap = new LinkedHashMap<String, KeyValue>();
    tablePrivateOptionsMap = new LinkedHashMap<TablePrivateOption, Object>();
    tableOptionsMap.put(EXCLUDEID_STR, KeyValueBuilder.initFrom(false));
    tableOptionsMap.put(BUFFERWRITE_STR, KeyValueBuilder.initFrom(config.getBoolean(BUFFERWRITE_STR, true)));
    TableProperties tblprop = maprTable.getInode().getTableProperties();
    if (!tblprop.getAttr().getJson()) {
        throw new OpNotPermittedException(tablePath + ": OJAI APIs are currently not supported with binary tables!");
    }
    insOrderInTable = tblprop.getAttr().getInsertionOrder();
    tableOptionsMap.put(KEEPINSERTIONORDER_STR, KeyValueBuilder.initFrom(insOrderInTable));
    tablePrivateOptionsMap.put(TablePrivateOption.PRESERVE_TIMESTAMP, config.getBoolean(PRESERVE_TS_STR, false));
    tablePrivateOptionsMap.put(TablePrivateOption.GET_DELETES, config.getBoolean(GET_DELETES_STR, false));
    tablePrivateOptionsMap.put(TablePrivateOption.DECODE_TIMESTAMP, false);
    tablePrivateOptionsMap.put(TablePrivateOption.EXCLUDE_EMBEDDEDFAMILY, config.getBoolean(EXCLUDE_EMBEDDEDFAMILY_STR, false));
    tablePrivateOptionsMap.put(TablePrivateOption.DECOMPRESS, config.getBoolean(DECOMPRESS_STR, true));
    tablePrivateOptionsMap.put(TablePrivateOption.READ_ALL_CFS, config.getBoolean(READ_ALL_CFS_STR, true));

    MapRDBTableImplHelper.ComboMap ret = new MapRDBTableImplHelper.ComboMap();
    MapRDBTableImplHelper.getMaps(cfAttrs, ret);
    idToCFNameMap = ret.idToName;
    unSortedIdPathMap = ret.pathToId;
    sortedById = MapRDBTableImplHelper.sortByValueToList(unSortedIdPathMap);
    idPathMap = MapRDBTableImplHelper.sortByValue(unSortedIdPathMap);
    sortedByPathMap = MapRDBTableImplHelper.sortByPath(unSortedIdPathMap);
    tableDesc = new TableDescriptorImpl(tablePath, cfAttrs, tblprop, insOrderInTable);
    if (logger.isTraceEnabled() && (idPathMap.size() > 1)) {
      logger.trace("CF Path to Id Map unsorted: '{}'", unSortedIdPathMap);
      logger.trace("CF id List: '{}'", sortedById);
      logger.trace("CF Path to Id Map sorted by id: '{}'", idPathMap);
      logger.trace("CF Path to Id Map sorted by path: '{}'", sortedByPathMap);
      logger.trace("CF id to CF name Map: '{}'", idToCFNameMap);
    }
  }

  public MapRHTable maprTable() { return maprTable; }
  public BiMap<FieldPath, Integer> idPathMap() { return idPathMap; }
  List<Map.Entry<FieldPath, Integer>> sortedById() { return sortedById; }
  public Map<FieldPath, Integer> sortedByPath() { return sortedByPathMap; }
  public Map<Integer, String> idToCFNameMap() { return idToCFNameMap; }

  @Override
  public MapRDBTableImpl setOption(TableOption option, boolean value) {
    switch (option) {
    case EXCLUDEID:
      tableOptionsMap.put(EXCLUDEID_STR, KeyValueBuilder.initFrom(value));
      break;
    case BUFFERWRITE:
      tableOptionsMap.put(BUFFERWRITE_STR, KeyValueBuilder.initFrom(value));
      break;
    case KEEPINSERTIONORDER:
      // If client is setting need insertion order, but server doesnt track it
      // (as table was not created with it), then we will ignore this setting
      if (value && !insOrderInTable) {
        logger.error("Cannot set insertion order if table " +
                     "creation did not specify it. Ignoring the setting.");
        return this;
      }

      tableOptionsMap.put(KEEPINSERTIONORDER_STR, KeyValueBuilder.initFrom(value));

      break;
    default:
      throw new IllegalArgumentException("Unknown table option " + option);
    }
    return this;
  }

  private boolean isExcludeId() {
    return tableOptionsMap.get(EXCLUDEID_STR).getBoolean();
  }

  private boolean isBufferWrite() {
    return tableOptionsMap.get(BUFFERWRITE_STR).getBoolean();
  }

  public boolean isKeepInsertionOrder() {
    return tableOptionsMap.get(KEEPINSERTIONORDER_STR).getBoolean();
  }

  public boolean getDeletes() {
    return getPrivateOption(TablePrivateOption.GET_DELETES);
  }

  public boolean isStream() {
    return tableDesc.isStream();
  }

  public boolean isPreserveTS() {
    return getPrivateOption(TablePrivateOption.PRESERVE_TIMESTAMP);
  }

  public boolean shouldDecompress() {
    return getPrivateOption(TablePrivateOption.DECOMPRESS);
  }

  public boolean decodeTimestamp() {
    return getPrivateOption(TablePrivateOption.DECODE_TIMESTAMP);
  }

  public void setExcludeEmbeddedFamily(boolean v) {
    setPrivateOption(TablePrivateOption.EXCLUDE_EMBEDDEDFAMILY, v);
  }

  public boolean excludeEmbeddedFamily() {
    return getPrivateOption(TablePrivateOption.EXCLUDE_EMBEDDEDFAMILY);
  }

  public void setReadAllCfs(boolean v) {
    setPrivateOption(TablePrivateOption.READ_ALL_CFS, v);
  }

  public boolean readAllCfs() {
    return getPrivateOption(TablePrivateOption.READ_ALL_CFS);
  }

  @Override
  public Value getOption(TableOption option) {
    switch (option) {
    case EXCLUDEID:
      return tableOptionsMap.get(EXCLUDEID_STR);
    case BUFFERWRITE:
      return tableOptionsMap.get(BUFFERWRITE_STR);
    case KEEPINSERTIONORDER:
      return tableOptionsMap.get(KEEPINSERTIONORDER_STR);
    default:
      throw new IllegalArgumentException("Unknown table option " + option);
    }
  }

  public MapRDBTableImpl setPrivateOption(TablePrivateOption option, boolean value) {
    switch(option) {
    case PRESERVE_TIMESTAMP:
    case GET_DELETES:
    case DECODE_TIMESTAMP:
    case EXCLUDE_EMBEDDEDFAMILY:
    case DECOMPRESS:
    case READ_ALL_CFS:
      tablePrivateOptionsMap.put(option, value);
      break;
    default:
      throw new IllegalArgumentException("Unknown table option " + option);
    }
    return this;
  }

  public boolean getPrivateOption(TablePrivateOption option) {
    switch(option) {
    case PRESERVE_TIMESTAMP:
    case GET_DELETES:
    case DECODE_TIMESTAMP:
    case EXCLUDE_EMBEDDEDFAMILY:
    case DECOMPRESS:
    case READ_ALL_CFS:
      return (Boolean) tablePrivateOptionsMap.get(option);
    default:
      throw new IllegalArgumentException("Unknown table option " + option);
    }
  }

  /**
   * Returns just the name of the table (not the full path).
   */
  @Override
  public String getName() {
    return getPath().getName();
  }

  /**
   * Returns the full path of the table.
   */
  @Override
  public Path getPath() {
    return tableDesc.getPath();
  }

  @Override
  public TableDescriptor getTableDescriptor() {
    return tableDesc;
  }

  /**
   * Flushes all the buffered update operations on the client.
   * @throws DBException if the flush failed or the flush of any buffered
   *         operation resulted in an error
   */
  @Override
  public void flush() throws DBException {
    try {
      maprTable.flushCommits();
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "flush()");
    }
  }

  private void checkClosed() throws DBException {
    if (closed) {
      throw new TableClosedException(
          "Cannot perform the op on a closed table " + getPath());
    }
  }

  /**
   * Reads a single full record identified by the id. A NULL record is returned if
   * the record with the given id doesn't exist in the table.
   *
   * @param id row key
   * @return record for the row if it exists, otherwise null
   * @throws TableNotFoundException when the table from which to get
   * the row does not exist
   */
  @Override
  public Document findById(String id) throws DBException {
    return _findById(IdCodec.encode(id), (QueryCondition )null, (String[])null);
  }

  @Override
  public Document findById(ByteBuffer id) throws DBException {
    return _findById(IdCodec.encode(id), (QueryCondition )null, (String[])null);
  }

  @Override
  public Document findById(Value id) throws DBException {
    return _findById(IdCodec.encode(id), (QueryCondition )null, (String[])null);
  }

  private String[] getStringPaths(FieldPath...fields) {
    if (fields == null) {
      return null;
    }
    String[] stringpaths = new String[fields.length];
    for (int i = 0; i < fields.length; i++) {
      stringpaths[i] = fields[i].asPathString();
    }
    return stringpaths;
  }

  @Override
  public Document findById(String id, FieldPath...fields) throws DBException {
    return findById(id, getStringPaths(fields));
  }

  @Override
  public Document findById(String id, String...paths) throws DBException {
    return _findById(IdCodec.encode(id), (QueryCondition )null, paths);
  }

  @Override
  public Document findById(ByteBuffer id, FieldPath... fields) throws DBException {
    return findById(id, getStringPaths(fields));
  }

  @Override
  public Document findById(ByteBuffer id, String... paths) throws DBException {
    return _findById(IdCodec.encode(id), (QueryCondition )null, paths);
  }

  //@Override
  public Document findById(Value id, FieldPath...fields) throws DBException {
    return findById(id, getStringPaths(fields));
  }

  //@Override
  public Document findById(Value id, String...paths) throws DBException {
    return _findById(IdCodec.encode(id), (QueryCondition )null, paths);
  }

  @Override
  public Document findById(String id, QueryCondition  c) throws DBException {
    return _findById(IdCodec.encode(id), c);
  }

  @Override
  public Document findById(ByteBuffer id, QueryCondition  c) throws DBException {
    return _findById(IdCodec.encode(id), c);
  }

  //@Override
  public Document findById(Value id, QueryCondition  c) throws DBException {
    return _findById(IdCodec.encode(id), c);
  }

  @Override
  public Document findById(String id, QueryCondition  c, FieldPath...fields)
      throws DBException {
    return findById(id, c, getStringPaths(fields));
  }

  @Override
  public Document findById(String id, QueryCondition c, String...paths)
      throws DBException {
    return _findById(IdCodec.encode(id), c, paths);
  }

  @Override
  public Document findById(ByteBuffer id, QueryCondition c, FieldPath... fields)
      throws DBException {
    return findById(id, c, getStringPaths(fields));
  }

  @Override
  public Document findById(ByteBuffer id, QueryCondition c, String... paths)
      throws DBException {
    return _findById(IdCodec.encode(id), c, paths);
  }

  //@Override
  public Document findById(Value id, QueryCondition c, String...paths)
      throws DBException {
    return _findById(IdCodec.encode(id), c, paths);
  }

  //@Override
  public Document findById(Value id, QueryCondition c, FieldPath...fields)
      throws DBException {
    return findById(id, c, getStringPaths(fields));
  }

  private Document _findById(ByteBuffer id, QueryCondition c, String...paths)
      throws DBException {
    checkClosed();
    Document d = null;
    while (true) {
      try {
        d = MapRDBTableImplHelper.doGet(this, id, c, isExcludeId(), paths);
        break;
      } catch (DBRetryException re) {
        updateSchema();
      }
    }

    return d;
  }

  /*
   * Async APIs
   */
  ExecutorService executor = Executors.newFixedThreadPool(NUM_ASYNC_READERS);

  class AsyncReader implements Runnable {
    OpListener cbListener;
    ByteBuffer encodedId;
    QueryCondition c;
    String[] paths;

    AsyncReader(OpListener list, ByteBuffer inId, QueryCondition inCond, String... inPaths) {
      cbListener = list;
      encodedId = inId;
      c  = inCond;
      paths = inPaths;
    }

    @Override
    public void run() {
      try {
        cbListener.onSuccess(_findById(encodedId, c, paths));
      } catch (Exception e) {
        cbListener.onFailure(e);
      }
    }
  }

  /**
   * Non-blocking read operations. Gets called back on the listener object
   * on success or failure.
   */
  @Override
  public void findById(OpListener listener, String id) {
    _findById(listener, IdCodec.encode(id), (QueryCondition)null, (String[])null);
  }

  @Override
  public void findById(OpListener listener, ByteBuffer id) {
    _findById(listener, IdCodec.encode(id), (QueryCondition)null, (String[])null);
  }

  //@Override
  public void findById(OpListener listener, Value id) {
    _findById(listener, IdCodec.encode(id), (QueryCondition)null, (String[])null);
  }

  @Override
  public void findById(OpListener listener, String id, FieldPath...fields) {
    findById(listener, id, getStringPaths(fields));
  }

  @Override
  public void findById(OpListener listener, String id, String... paths) {
    _findById(listener, IdCodec.encode(id), null, paths);
  }

  @Override
  public void findById(OpListener listener, ByteBuffer id, FieldPath... fields) {
    findById(listener, id, getStringPaths(fields));
  }

  @Override
  public void findById(OpListener listener, ByteBuffer id, String... paths) {
    _findById(listener, IdCodec.encode(id), null, paths);
  }

  //@Override
  public void findById(OpListener listener, Value id, String... paths) {
    _findById(listener, IdCodec.encode(id), null, paths);
  }

  //@Override
  public void findById(OpListener listener, Value id, FieldPath...fields) {
    findById(listener, id, getStringPaths(fields));
  }

  @Override
  public void findById(OpListener listener, String id, QueryCondition c) {
    _findById(listener, IdCodec.encode(id), c);
  }

  @Override
  public void findById(OpListener listener, ByteBuffer id, QueryCondition c) {
    _findById(listener, IdCodec.encode(id), c);
  }

  //@Override
  public void findById(OpListener listener, Value id, QueryCondition c) {
    _findById(listener, IdCodec.encode(id), c);
  }

  @Override
  public void findById(OpListener listener, String id, QueryCondition c, FieldPath...fields) {
    findById(listener, id, c, getStringPaths(fields));
  }

  @Override
  public void findById(OpListener listener, String id, QueryCondition c, String... paths) {
    _findById(listener, IdCodec.encode(id), c, paths);
  }

  @Override
  public void findById(OpListener listener, ByteBuffer id, QueryCondition c, FieldPath... fields) {
    findById(listener, id, c, getStringPaths(fields));
  }

  @Override
  public void findById(OpListener listener, ByteBuffer id, QueryCondition c, String... paths) {
    _findById(listener, IdCodec.encode(id), c, paths);
  }

  //@Override
  public void findById(OpListener listener, Value id, QueryCondition c, FieldPath...fields) {
    findById(listener, id, c, getStringPaths(fields));
  }

  //@Override
  public void findById(OpListener listener, Value id, QueryCondition c, String... paths) {
    _findById(listener, IdCodec.encode(id), c, paths);
  }

  private void _findById(OpListener listener, ByteBuffer id, QueryCondition c, String... paths) {
    executor.execute(new AsyncReader(listener, id, c, paths));
  }

  /**
   * Returns a record stream for all the rows in the table.
   *
   * @return record stream that can be used to retrieve each row in the table
   * @throws TableNotFoundException when the table from which to get any row
   * does not exist
   */
  @Override
  public DocumentStream find() throws DBException {
    return find((QueryCondition)null, (String[])null);
  }

  /**
   * Returns a record stream for all the rows in the table. Each Document will
   * contain only the fields that are specified in the argument. If a given
   * field doesn't exist in the row or the user doesn't have permissions to
   * access that field, then that field won't be returned in the read record.
   *
   * @return record stream that can be used to get rows with requested fields
   * @throws TableNotFoundException when the table from which to get any row
   * does not exist
   */
  @Override
  public DocumentStream find(String...paths)
      throws DBException {
    return find((QueryCondition)null, paths);
  }

  @Override
  public DocumentStream find(FieldPath...fields)
      throws DBException {
    return find((QueryCondition)null, getStringPaths(fields));
  }

  /**
   * Returns a record stream for all the rows in the table that match the
   * condition.
   *
   * @return record stream that can be used to get rows with requested fields
   * @throws TableNotFoundException when the table from which to get any row
   * does not exist
   */
  @Override
  public DocumentStream find(QueryCondition c) throws DBException {
    return find(c, (String[])null);
  }

  /**
   * Returns a record stream for all the rows in the table that match the
   * condition. Each Document will
   * contain only the fields that are specified in the argument. If a given
   * field doesn't exist in the row or the user doesn't have permission to
   * access that field, then that field won't be returned in the read record.
   *
   * @return record stream that can be used to get rows with requested fields
   * @throws TableNotFoundException when the table from which to get any row
   * does not exist
   */
  @Override
  public DocumentStream find(QueryCondition c, String...paths)
      throws DBException {
    checkClosed();
    DocumentStream d = null;
    while (true) {
      try {
        d = doFind(c, paths);
        break;
      } catch (DBRetryException re) {
        updateSchema();
      }
    }
    return d;
  }

  private DocumentStream doFind(QueryCondition c, String...paths)
      throws DBException {
    MapRDBTableImplHelper.CondAndProjPaths bothpaths =
        new MapRDBTableImplHelper.CondAndProjPaths();
    MapRDBTableImplHelper.setPaths(c, paths, bothpaths);
    MapRScan maprscan =
        MapRDBTableImplHelper.toMapRScan(this, c, bothpaths.allPaths);
    try {
      long id = maprTable.getInode().getScanner(maprscan);
      MapRResultScanner scanner = new MapRResultScanner(maprscan, maprTable, id);
      maprTable.addScanner(scanner);

      if (paths != null) {
        return new DBDocumentStream(scanner, isExcludeId(),
                                    this, bothpaths.condPaths, paths);
      } else {
        return new DBDocumentStream(scanner, isExcludeId(), this);
      }
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "find()");
    }
  }

  @Override
  public DocumentStream find(QueryCondition c, FieldPath...fields)
      throws DBException {
    return find(c, getStringPaths(fields));
  }

  /**
   * Inserts a new record in the table. The row key is either explicitly specified
   * as parameter "id" or it is implicitly specified as the field "_id" in the
   * passed record. If the row key is explicitly passed then the record should
   * not contain an "_id" field or its value should be the same as the explicitly
   * specified id. Otherwise, the operation will fail.
   *
   * If the row with the given key exists on the server, then it will be
   * overwritten by the new record.
   *
   * @param r JSON record as the new value for the given row
   * @throws TableNotFoundException when table does not exist to add this row
   * @throws AccessDeniedException user credentials mismatch
   * @throws ReadOnlyException when the table is not accepting writes
   * @throws OpNotPermittedException server returned EPERM
   */
  @Override
  public void insertOrReplace(Document r) throws DBException {
    Preconditions.checkNotNull(r, "Document being inserted cannot be null");

    Document mdbRec = RowcolCodec.getDBDocument(r);
    Value id = mdbRec.getId();
    Preconditions.checkArgument(id != null, "Document needs to have '_id' in the map as a key.");
    insertOrReplace(id, r);
  }

  @Override
  public void insertOrReplace(String id, Document r) throws DBException {
    _insertOrReplace(IdCodec.encode(id), r);
  }

  @Override
  public void insertOrReplace(ByteBuffer id, Document r) throws DBException {
    _insertOrReplace(IdCodec.encode(id), r);
  }

  //@Override
  @Override
  public void insertOrReplace(Value id, Document r) throws DBException {
    _insertOrReplace(IdCodec.encode(id), r);
  }

  // Ensure that the user passed-in rowkey/id matches the one is document
  private void checkDuplicateIds(ByteBuffer id, Document r) {
    Value recIdVal = r.getId();
    ByteBuffer encodedRecId = null;

    if (recIdVal != null) {
      encodedRecId = IdCodec.encode(recIdVal);
      if (!encodedRecId.equals(id)) {
        String errorMsg = null;
        if (recIdVal.getType() == Type.STRING) {
          errorMsg = "Document needs to have '_id' value same as passed id of "+
                      IdCodec.decodeString(id) + " while Document has "+recIdVal.getString();
        } else {
          errorMsg = "Document needs to have '_id' value " +
              "same as the passed in id ";
        }
        throw new IllegalArgumentException(errorMsg);
      }
    }

  }

  private void _insertOrReplace(ByteBuffer inId, Document r) throws DBException {
    Preconditions.checkNotNull(r, "Document being inserted cannot be null");
    checkClosed();
    ByteBuffer id = ByteBufs.ensurePreferred(inId);

    checkDuplicateIds(id, r);

    // convert record to rowcol and pass it down to jni+server, incl flush.
    SerializedFamilyInfo[] info = RowcolCodec.encode(r,
                                                     idPathMap,
                                                     false /*isBulkload*/,
                                                     isPreserveTS()/*useEncoded*/);
    assert(info.length == sortedById().size());
    EncodedBufFamIdInfo ebf = MapRDBTableImplHelper.getEncBufsAndFamilyIds(info);
    MapRPut mput = MapRDBTableImplHelper.toMapRPut(
        id, ebf.familyIds, ebf.encBuffers, PutConstants.TYPE_PUT_ROW);
    try {
      if (isBufferWrite()) {
        maprTable.put(mput);
      } else {
        maprTable.syncPut(mput, /* flush and block = */true);
      }
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "insertOrReplace()");
    }
  }

  private ByteBuffer getKeyFieldsValue(Document r, String fieldAsKey)
      throws DBException {
    if (r == null) {
      throw new IllegalArgumentException("Document cannot be null");
    } else if (fieldAsKey == null) {
      throw new IllegalArgumentException("Requested key cannot be null.");
    }

    Value value = r.getValue(fieldAsKey);
    if (value == null) {
      throw new IllegalArgumentException("Requested key's value cannot be null in the record.");
    }
    return IdCodec.encode(value);
  }

  @Override
  public void insertOrReplace(Document r, FieldPath fieldAsKey)
      throws DBException {
    insertOrReplace(r, fieldAsKey.asPathString());
  }

  @Override
  public void insertOrReplace(Document r, String fieldAsKey) throws DBException {
    ByteBuffer id = getKeyFieldsValue(r, fieldAsKey);

    _insertOrReplace(id, r);
  }

  // One way to error inject for now - set counter at which we sim err.
  @VisibleForTesting
  public int testMulitOpExIdx = 0;

  // Batch info to track parallel operation completion and their error, if any.
  private class AsyncBatchInfo {
    List<FailedOp> failedDocuments;
    volatile int numObj;
    volatile int numCompl;
    BatchingType type;
    String fieldAsKey;

    AsyncBatchInfo(int num, BatchingType opType, String fieldforKey) {
      numObj = num;
      numCompl = 0;
      type = opType;
      failedDocuments = null;
      fieldAsKey = fieldforKey;
    }

    public synchronized void addToFailList(Document rec, Exception e) {
      if (failedDocuments == null) {
        failedDocuments = new ArrayList<FailedOp>();
      }
      failedDocuments.add(new FailedOp(rec, e));
    }

    public synchronized void incrementCompl() {
      numCompl++;

      if (numCompl == numObj) {
        notify();
      }
    }
  }

  // Each object to be async processed from a batch. Has reference to the batch
  // as part of the batch info field.
  class AsyncBatchElement implements Runnable {
    Document rec;
    int recIdx;
    AsyncBatchInfo info;

    AsyncBatchElement(Document inRec, AsyncBatchInfo abi, int myIdx) {
      rec  = inRec;
      info = abi;
      recIdx = myIdx;
    }

    @Override
    public void run() {
      try {
        if ((testMulitOpExIdx != 0) && (recIdx == testMulitOpExIdx)) {
          throw new DBException("Simulated insert error for " + recIdx);
        }

        ByteBuffer useKey = null;
        if (info.fieldAsKey != null) {
          useKey = getKeyFieldsValue(rec, info.fieldAsKey);
        }

        switch (info.type) {
        case INSERTORREPLACE:
          if (useKey == null)
            insertOrReplace(rec);
          else
            _insertOrReplace(useKey, rec);
          break;

        case INSERT:
          if (useKey == null)
            insert(rec);
          else
            _insert(useKey, rec);
          break;

        case REPLACE:
          if (useKey == null)
            replace(rec);
          else
            _replace(useKey, rec);
          break;

        case DELETE:
          if (useKey == null)
            delete(rec);
          else
            _delete(useKey);
          break;

        default:
          throw new IllegalArgumentException("Unsupported batching type " +
                                             info.type);
        }
      } catch (Exception e) {
        info.addToFailList(rec, e);
      } catch (OutOfMemoryError oom) {
        info.addToFailList(rec, new DBException("Out of memory", oom));
      }

      info.incrementCompl();
    }
  }

  // Use the thread pool to send all in parallel and process any errors
  private List<FailedOp> processBatch(List<Document> batch,
                                          BatchingType type,
                                          String fieldAsKey) {
    AsyncBatchInfo abi = new AsyncBatchInfo(batch.size(), type, fieldAsKey);
    for (int i = 0; i < batch.size(); i++) {
      executor.execute(new AsyncBatchElement(batch.get(i), abi, i));
    }

    // wait for all to finish
    try {
      synchronized(abi) {
        while (abi.numCompl < abi.numObj) {
          abi.wait();
        }
      }
    } catch (InterruptedException e) {
      ;
    }

    batch.clear();

    return abi.failedDocuments;
  }

  @Override
  public void insertOrReplace(DocumentStream rs)
  throws MultiOpException {
    String fak = null;
    insertOrReplace(rs, fak);
  }

  @Override
  public void insertOrReplace(DocumentStream rs,
                              FieldPath fieldAsKey) throws MultiOpException {
    insertOrReplace(rs, fieldAsKey.asPathString());
  }

  /**
   * Internally this API will get a number of records from the stream and
   * write them to the table. Make sure the write batch is flushed to
   * server to ensure that the error list can be populated correctly.
   */
  @Override
  public void insertOrReplace(DocumentStream rs,
                              String fieldAsKey) throws MultiOpException {
    Iterator<? extends Document> itrs = rs.iterator();
    List<FailedOp> failedDocuments = null;
    List<Document> batchedDocumentList = new ArrayList<Document>();
    Document readDocument;
    boolean hitWriteError = false;

    while (itrs.hasNext()) {
      readDocument = itrs.next();

      if (readDocument == null) {
        failedDocuments = new ArrayList<FailedOp>();
        failedDocuments.add(new FailedOp(readDocument,
                                         new IllegalArgumentException()));
        if (rs instanceof DBDocumentStream)
          ((DBDocumentStream)rs).makeIteratorNotOpen();
        throw new MultiOpException(failedDocuments);
      }

      // batch it and handle multiple errors in one batch call
      batchedDocumentList.add(readDocument);

      if (batchedDocumentList.size() == JSON_INSERT_BATCH_SIZE) {
        failedDocuments = processBatch(batchedDocumentList,
                                     BatchingType.INSERTORREPLACE, fieldAsKey);
      }

      if ((failedDocuments != null) && failedDocuments.size() != 0) {
        hitWriteError = true;
        break;
      }
    }

    // Finish off any remaining in the batch
    if (!hitWriteError && batchedDocumentList.size() != 0) {
      failedDocuments = processBatch(batchedDocumentList,
                                   BatchingType.INSERTORREPLACE, fieldAsKey);
    }

    if (failedDocuments != null && failedDocuments.size() != 0) {
      if (rs instanceof DBDocumentStream)
        ((DBDocumentStream)rs).makeIteratorNotOpen();
      throw new MultiOpException(failedDocuments);
    }
  }

  /**
   * Applies a row mutation on the row identified by the row id. All the updates
   * specified by the mutation object are applied atomically, meaning either
   * all of the updates in the mutation are applied or none of them are applied.
   *
   * For example, the mutation object specifies an increment on a field
   * "address" by 5 and a set of field "zip" as "95134". On the server, if
   * the "address" field is not of a numeric type then the increment operation
   * will fail. Since the set of field "zip" is also part of same mutation, that
   * operation will also not be applied on the server.
   *
   * @param id row id
   * @param m mutation object specifying the operation on fields of a row
   * @throws TableNotFoundException when the table does not exist to add this row
   * @throws AccessDeniedException user credentials mismatch
   * @throws ReadOnlyException when the table is not accepting writes
   */
  @Override
  public void update(String id, DocumentMutation m) throws DBException {
    _update(IdCodec.encode(id), m);
  }

  @Override
  public void update(ByteBuffer id, DocumentMutation m) throws DBException {
    _update(IdCodec.encode(id), m);
  }

  @Override
  public void update(Value id, DocumentMutation m) throws DBException {
    _update(IdCodec.encode(id), m);
  }

  private void _update(ByteBuffer inId, DocumentMutation m) throws DBException {
    checkClosed();

    ByteBuffer id = ByteBufs.ensurePreferred(inId);

    MutationImpl rmi = ((MutationImpl)m);
    SerializedFamilyInfo[] info = rmi.rowcolSerialize(idPathMap);
    boolean isRMW = rmi.needsReadOnServer(); // is it a read modify write on server

    if (isRMW) {
      MapRUpdateAndGet muag = new MapRUpdateAndGet();
      assert(info.length == idPathMap.size());

      EncodedBufFamIdInfo ebf = MapRDBTableImplHelper.getEncBufsAndFamilyIds(info);
      Map<Integer, List<String>> fieldsMap = rmi.getFieldsNeedRead(idPathMap);
      byte[] serRowConstraint = MapRDBTableImplHelper.fieldPathsToSerRowConstraint(fieldsMap);
      assert(serRowConstraint != null);
      try {
        maprTable.updateRecord(id, ebf.encBuffers, ebf.familyIds,
                               serRowConstraint,
                               isBufferWrite(), muag);
      } catch (IOException e) {
        throw ExceptionHandler.handle(e, "update()");
      }
    } else {
      // just modify/put on the server
      MapRDBTableImplHelper.insertOrReplace(this, id, info);
    }
  }

  /**
   * Deletes a row with the given rowkey. This operation is successful even
   * when the row with the given id doesn't exist.
   *
   * @param id row id
   * @throws TableNotFoundException when a table does not exist to add this row
   * @throws AccessDeniedException on a user credentials mismatch
   * @throws ReadOnlyException when a table is not accepting writes
   */
  @Override
  public void delete(String id) throws DBException {
    _delete(IdCodec.encode(id));
  }

  @Override
  public void delete(ByteBuffer id) throws DBException {
    _delete(IdCodec.encode(id));
  }

  //@Override
  @Override
  public void delete(Value id) throws DBException {
    _delete(IdCodec.encode(id));
  }

  @Override
  public void delete(Document r) throws DBException {
    Preconditions.checkArgument(r != null, "Document being deleted cannot be null");
    Document mdbRec = RowcolCodec.getDBDocument(r);

    Value id = mdbRec.getId();
    Preconditions.checkArgument(id != null, "Document needs to have '_id' in the map as a key.");
    delete(id);
  }

  @Override
  public void delete(Document r, FieldPath fieldAsKey) throws DBException {
    delete(r, fieldAsKey.asPathString());
  }

  @Override
  public void delete(Document r, String fieldAsKey) throws DBException {
    _delete(getKeyFieldsValue(r, fieldAsKey));
  }

  private void _delete(ByteBuffer inId) throws DBException {
    checkClosed();

    int[] famIds = new int[sortedById().size()];
    int i = 0;
    for (Map.Entry<FieldPath, Integer> entry : sortedById()) {
      famIds[i] = entry.getValue();
      i++;
    }

    ByteBuffer id = ByteBufs.ensurePreferred(inId);
    // no rowcol as its a row delete - pass it down to jni+server, incl flush.
    MapRPut mput = MapRDBTableImplHelper.toMapRPut(
        id, famIds, null, PutConstants.TYPE_DELETE_ROW);

    try {
      if (isBufferWrite()) {
        maprTable.put(mput);
      }
      else {
        maprTable.syncPut(mput, /* flush and block = */true);
      }
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "delete()");
    }
  }

  @Override
  public void delete(DocumentStream rs,
                     FieldPath fieldAsKey) throws MultiOpException {
    delete(rs, fieldAsKey.asPathString());
  }

  @Override
  public void delete(DocumentStream rs) throws MultiOpException{
    String fak = null;
    delete(rs, fak);
  }

  @Override
  public void delete(DocumentStream rs,
                     String fieldAsKey) throws MultiOpException {
    Iterator<? extends Document> itrs = rs.iterator();
    List<FailedOp> failedDocuments = new ArrayList<FailedOp>();
    List<Document> batchedDocumentList = new ArrayList<Document>();
    Document readDocument;
    boolean hitDeleteError = false;

    while (itrs.hasNext()) {
      readDocument = itrs.next();

      if (readDocument == null) {
        failedDocuments = new ArrayList<FailedOp>();
        failedDocuments.add(new FailedOp(readDocument,
                                                new IllegalArgumentException()));
        if (rs instanceof DBDocumentStream)
          ((DBDocumentStream)rs).makeIteratorNotOpen();
        throw new MultiOpException(failedDocuments);
      }

      // batch it and handle multiple errors in one batch call
      batchedDocumentList.add(readDocument);

      if (batchedDocumentList.size() == JSON_INSERT_BATCH_SIZE) {
          failedDocuments = processBatch(batchedDocumentList, BatchingType.DELETE,
                                       fieldAsKey);
      }

      if ((failedDocuments != null) && failedDocuments.size() != 0) {
        hitDeleteError = true;
        break;
      }
    }

    // Finish off any remaining in the batch
    if (!hitDeleteError && batchedDocumentList.size() != 0) {
        failedDocuments = processBatch(batchedDocumentList, BatchingType.DELETE,
                                     fieldAsKey);
    }

    if (failedDocuments != null && failedDocuments.size() != 0) {
      if (rs instanceof DBDocumentStream)
        ((DBDocumentStream)rs).makeIteratorNotOpen();
      throw new MultiOpException(failedDocuments);
    }
  }

  /*
   * Update methods
   */

  @Override
  public void increment(String id, String field, long inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(ByteBuffer id, String field, long inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  //@Override
  @Override
  public void increment(Value id, String field, long inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(String id, String field, float inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(ByteBuffer id, String field, float inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  //@Override
  @Override
  public void increment(Value id, String field, float inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(String id, String field, double inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(ByteBuffer id, String field, double inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  //@Override
  @Override
  public void increment(Value id, String field, double inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(String id, String field, BigDecimal inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(ByteBuffer id, String field, BigDecimal inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  //@Override
  @Override
  public void increment(Value id, String field, BigDecimal inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(String id, String field, byte inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(ByteBuffer id, String field, byte inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  //@Override
  @Override
  public void increment(Value id, String field, byte inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(String id, String field, short inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(ByteBuffer id, String field, short inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  //@Override
  @Override
  public void increment(Value id, String field, short inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(String id, String field, int inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void increment(ByteBuffer id, String field, int inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  //@Override
  @Override
  public void increment(Value id, String field, int inc) throws DBException {
    _update(IdCodec.encode(id), new MutationImpl().increment(field, inc));
  }

  @Override
  public void insert(String id, Document r) throws DBException {
    _insert(IdCodec.encode(id), r);
  }

  @Override
  public void insert(ByteBuffer id, Document r) throws DBException {
    _insert(IdCodec.encode(id), r);
  }

  //@Override
  @Override
  public void insert(Value id, Document r) throws DBException {
    _insert(IdCodec.encode(id), r);
  }

  @Override
  public void insert(Document r) throws DBException {
    Document mdbRec;
    if (r instanceof Document) {
      mdbRec = r;
    } else {
      mdbRec = (Document)KeyValueBuilder.initFrom(r);
    }

    Value id = mdbRec.getId();
    if (id == null) {
      throw new IllegalArgumentException("Document needs to have '_id' in the map as a key.");
    }

    insert(id, r);
  }

  private void _insert(ByteBuffer inId, Document r) throws DBException {
    checkClosed();
    ByteBuffer id = ByteBufs.ensurePreferred(inId);
    if (!_checkAndReplace(id, ROW_NOT_EXISTS_CONDITION, r)) {
      throw new DocumentExistsException("A row with key \"" +
          Bytes.toStringBinary(id) + "\" already exist in the table");
    }
   }

  @Override
  public void insert(Document r, FieldPath fieldAsKey) throws DBException {
    insert(r, fieldAsKey.asPathString());
  }

  @Override
  public void insert(Document r, String fieldAsKey) throws DBException {
    _insert(getKeyFieldsValue(r, fieldAsKey), r);
  }

  @Override
  public void insert(DocumentStream rs,
      FieldPath fieldAsKey) throws MultiOpException {
    insert(rs, fieldAsKey.asPathString());
  }

  @Override
  public void insert(DocumentStream rs)
      throws MultiOpException{
    String fak = null;
    insertOrReplace(rs, fak);
  }

  @Override
  public void insert(DocumentStream rs,
                     String fieldAsKey) throws MultiOpException {
    Iterator<? extends Document> itrs = rs.iterator();
    List<FailedOp> failedDocuments = null;
    List<Document> batchedDocumentList = new ArrayList<Document>();
    Document readDocument;
    boolean hitError = false;

    while (itrs.hasNext()) {
      readDocument = itrs.next();

      if (readDocument == null) {
        failedDocuments = new ArrayList<FailedOp>();
        failedDocuments.add(new FailedOp(readDocument,
                                         new IllegalArgumentException()));
        if (rs instanceof DBDocumentStream)
          ((DBDocumentStream)rs).makeIteratorNotOpen();
        throw new MultiOpException(failedDocuments);
      }

      // batch it and handle multiple errors in one batch call
      batchedDocumentList.add(readDocument);

      if (batchedDocumentList.size() == JSON_INSERT_BATCH_SIZE) {
          failedDocuments = processBatch(batchedDocumentList, BatchingType.INSERT,
                                       fieldAsKey);
      }

      if ((failedDocuments != null) && failedDocuments.size() != 0) {
        hitError = true;
        break;
      }
    }

    // Finish off any remaining in the batch
    if (!hitError && batchedDocumentList.size() != 0) {
      failedDocuments = processBatch(batchedDocumentList, BatchingType.INSERT,
                                   fieldAsKey);
    }

    if (failedDocuments != null && failedDocuments.size() != 0) {
      if (rs instanceof DBDocumentStream)
        ((DBDocumentStream)rs).makeIteratorNotOpen();
      throw new MultiOpException(failedDocuments);
    }
  }

  @Override
  public void replace(String id, Document r) throws DBException {
    _replace(IdCodec.encode(id), r);
  }

  @Override
  public void replace(ByteBuffer id, Document r) throws DBException {
    _replace(IdCodec.encode(id), r);
  }

  //@Override
  @Override
  public void replace(Value id, Document r) throws DBException {
    _replace(IdCodec.encode(id), r);
  }

  @Override
  public void replace(Document r) throws DBException {
    Document mdbRec;
    if (r instanceof Document) {
      mdbRec = r;
    } else {
      mdbRec = (Document)KeyValueBuilder.initFrom(r);
    }
    Value id = mdbRec.getId();
    if (id == null) {
      throw new IllegalArgumentException("Document needs to have '_id' in the map as a key.");
    }
    replace(id, r);
  }

  private void _replace(ByteBuffer inId, Document r) throws DBException {
    checkClosed();
    ByteBuffer id = ByteBufs.ensurePreferred(inId);
    if (!_checkAndReplace(id, ROW_EXISTS_CONDITION, r)) {
      throw new DocumentNotFoundException("A row with key \"" +
          Bytes.toStringBinary(id) + "\" does not exist in the table");
    }
  }

  @Override
  public void replace(Document r, FieldPath fieldAsKey) throws DBException {
    replace(r, fieldAsKey.asPathString());
  }

  @Override
  public void replace(Document r, String fieldAsKey) throws DBException {
    ByteBuffer id = getKeyFieldsValue(r, fieldAsKey);

    _replace(id, r);
  }

  @Override
  public void replace(DocumentStream rs, FieldPath fieldAsKey)
      throws MultiOpException {
    replace(rs, fieldAsKey.asPathString());
  }

  @Override
  public void replace(DocumentStream rs) throws MultiOpException{
    String fak = null;
    replace(rs, fak);
  }

  @Override
  public void replace(DocumentStream rs, String fieldAsKey)
      throws MultiOpException {
    Iterator<? extends Document> itrs = rs.iterator();
    List<FailedOp> failedDocuments = null;
    List<Document> batchedDocumentList = new ArrayList<Document>();
    Document readDocument;
    boolean hitError = false;

    while (itrs.hasNext()) {
      readDocument = itrs.next();

      if (readDocument == null) {
        failedDocuments = new ArrayList<FailedOp>();
        failedDocuments.add(new FailedOp(readDocument,
                                           new IllegalArgumentException()));
        if (rs instanceof DBDocumentStream)
           ((DBDocumentStream)rs).makeIteratorNotOpen();
        throw new MultiOpException(failedDocuments);
      }

      // batch it and handle multiple errors in one batch call
      batchedDocumentList.add(readDocument);

      if (batchedDocumentList.size() == JSON_INSERT_BATCH_SIZE) {
        failedDocuments = processBatch(batchedDocumentList, BatchingType.REPLACE,
                                     fieldAsKey);
      }

      if ((failedDocuments != null) && failedDocuments.size() != 0) {
        hitError = true;
        break;
      }
    }

    // Finish off any remaining in the batch
    if (!hitError && batchedDocumentList.size() != 0) {
      failedDocuments = processBatch(batchedDocumentList, BatchingType.REPLACE,
                                   fieldAsKey);
    }

    if (failedDocuments != null && failedDocuments.size() != 0) {
      if (rs instanceof DBDocumentStream)
        ((DBDocumentStream)rs).makeIteratorNotOpen();
      throw new MultiOpException(failedDocuments);
    }
  }

  /**
   * Atomically evaluates the condition on a given row and if the condition
   * holds true for the row then the mutation is applied on the row.
   *
   * If the id doesn't exist, returns false (no exception is thrown).
   * If the mutation operation fails, it throws an exception.
   *
   * @param id row id
   * @param condition the condition to evaluate on the row
   * @param m the mutation to apply on the row
   * @return True if the condition is true for the row, otherwise false
   * @throws DBException
   */
  @Override
  public boolean checkAndMutate(String id,
      QueryCondition condition, DocumentMutation m) throws DBException {
    return _checkAndMutate(IdCodec.encode(id), condition, m);
  }

  @Override
  public boolean checkAndMutate(ByteBuffer id,
      QueryCondition condition, DocumentMutation m) throws DBException {
    return _checkAndMutate(IdCodec.encode(id), condition, m);
  }

  //@Override
  @Override
  public boolean checkAndMutate(Value id,
      QueryCondition condition, DocumentMutation m) throws DBException {
    return _checkAndMutate(IdCodec.encode(id), condition, m);
  }

  private boolean _checkAndMutate(ByteBuffer inId,
      QueryCondition condition, DocumentMutation m) throws DBException {
    if (m == null) {
      throw new NullPointerException("DocumentMutation cannot be null");
    } if (condition == null || condition.isEmpty()) {
      throw new IllegalArgumentException("QueryCondition cannot be null or empty");
    }

    checkClosed();

    ConditionDescriptor condDesc = ((ConditionImpl)condition).getDescriptor(idPathMap);
    ByteBuffer serCond = ByteBufs.ensurePreferred(condDesc.getSerialized());

    MutationImpl rmi = ((MutationImpl)m);
    SerializedFamilyInfo[] info = rmi.rowcolSerialize(idPathMap);
    assert(info.length == idPathMap.size());

    MapRUpdateAndGet muag = new MapRUpdateAndGet();

    EncodedBufFamIdInfo ebf =
        MapRDBTableImplHelper.getEncBufsAndFamilyIds(info);


    byte[] serRowConstraint =
        MapRDBTableImplHelper.fieldPathsToSerRowConstraint(
            MapRDBTableImplHelper.mergeFieldPathList(
                rmi.getFieldsNeedRead(idPathMap),
                MapRDBTableImplHelper.condFieldPathMapToCondFieldPathStrMap(condDesc.getFamilyFieldPathsMap())));

    ByteBuffer id = ByteBufs.ensurePreferred(inId);
    try {
      maprTable.checkAndMutate(id, ebf.encBuffers, ebf.familyIds,
          serRowConstraint, serCond, isBufferWrite(), muag);
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "checkAndMutate()");
    }
    return muag.conditionSuccess;
  }

  /**
   * Atomically evaluates the condition on a given row and if the
   * condition holds true for the row then it is atomically deleted.
   *
   * @param id row id
   * @param condition the condition to evaluate on the row
   * @return True if the condition is true for the row, otherwise false
   * @throws DBException
   */
  @Override
  public boolean checkAndDelete(String id, QueryCondition condition) throws DBException {
    return _checkAndDelete(IdCodec.encode(id), condition);
  }

  @Override
  public boolean checkAndDelete(ByteBuffer id, QueryCondition condition) throws DBException {
    return _checkAndDelete(IdCodec.encode(id), condition);
  }

  //@Override
  @Override
  public boolean checkAndDelete(Value id, QueryCondition condition) throws DBException {
    return _checkAndDelete(IdCodec.encode(id), condition);
  }

  private boolean _checkAndDelete(ByteBuffer inId, QueryCondition condition) throws DBException {
    if (condition == null || condition.isEmpty()) {
      throw new IllegalArgumentException("QueryCondition cannot be null or empty");
    }

    checkClosed();

    ByteBuffer id = ByteBufs.ensurePreferred(inId);

    ConditionDescriptor condDesc = ((ConditionImpl)condition).getDescriptor(idPathMap);
    ByteBuffer serCond= ByteBufs.ensurePreferred(condDesc.getSerialized());

    int[] famIds = new int[idPathMap().size()];
    int i = 0;
    for (Map.Entry<FieldPath, Integer> entry : idPathMap().entrySet()) {
      famIds[i++] = entry.getValue();
    }

    MapRUpdateAndGet muag = new MapRUpdateAndGet();
    byte[] serRowConstraint =
        MapRDBTableImplHelper.fieldPathsToSerRowConstraint(
                MapRDBTableImplHelper.condFieldPathMapToCondFieldPathStrMap(condDesc.getFamilyFieldPathsMap()));

    try {
      maprTable.checkAndReplaceOrDelete(id, null, famIds,
          serRowConstraint,
          serCond, isBufferWrite(), true/*isDelete*/, muag);
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "checkAndDelete");
    }
    return muag.conditionSuccess;
  }

  /**
   * Atomically evaluates the condition on a given row on the server and if the
   * condition holds true for the row then it atomically replaces the row
   * with the given record.
   *
   * @param id row id
   * @param condition the condition to evaluate on the row
   * @param r record to replace
   * @return True if the condition is true for the row, otherwise false
   * @throws DBException
   */

  @Override
  public boolean checkAndReplace(String id,
      QueryCondition condition, Document r) throws DBException {
    return _checkAndReplace(IdCodec.encode(id), condition, r);
  }

  @Override
  public boolean checkAndReplace(ByteBuffer id,
      QueryCondition condition, Document r) throws DBException {
    return _checkAndReplace(IdCodec.encode(id), condition, r);
  }

  @Override
  public boolean checkAndReplace(Value id,
      QueryCondition condition, Document r) throws DBException {
    return _checkAndReplace(IdCodec.encode(id), condition, r);
  }

  private boolean _checkAndReplace(ByteBuffer inId,
      QueryCondition condition, Document r) throws DBException {
    checkClosed();
    Preconditions.checkNotNull(r, "Document being inserted cannot be null");
    Preconditions.checkArgument(condition != null && !condition.isEmpty(), "QueryCondition cannot be null or empty");

    ByteBuffer id = ByteBufs.ensurePreferred(inId);

    ConditionDescriptor condDesc = ((ConditionImpl)condition).getDescriptor(idPathMap);
    ByteBuffer serCond = ByteBufs.ensurePreferred(condDesc.getSerialized());

    SerializedFamilyInfo[] info = RowcolCodec.encode(r, idPathMap);
    assert(info.length == idPathMap.size());

    EncodedBufFamIdInfo ebf =
        MapRDBTableImplHelper.getEncBufsAndFamilyIds(info);

    MapRUpdateAndGet muag = new MapRUpdateAndGet();
    byte[] serRowConstraint =
        MapRDBTableImplHelper.fieldPathsToSerRowConstraint(
                MapRDBTableImplHelper.condFieldPathMapToCondFieldPathStrMap(condDesc.getFamilyFieldPathsMap()));
    try {
      maprTable.checkAndReplaceOrDelete(id, ebf.encBuffers, ebf.familyIds,
          serRowConstraint, serCond, isBufferWrite(), false/*isDelete*/, muag);
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "checkAndDelete");
    }

    return muag.conditionSuccess;
  }

  @Override
  public synchronized void close() throws DBException {
    try {
      if (!closed) {
        maprTable.close();
        executor.shutdownNow();
        closed = true;
      }
    } catch (IOException ioe) {
      throw new DBException(ioe.getMessage(), ioe);
    }
  }

  @Override
  public TabletInfo getTabletInfo(String _id) throws DBException {
    return getTabletInfo(KeyValueBuilder.initFrom(_id));
  }

  @Override
  public TabletInfo getTabletInfo(ByteBuffer _id) throws DBException {
    return getTabletInfo(KeyValueBuilder.initFrom(_id));
  }

  //@Override
  public TabletInfo getTabletInfo(Value _id) throws DBException {
    try {
      MapRTabletScanner scanner = maprTable
          .getTabletScanner(Bytes.getBytes(IdCodec.encode(_id)));
      TabletDesc tablet = scanner.next();
      if (tablet != null) {
        return toTabletInfo(tablet);
      }
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "getTabletInfo()");
    }
    return null;
  }

  @Override
  public TabletInfo[] getTabletInfos() throws DBException {
    List<TabletInfo> tabletInfos = Lists.newArrayList();
    try {
      MapRTabletScanner scanner = maprTable.getTabletScanner();
      List<TabletDesc> nextTabletSet;
      while ((nextTabletSet = scanner.nextSet()) != null) {
        for (TabletDesc tablet : nextTabletSet) {
          tabletInfos.add(toTabletInfo(tablet));
        }
      }
      return tabletInfos.toArray(new TabletInfo[tabletInfos.size()]);
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "getTabletInfos()");
    }
  }

  @Override
  public TabletInfo[] getTabletInfos(QueryCondition condition) throws DBException {
    try {
      return _getTabletInfos(condition);
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "getTabletInfos()");
    }
  }

  private TabletInfo[] _getTabletInfos(QueryCondition condition) throws IOException {
    if (condition == null) {
      return getTabletInfos();
    }

    ConditionImpl cond = (ConditionImpl)condition;
    List<RowkeyRange> rowkeys = cond.getRowkeyRanges();
    boolean noKeyRange = false;

    byte[] startRow = null;
    byte[] endRow = null;

    if (rowkeys == null) {
      noKeyRange = true;
    } else {
      startRow = rowkeys.get(0).getStartRow();
      endRow = rowkeys.get(0).getStopRow();
      if (((startRow == null) && (endRow == null)) ||
          ((startRow.length == 0) && (endRow.length == 0))){
        noKeyRange = true;
      }
    }

    if (noKeyRange) {
      return getTabletInfos();
    }

    List<TabletInfo> tabletInfos = Lists.newArrayList();
    MapRTabletScanner scanner = maprTable.getTabletScanner();
    List<TabletDesc> nextTabletSet;

    int leftRangeComp, rightRangeComp;
    boolean doneScanningTablets = false;

    while ((nextTabletSet = scanner.nextSet()) != null) {
      for (TabletDesc tablet : nextTabletSet) {
        byte[] srow = tablet.getStartKey().toByteArray();
        byte[] erow = tablet.getEndKey().toByteArray();

        if ((srow == null) || (erow == null)) {
          throw new DBException("Missing start and/or endkey in tablet");
        }

        /* rowkey limit is generally provided using an interval with upper and
         * and lower limit. If there is no upper limit, we set leftRangeComp to
         * a negative value. The opposite is applicable to rightRangeComp.
         */
        if ((startRow != null) && (startRow.length != 0)) {
          leftRangeComp = (erow.length == 0) ? -1 :  Bytes.compareTo(startRow, erow);
        } else {
          /* left range is open */
          leftRangeComp = -1;
        }

        if ((endRow != null) && (endRow.length != 0)) {
          rightRangeComp = (srow.length == 0) ? 1 :  Bytes.compareTo(endRow, srow);
        } else {
          rightRangeComp = 1;
        }

        if ((leftRangeComp < 0) && (rightRangeComp < 0)) {
          /* the remaining tablets are not within the row key rage. We can
           * skip these checks.
           */
          doneScanningTablets = true;
          break;
        }

        if (startRow == endRow) {
          if ((leftRangeComp < 0) && (rightRangeComp >= 0)) {
            tabletInfos.add(toTabletInfo(tablet));
          }
        } else {
          if ((leftRangeComp < 0) && (rightRangeComp > 0)) {
            tabletInfos.add(toTabletInfo(tablet));
          }
        }
      }
      if (doneScanningTablets) {
        break;
      }
    }

    return tabletInfos.toArray(new TabletInfo[tabletInfos.size()]);
  }


  private TabletInfo toTabletInfo(TabletDesc tablet) throws IOException {
    int cid = tablet.getFid().getCid();
    String host = maprTable.getServerForCid(cid);
    String[] tokens = host.split(":");
    if (tokens == null || tokens.length != 2) {
      throw new IOException("Bad host information for cid=" + cid + ", host=" + host);
    }

    long estimatedSize = 0;
    long estimatedNumRows = 0;
    TabletStatResponse tsr =
        maprTable.getMapRFS().getTabletStat(tableDesc.getPath(), tablet.getFid());
    if (tsr != null && tsr.hasUsage()) {
      SpaceUsage su = tsr.getUsage();
      estimatedNumRows = su.getNumRows();
      estimatedSize = su.getNumLogicalBlocks() * DEFAULT_BLOCK_SIZE;
    }

    ConditionImpl c = getCondition(
        IdCodec.decode(tablet.getStartKey().asReadOnlyByteBuffer()),
        IdCodec.decode(tablet.getEndKey().asReadOnlyByteBuffer()));

    return new TabletInfoImpl(c, new String[] { tokens[0] }, estimatedSize, estimatedNumRows);
  }

  private ConditionImpl getCondition(Value startValue, Value stopValue) {
    ConditionImpl c = new ConditionImpl();
    if (!isNullValue(startValue) && !isNullValue(stopValue)) {
      c.and();
    }

    if (!isNullValue(startValue)) {
      switch (startValue.getType()) {
      case BINARY:
        c.is(ID_FIELD, Op.GREATER_OR_EQUAL, startValue.getBinary());
        break;
      case STRING:
        c.is(ID_FIELD, Op.GREATER_OR_EQUAL, startValue.getString());
        break;
      default:
        throw new IllegalStateException("Encountered an unsupported type "
            + startValue.getType() + " for _id");
      }
    }

    if (!isNullValue(stopValue)) {
      switch (stopValue.getType()) {
      case BINARY:
        c.is(ID_FIELD, Op.LESS, stopValue.getBinary());
        break;
      case STRING:
        c.is(ID_FIELD, Op.LESS, stopValue.getString());
        break;
      default:
        throw new IllegalStateException("Encountered an unsupported type "
            + stopValue.getType() + " for _id");
      }
    }

    if (!isNullValue(startValue) && !isNullValue(stopValue)) {
      c.close();
    }

    return c.build();
  }

  private boolean isNullValue(Value startValue) {
    return startValue == null || startValue.getType() == Type.NULL;
  }

  /**
   * @return a sorted array of family IDs combining the mutation and condition families
   */
  private int[] mergeFamilyIDs(int[] mutationFamilies, Set<Integer> conditionFamilies) {
    Set<Integer> mergedSet = Sets.newTreeSet(conditionFamilies);
    for (int i : mutationFamilies) {
      mergedSet.add(i);
    }
    return Collections.toIntArray(mergedSet);
  }

  @Override
  public boolean isReadOnly() {
    return false;
  }

  @API.Internal
  public DocumentStream segmentKeyScan()
      throws DBException {
    checkClosed();

    MapRScan maprscan = new MapRScan();
    maprscan.batch = 0;
    maprscan.caching = 0;
    maprscan.startRow = EMPTY_START_ROW;
    maprscan.stopRow = EMPTY_END_ROW;
    maprscan.filter = null;
    maprscan.rowConstraint = MapRDBTableImplHelper.toRowConstraint(this, null);

    KeySamplingFilterProto ksFilter = KeySamplingFilterProto.newBuilder()
        .build();
    FilterMsg filterMsg = FilterMsg.newBuilder()
        .setId(HASH_OF_KEYSAMPLING_FILTER)
        .setSerializedState(ksFilter.toByteString())
        .build();
    byte[] serFilt = Bytes.getBytes(filterMsg.toByteString().asReadOnlyByteBuffer());
    maprscan.setFilter(serFilt);

    try {
      long id = maprTable.getInode().getScanner(maprscan);
      MapRResultScanner scanner = new MapRResultScanner(maprscan, maprTable, id);
      maprTable.addScanner(scanner);

      return new DBDocumentStream(scanner, isExcludeId(), this);
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "segmentKeyScan()");
    }
  }

  // refresh family schema related information and update the table maps
  public void updateSchema() throws DBException {
    List<ColumnFamilyAttr> cfAttrs = null;

    try {
      cfAttrs = maprTable.getMapRFS().listColumnFamily(maprTable.getTablePath(), false);
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "updateSchema()");
    }

    MapRDBTableImplHelper.ComboMap ret = new MapRDBTableImplHelper.ComboMap();
    MapRDBTableImplHelper.getMaps(cfAttrs, ret);
    idToCFNameMap = ret.idToName;
    unSortedIdPathMap = ret.pathToId;
    sortedById = MapRDBTableImplHelper.sortByValueToList(unSortedIdPathMap);
    idPathMap = MapRDBTableImplHelper.sortByValue(unSortedIdPathMap);
    sortedByPathMap = MapRDBTableImplHelper.sortByPath(unSortedIdPathMap);
  }

  // Delete family related information from the table maps
  private void deleteFamilyInfo(String familyName) throws DBException {
    Integer idToDel = null;

    for (Map.Entry<Integer, String> entry : idToCFNameMap.entrySet()) {
      Integer cfId = entry.getKey();
      String  cfName = entry.getValue();
      if (cfName.equals(familyName)) {
        idToDel = cfId;
        break;
      }
    }

    if (idToDel != null) {
      idToCFNameMap.remove(idToDel);
      BiMap<Integer, FieldPath> pathIdMap = idPathMap.inverse();
      unSortedIdPathMap.remove(pathIdMap.get(idToDel));
      sortedById = MapRDBTableImplHelper.sortByValueToList(unSortedIdPathMap);
      idPathMap = MapRDBTableImplHelper.sortByValue(unSortedIdPathMap);
      sortedByPathMap = MapRDBTableImplHelper.sortByPath(unSortedIdPathMap);
    }
  }
}
