/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LikeToRegexConvertor {
  private static final Logger logger = LoggerFactory.getLogger(LikeToRegexConvertor.class);

  /**
   * Regular expression pattern to parse the value operand of the SQL LIKE operator.
   * The tokens could be one of the 3 types.<br/>
   * <ol>
   * <li>Wildcards, i.e. "%" or "_" ==> first regex group ([%_])</li>
   * <li>Character ranges, i.e. "[]" or "[^]" ==> second regex group (\[[^]]*\])</li>
   * <li>Literals ==> third regex group ([^%_\[]+)</li>
   * </ol>
   */
  private static final Pattern SQL_LIKE_REGEX = Pattern.compile("([%_])|(\\[[^]]*\\])|([^%_\\[]+)");

  private static final String SQL_LIKE_ESCAPE_REGEX_STR = "(%s.?)|([%%_])|(\\[[^]]*\\])|([^%%_\\[%s]+)";

  private static final String JAVA_REGEX_SPECIALS = ".()[]{}<>|^-+=*?!$\\";

  private final String likeString_;

  private final String escapeChar_;

  private String regexString_ = null;

  private String prefixString_ = null;

  public LikeToRegexConvertor(String likeString) {
    this(likeString, null);
  }

  public LikeToRegexConvertor(String likeString, Character escapeChar) {
    likeString_ = likeString;
    if (escapeChar == null) {
      escapeChar_ = null;
    } else {
      escapeChar_ = JAVA_REGEX_SPECIALS.indexOf(escapeChar) == -1
          ? String.valueOf(escapeChar) : ("\\" + escapeChar);
    }
  }

  /**
   * Convert a SQL LIKE operator Value to a Regular Expression.
   */
  public LikeToRegexConvertor parse() {
    if (regexString_ != null) {
      return this;
    }

    Matcher matcher = null;
    StringBuilder prefixSB = new StringBuilder();
    StringBuilder regexSB = new StringBuilder("^"); // starts with
    if (escapeChar_ == null) {
      matcher = SQL_LIKE_REGEX.matcher(likeString_);
    } else {
      /*
       * When an escape character is specified, add another capturing group
       * with the escape character in the front for the escape sequence and
       * add the escape character to the exclusion list of literals
       */
      matcher = Pattern.compile(
          String.format(SQL_LIKE_ESCAPE_REGEX_STR, escapeChar_, escapeChar_))
        .matcher(likeString_);
    }
    String fragment = null;
    boolean literalsSoFar = true;
    while (matcher.find()) {
      if (escapeChar_ != null && matcher.group(1) != null) {
        fragment = matcher.group(1);
        if (fragment.length() != 2) {
          throw new IllegalArgumentException("Invalid fragment '"
              + fragment + "' at index " + matcher.start()
              + " in the LIKE operand '" + likeString_ + "'");
        }
        String escapedChar = fragment.substring(1);
        if (literalsSoFar) {
          prefixSB.append(escapedChar);
        }
        regexSB.append(Pattern.quote(escapedChar));
      } else {
        fragment = matcher.group();
        switch (fragment) {
        case "_": // LIKE('_') => REGEX('.')
          literalsSoFar = false;
          regexSB.append(".");
          break;
        case "%": // LIKE('%') => REGEX('.*')
          literalsSoFar = false;
          regexSB.append(".*");
          break;
        default: // ALL other including character ranges
          if (fragment.startsWith("[") && fragment.endsWith("]")) {
            literalsSoFar = false;
            regexSB.append(fragment);
          } else {
            if (literalsSoFar) {
              prefixSB.append(fragment);
            }
            // found literal, just quote it.
            regexSB.append(Pattern.quote(fragment));
          }
          break;
        }
      }
    }
    prefixString_ = prefixSB.toString();
    regexString_ = regexSB.append('$').toString();

    logger.debug("Converted LIKE string '{}' to REGEX string '{}'.", likeString_, regexString_);
    return this;
  }

  public String getRegexString() {
    return regexString_;
  }

  public String getPrefixString() {
    return prefixString_;
  }

  public String getLikeString() {
    return likeString_;
  }

}
