/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import static com.mapr.db.impl.Constants.MAX_ROW_KEY_SIZE;
import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;

import java.nio.ByteBuffer;
import java.util.Arrays;

import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.annotation.API;
import org.ojai.exceptions.DecodingException;
import org.ojai.exceptions.EncodingException;

import com.mapr.db.util.ByteBufs;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

@API.Internal
public class IdCodec {

  private static final boolean TYPE_ENABLED_FOR_ID = true;

  private static final ByteBuffer EMPTY_BYTE_BUFFER = ByteBufs.allocatePreferred(0);
  private static final byte[] EMPTY_BYTE_ARRAY = new byte[0];
  private static final String EMPTY_STRING = "";

  /**
   * @param type the Type to test
   * @return {@code true} if the specified type is a supported Type for '_id' field,
   *         otherwise, {@code false}.
   */
  public static boolean isSupportedType(Type type) {
    switch (type) {
    case BINARY:
    case STRING:
      return true;
    default:
      return false;
    }
  }

  /**
   * Returns the String representation of the Value.
   * @param in
   * @return
   */
  public static String asString(Value in) {
    if (in != null) {
      switch (in.getType()) {
      case STRING:
        return in.getString();
      case BINARY:
        return Bytes.toStringBinary(in.getBinary());
      default:
        throw new EncodingException("Id of type "
            + in.getType() + " is currently not supported.");
      }
    }
    return String.valueOf(in);
  }

  /**
   * Encode a binary encoded string using the _id encoding scheme.
   *
   * @param in the binary encoded string to encode
   * @return the byte array containing the encoded Value
   */
  public static byte[] encodeStringBinary(String in) {
      return Bytes.toBytesBinary(TYPE_ENABLED_FOR_ID
          ? String.format("\\x%02d", Type.STRING.getCode()) + in : in);
  }

  /**
   * Encodes a Value into a ByteBuffer using the _id encoding scheme.
   *
   * @param in the Value to encode
   * @return the ByteBuffer containing the encoded Value
   * @throws EncodingException if the input Value is not a supported type
   */
  public static ByteBuffer encode(Value in) {
    if (in != null) {
      switch (in.getType()) {
      case NULL:
        return EMPTY_BYTE_BUFFER;
      case STRING:
        return encode(in.getString());
      case BINARY:
        return encode(in.getBinary());
      default:
        throw new EncodingException("Encoding of type "
            + in.getType() + " is currently not supported.");
      }
    }
    return null;
  }

  /**
   * Encodes a Value into a byte array using the _id encoding scheme.
   *
   * @param in the Value to encode
   * @return the byte array containing the encoded Value
   * @throws EncodingException if the input Value is not a supported type
   */
  public static byte[] encodeAsBytes(Value in) {
    if (in != null) {
      switch (in.getType()) {
      case NULL:
        return EMPTY_BYTE_ARRAY;
      case STRING:
        return encodeAsBytes(in.getString());
      case BINARY:
        return encodeAsBytes(in.getBinary());
      default:
        throw new EncodingException("Encoding of type "
            + in.getType() + " is currently not supported.");
      }
    }
    return null;
  }

  /**
   * Decodes a Value from the provided byte array. If the byte array is of Zero length,
   * a NULL type Value is returned.
   *
   * @param in byte array containing the encoded Value
   * @return decoded Value
   * @throws DecodingException if the input byte array does not contain a supported type
   */
  public static Value decode(byte[] in) {
    if (in != null && in.length > 0) {
      if (!TYPE_ENABLED_FOR_ID) {
        return KeyValueBuilder.initFrom(ByteBufs.wrap(decodeBytes(in)));
      } else {
        Type type = getType(in[0]);
        switch (type) {
        case STRING:
          return KeyValueBuilder.initFrom(decodeString(in));
        case BINARY:
          return KeyValueBuilder.initFrom(ByteBufs.wrap(decodeBytes(in)));
        default:
          assert false;
        }
      }
    }
    return in == null ? null : KeyValueBuilder.initFromNull();
  }

  /**
   * Decodes a Value from the provided ByteBuffer. If the ByteBuffer has zero remaining
   * bytes, a NULL type Value is returned.
   *
   * @param in ByteBuffer containing the encoded Value
   * @return decoded Value
   * @throws DecodingException if the input ByteBuffer does not contain a supported type
   */
  public static Value decode(ByteBuffer in) {
    if (in != null && in.remaining() > 0) {
      if (!TYPE_ENABLED_FOR_ID) {
        return KeyValueBuilder.initFrom(decodeBinary(in));
      } else {
        in = (ByteBuffer) in.slice().mark();
        Type type = getType(in.get());
        in.reset();
        switch (type) {
        case STRING:
          return KeyValueBuilder.initFrom(decodeString(in));
        case BINARY:
          return KeyValueBuilder.initFrom(decodeBinary(in));
        default:
          assert false;
        }
      }
    }
    return in == null ? null : KeyValueBuilder.initFromNull();
  }

  /**
   * Encodes a String into a ByteBuffer using the _id encoding scheme.
   *
   * @param in the String to encode
   * @return the ByteBuffer containing the encoded String
   */
  public static ByteBuffer encode(String in) {
    if (in != null) {
      if (!TYPE_ENABLED_FOR_ID) {
        ByteBuffer buf = Bytes.toByteBuffer(in);
        checkMaxRowKeySize(buf.remaining());
        return buf;
      } else {
        byte[] encodedBytes = Bytes.toBytes(in);
        int rowKeySize = encodedBytes.length + 1 /*type*/;
        checkMaxRowKeySize(rowKeySize);
        return (ByteBuffer) ByteBufs
            .allocatePreferred(rowKeySize)
            .put(Type.STRING.getCode())
            .put(encodedBytes)
            .flip();
      }
    }
    return null;
  }

  /**
   * Encodes a String into a ByteBuffer using the _id encoding scheme.
   *
   * @param in the String to encode
   * @return the ByteBuffer containing the encoded String
   */
  public static byte[] encodeAsBytes(String in) {
    if (in != null) {
      if (!TYPE_ENABLED_FOR_ID) {
        byte[] bytes = Bytes.toBytes(in);
        checkMaxRowKeySize(bytes.length);
        return bytes;
      } else {
        byte[] encodedBytes = Bytes.toBytes(in);
        int rowKeySize = encodedBytes.length + 1 /*type*/;
        checkMaxRowKeySize(rowKeySize);
        byte[] retVal = new byte[rowKeySize];
        retVal[0] = Type.STRING.getCode();
        System.arraycopy(encodedBytes, 0, retVal, 1, encodedBytes.length);
        return retVal;
      }
    }
    return null;
  }

  /**
   * Decodes a STRING from the provided byte array.
   *
   * @param in byte array containing the encoded String
   * @return the decoded String
   * @throws DecodingException if the input byte array does not contain a STRING value
   */
  public static String decodeString(byte[] in) {
    if (in != null && in.length > 0) {
      if (!TYPE_ENABLED_FOR_ID) {
        return Bytes.toString(in);
      } else {
        checkType(in[0], Type.STRING);
        return Bytes.toString(in, 1, in.length-1);
      }
    }
    return in == null ? null : EMPTY_STRING;
  }

  /**
   * Decodes a STRING value from the provided ByteBuffer.
   *
   * @param in ByteBuffer containing the encoded String
   * @return the decoded String
   * @throws DecodingException if the input ByteBuffer does not contain a STRING value
   */
  public static String decodeString(ByteBuffer in) {
    if (in != null) {
      in = in.slice();
      if (TYPE_ENABLED_FOR_ID) {
        checkType(in.get(), Type.STRING);
      }
      return Bytes.toString(in);
    }
    return null;
  }

  /**
   * Encodes a BINARY value into a byte array using the _id encoding scheme.
   *
   * @param in the byte array containing the BINARY data
   * @return the byte array containing the encoded BINARY value
   */
  public static byte[] encodeAsBytes(byte[] in) {
    if (in != null) {
      if (!TYPE_ENABLED_FOR_ID) {
        checkMaxRowKeySize(in.length);
        return Arrays.copyOf(in, in.length);
      } else {
        checkMaxRowKeySize(in.length+1);
        byte[] retVal = new byte[in.length+1];
        retVal[0] = Type.BINARY.getCode();
        System.arraycopy(in, 0, retVal, 1, in.length);
        return retVal;
      }
    }
    return null;
  }

  /**
   * Decodes a BINARY value from the provided byte array.
   *
   * @param in byte array containing the encoded BINARY Value
   * @return a byte array containing BINARY value
   * @throws DecodingException if the input byte array does not contain a BINARY value
   */
  public static byte[] decodeBytes(byte[] in) {
    if (in != null && in.length > 0) {
      if (!TYPE_ENABLED_FOR_ID) {
        return Arrays.copyOf(in, in.length);
      } else {
        checkType(in[0], Type.BINARY);
        byte[] retVal = new byte[in.length-1];
        System.arraycopy(in, 1, retVal, 0, retVal.length);
        return retVal;
      }
    }
    return in == null ? null : EMPTY_BYTE_ARRAY;
  }

  /**
   * Encodes a BINARY value into a ByteBuffer using the _id encoding scheme.
   *
   * @param in the ByteBuffer containing the BINARY data
   * @return the ByteBuffer containing the encoded BINARY value
   */
  public static ByteBuffer encode(ByteBuffer in) {
    if (in != null) {
      if (!TYPE_ENABLED_FOR_ID) {
        checkMaxRowKeySize(in.remaining());
        return (ByteBuffer) ByteBufs
            .allocatePreferred(in.remaining())
            .put(in.slice())
            .flip();
      } else {
        int rowKeySize = in.remaining() + 1 /*type*/;
        checkMaxRowKeySize(rowKeySize);
        return (ByteBuffer) ByteBufs
            .allocatePreferred(rowKeySize)
            .put(Type.BINARY.getCode())
            .put(in.slice())
            .flip();
      }
    }
    return null;
  }

  /**
   * Encodes a BINARY value into a byte array using the _id encoding scheme.
   *
   * @param in the ByteBuffer containing the BINARY data
   * @return the byte array containing the encoded BINARY value
   */
  public static byte[] encodeAsBytes(ByteBuffer in) {
    if (in != null) {
      return Bytes.toBytes(encode(in));
    }
    return null;
  }

  /**
   * Decodes a BINARY value from the provided ByteBuffer. The returned ByteBuffer does
   * not share its content with the input ByteBuffer.
   *
   * @param in ByteBuffer containing the encoded BINARY Value
   * @return a ByteBuffer containing BINARY value
   * @throws DecodingException if the input ByteBuffer does not contain a BINARY value.
   */
  public static ByteBuffer decodeBinary(ByteBuffer in) {
    if (in != null && in.remaining() > 0) {
      in = in.slice();
      if (TYPE_ENABLED_FOR_ID) {
        checkType(in.get(), Type.BINARY);
      }
      return (ByteBuffer) ByteBufs
          .allocatePreferred(in.remaining())
          .put(in)
          .flip();
    }
    return in == null ? null : EMPTY_BYTE_BUFFER;
  }

  private static Type getType(byte typeCode) {
    Type type = Type.valueOf(typeCode);
    if (type == null) {
      throw new DecodingException("Encountered unknown type code: " + typeCode);
    } else if (!isSupportedType(type)) {
      throw new DecodingException("Decoding of type " + type + " is currently not supported.");
    }
    return type;
  }

  private static void checkType(byte code, Type type) {
    if (code  != type.getCode()) {
      throw new DecodingException(
          String.format("Expected type %d (%s), found %d while decoding.",
              type.getCode(), type, code));
    }
  }

  private static void checkMaxRowKeySize(int rowKeySize) {
    if (rowKeySize > MAX_ROW_KEY_SIZE) {
      throw new IllegalArgumentException(String.format(
          "The encoded size of _id field (%d) is greater than maximum allowed size (%d).",
          rowKeySize, MAX_ROW_KEY_SIZE));
    }
  }

}
