/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import static com.mapr.db.impl.Constants.DEFAULT_FAMILY;

import java.util.Iterator;

import org.ojai.FieldPath;
import org.ojai.FieldSegment;
import org.ojai.annotation.API;

import com.google.common.base.Preconditions;
import com.mapr.db.FamilyDescriptor;
import com.mapr.fs.proto.Common.FileCompressionType;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.SchemaFamily;

@API.Internal
public class FamilyDescriptorImpl implements FamilyDescriptor {

  private SchemaFamily.Builder schemaFamily_;
  private FieldPath jsonFamilyPath_;

  public FamilyDescriptorImpl() {
    this(defaultSchemaFamily(), FieldPath.EMPTY);
  }

  public FamilyDescriptorImpl(String familyName, String jsonPath) {
    this(defaultSchemaFamily().setName(familyName),
        FieldPath.parseFrom(jsonPath));
  }

  public FamilyDescriptorImpl(String familyName, FieldPath jsonPath) {
    this(defaultSchemaFamily().setName(familyName), jsonPath);
  }

  FamilyDescriptorImpl(ColumnFamilyAttr cfAttr) {
    this(cfAttr.getSchFamily().toBuilder(),
        FieldPath.parseFrom(cfAttr.getJsonFamilyPath()));
  }

  FamilyDescriptorImpl(FamilyDescriptorImpl other) {
    this(other.schemaFamily_.clone(), other.jsonFamilyPath_);
  }

  FamilyDescriptorImpl(SchemaFamily.Builder schemaFamily,
      FieldPath jsonFamilyPath) {
    Preconditions.checkNotNull(schemaFamily, "SchemaFamily must be initalized to a non-null value");
    Preconditions.checkNotNull(jsonFamilyPath, "FieldPath must be initalized to a non-null value");
    schemaFamily_ = schemaFamily;
    setJsonFieldPath(jsonFamilyPath);
  }

  /**
   * Read-only
   */
  public int getId() {
    return schemaFamily_.getId();
  }

  public SchemaFamily.Builder getSchemaFamily() {
    return schemaFamily_.clone();
  }

  @Override
  public boolean hasName() {
    return schemaFamily_.hasName();
  }

  @Override
  public String getName() {
    return schemaFamily_.getName();
  }

  @Override
  public FamilyDescriptor setName(String name) {
    schemaFamily_.setName(name);
    return this;
  }

  @Override
  public FieldPath getJsonFieldPath() {
    return jsonFamilyPath_;
  }

  @Override
  public FamilyDescriptor setJsonFieldPath(String fieldPath) {
    return setJsonFieldPath(FieldPath.parseFrom(fieldPath));
  }

  @Override
  public FamilyDescriptor setJsonFieldPath(FieldPath fieldPath) {
    validateFamilyPath(fieldPath);
    jsonFamilyPath_ = fieldPath;
    return this;
  }

  @Override
  public long getTTL() {
    return (schemaFamily_.getTtl() / 1000) /*ms to sec*/;
  }

  @Override
  public FamilyDescriptor setTTL(long ttl) {
    schemaFamily_.setTtl((ttl * 1000) /*sec to ms*/);
    return this;
  }

  @Override
  public boolean isInMemory() {
    return schemaFamily_.getInMemory();
  }

  @Override
  public FamilyDescriptor setInMemory(boolean inMemory) {
    schemaFamily_.setInMemory(inMemory);
    return this;
  }

  @Override
  public Compression getCompression() {
    return schemaFamily_.hasCompression() ?
        proto2Compression(schemaFamily_.getCompression()) : Compression.Inherited;
  }

  @Override
  public FamilyDescriptor setCompression(Compression compression) {
    FileCompressionType proto = compression2Proto(compression);
    if (proto != null) {
      schemaFamily_.setCompression(proto);
    } else {
      schemaFamily_.clearCompression();
    }
    return this;
  }

  private FileCompressionType compression2Proto(Compression compression) {
    switch (compression) {
    case Inherited:
      return null;
    case LZ4:
      return FileCompressionType.FCT_LZ4;
    case LZF:
      return FileCompressionType.FCT_LZF;
    case ZLIB:
      return FileCompressionType.FCT_ZLIB;
    case None:
    default:
      return FileCompressionType.FCT_OFF;
    }
  }

  private Compression proto2Compression(FileCompressionType compression) {
    switch(compression) {
    case FCT_LZ4:
      return Compression.LZ4;
    case FCT_LZF:
    case FCT_OLDLZF:
      return Compression.LZF;
    case FCT_ZLIB:
      return Compression.ZLIB;
    case FCT_OFF:
    default:
      return Compression.None;
    }
  }

  private void validateFamilyPath(FieldPath fieldPath) {
    Iterator<FieldSegment> iter = fieldPath.iterator();
    while (iter.hasNext()) {
      FieldSegment seg = iter.next();
      if (seg.isArray()) {
        throw new IllegalArgumentException(
            "A column family FieldPath cannot have array segments.");
      }
    }
  }

  public static final ColumnFamilyAttr DEFAULT_COLUMN_FAMILY_ATTR;
  static {
    SchemaFamily.Builder schemaFamily = defaultSchemaFamily()
        .setName(DEFAULT_FAMILY);
    DEFAULT_COLUMN_FAMILY_ATTR = ColumnFamilyAttr.newBuilder()
        .setSchFamily(schemaFamily)
        .build();
  }

  private static SchemaFamily.Builder defaultSchemaFamily() {
    return SchemaFamily.newBuilder()
        .setInMemory(false)
        .setMinVersions(0)
        .setMaxVersions(1)
        .setCompression(FileCompressionType.FCT_LZ4);
  }

  @Override
  public String toString() {
    return "{\"Id\": " + getId()
        + ", \"Name\": \"" + getName()
        + "\", \"JsonFieldPath\": \"" + getJsonFieldPath()
        + "\", \"TTL\": " + getTTL()
        + ", \"InMemory\": " + isInMemory()
        + ", \"Compression\": \"" + getCompression() + "\"}";
  }

  @Override
  public FamilyDescriptorImpl clone() {
    return new FamilyDescriptorImpl(this);
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + jsonFamilyPath_.hashCode();
    result = prime * result + (int)getTTL();
    result = prime * result + (isInMemory() ? 1 : 0);
    result = prime * result + ((getName() == null) ? 0 : getName().hashCode());
    result = prime * result + ((getCompression() == null) ? 0 : getCompression().ordinal());

    return result;
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    FamilyDescriptorImpl other = (FamilyDescriptorImpl) obj;
    if (jsonFamilyPath_ == null) {
      if (other.jsonFamilyPath_ != null)
        return false;
    } else if (!jsonFamilyPath_.equals(other.jsonFamilyPath_)) {
      return false;
    }
    if (getName() == null) {
      if (other.getName() != null)
        return false;
    } else if (!getName().equals(other.getName())) {
      return false;
    }
    if (getCompression() == null) {
      if (other.getCompression() != null)
        return false;
    } else if (!getCompression().equals(other.getCompression())) {
      return false;
    }
    if (isInMemory() != other.isInMemory()) {
      return false;
    }
    if (getTTL() != other.getTTL()) {
      return false;
    }
    return true;
  }

}
