/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.ojai.FieldPath;
import org.ojai.annotation.API;
import org.ojai.store.QueryCondition;

import com.google.common.collect.BiMap;
import com.google.common.collect.EnumBiMap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import com.mapr.fs.jni.MapRConstants;
import com.mapr.fs.proto.Dbfilters.CompareOpProto;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

@API.Internal
public abstract class ConditionNode implements Cloneable {

  /**
   * ConditionImpl.class.getName().hashCode()
   */
  static final String HASH_OF_CONDITION_FILTER    = "8cbdcd12";

  /**
   * org.apache.hadoop.hbase.filter.RowFilter.class.getName().hashCode()
   */
  static final String HASH_OF_ROW_FILTER          = "469dbd04";

  /**
   * org.apache.hadoop.hbase.filter.AlwaysFalseFilter.class.getName().hashCode()
   */
  static final String HASH_OF_ALWAYSFALSE_FILTER  = "d71875e1";

  /**
   * org.apache.hadoop.hbase.filter.KeySamplingFilter.class.getName().hashCode()
   */
  static final String HASH_OF_KEYSAMPLING_FILTER         = "3427db73";

  /**
   * org.apache.hadoop.hbase.filter.FilterList.class.getName().hashCode()
   */
  static final String HASH_OF_FILTER_LIST         = "a42ebf64";

  /**
   * org.apache.hadoop.hbase.filter.NullComparator.class.getName().hashCode()
   */
  static final String HASH_OF_NULL_COMPARATOR     = "8543f5eb";

  /**
   * org.apache.hadoop.hbase.filter.BinaryComparator.class.getName().hashCode()
   */
  static final String HASH_OF_BINARY_COMPARATOR   = "05f39865";

  /**
   * com.mapr.db.rowcol.DBDocument.class.getName().hashCode()
   */
  static final String HASH_OF_JSON_COMPARATOR     = "45a94888";

  /**
   * org.apache.hadoop.hbase.filter.RegexStringComparator.class.getName().hashCode()
   */
  static final String HASH_OF_REGEX_COMPARATOR    = "e2d7ba40";

  /**
   * com.mapr.fs.proto.Dbfilters.TypeComparatorProto.class.getName().hashCode()
   */
  static final String HASH_OF_TYPE_COMPARATOR     = "1e95fd6b";

  /**
   * com.mapr.fs.proto.Dbfilters.SizeComparatorProto.class.getName().hashCode()
   */
  static final String HASH_OF_SIZE_COMPARATOR     = "2e7025c4";

  /**
   * Used as the dummy key when we encode the value operand of a condition into
   * RowCol format.
   */
  static final String DUMMY_FIELD_X = "X";

  static final String EMPTY_STR = "";

  static final Set<Integer> EMPTY_FIELD_IDS = ImmutableSet.of();

  static final Set<FieldPath> EMPTY_FIELD_PATHS = ImmutableSet.of();

  boolean isLeaf() {
    return true;
  }

  StringBuilder treeBuilder(StringBuilder sb) {
    return treeBuilder(indent(sb, 2), 2);
  }

  abstract ConditionDescriptor getDescriptor(BiMap<FieldPath, Integer> pathIdMap);
  abstract StringBuilder expressionBuilder(StringBuilder sb);
  abstract StringBuilder treeBuilder(StringBuilder sb, int level);
  abstract StringBuilder jsonBuilder(StringBuilder sb);
  abstract List<RowkeyRange> getRowkeyRanges();
  abstract void addProjections(Set<FieldPath> projections);

  protected ConditionNode clone() {
    try {
      return (ConditionNode) super.clone();
    } catch (CloneNotSupportedException e) {
      assert false : "Clone failed!!!";
      return null;
    }
  }

  /**
   * Asks the condition node to recursively inspect itself or its children and
   * remove the nodes that would not alter the result of the scan.
   *
   * For example: Consider a conjunction on "_id" field, say<br/><br/>{@code
   *
   *    _id >= "efg" AND _id < "pqr"
   *
   * }<br/><br/>This can be converted to a range scan over the table with key
   * range ["efg", "pqr") and evaluating the condition will have no effect on
   * the ultimate result.
   *
   * @return {@code true} if this condition node can be pruned or
   *         removed from the parent node
   */
  abstract boolean checkAndPrune();

  static final char DOLLAR_CHAR = '$';
  static final char COLON_CHAR = ':';
  static final char QUOTE_CHAR = '"';
  static final char COMMA_CHAR = ',';
  static final char SPACE_CHAR = ' ';
  static final char NEWLINE_CHAR = '\n';
  static final char OPEN_BRACKET = '[';
  static final char CLOSE_BRACKET = ']';
  static final char OPEN_PARAN = '(';
  static final char CLOSE_PARAN = ')';
  static final char OPEN_MAP = '{';
  static final char CLOSE_MAP = '}';

  boolean isEmpty() {
    return false;
  }

  static StringBuilder indent(StringBuilder sb, int indent) {
    for (int i = 0; i < indent; i++) {
      sb.append(SPACE_CHAR);
    }
    return sb;
  }

  /**
   * The user visible ENUM has a limited number of operators to prevent accidental
   * incorrect usage.<br>For example, <pre>where("a.b.c", MATCHES, false)</pre>
   */
  static final BiMap<QueryCondition.Op, CompareOpProto> opProtoMap =
      EnumBiMap.create(QueryCondition.Op.class, CompareOpProto.class);
  static {
    opProtoMap.put(QueryCondition.Op.LESS, CompareOpProto.LESS);
    opProtoMap.put(QueryCondition.Op.LESS_OR_EQUAL, CompareOpProto.LESS_OR_EQUAL);
    opProtoMap.put(QueryCondition.Op.EQUAL, CompareOpProto.EQUAL);
    opProtoMap.put(QueryCondition.Op.NOT_EQUAL, CompareOpProto.NOT_EQUAL);
    opProtoMap.put(QueryCondition.Op.GREATER_OR_EQUAL, CompareOpProto.GREATER_OR_EQUAL);
    opProtoMap.put(QueryCondition.Op.GREATER, CompareOpProto.GREATER);
    assert opProtoMap.size() == QueryCondition.Op.values().length
        : "Map is missing some of the QueryCondition.Op elements";
  }

  /**
   * Map from QueryCondition operators to their symbolic string.
   */
  static final Map<CompareOpProto, String> opSymbolMap =
      Maps.newEnumMap(CompareOpProto.class);
  static {
    opSymbolMap.put(CompareOpProto.NO_OP, "!ERR!");
    opSymbolMap.put(CompareOpProto.LESS, "<");
    opSymbolMap.put(CompareOpProto.LESS_OR_EQUAL, "<=");
    opSymbolMap.put(CompareOpProto.EQUAL, "=");
    opSymbolMap.put(CompareOpProto.NOT_EQUAL, "!=");
    opSymbolMap.put(CompareOpProto.GREATER_OR_EQUAL, ">=");
    opSymbolMap.put(CompareOpProto.GREATER, ">");
//    opSymbolMap.put(CompareOpProto.CND_IN, "IN");
//    opSymbolMap.put(CompareOpProto.CND_NOT_IN, "NOT_IN");
//    opSymbolMap.put(CompareOpProto.CND_MATCHES, "MATCHES");
//    opSymbolMap.put(CompareOpProto.CND_NOT_MATCHES, "NOT_MATCHES");
//    opSymbolMap.put(CompareOpProto.CND_TYPE_OF, "TYPE_OF");
//    opSymbolMap.put(CompareOpProto.CND_NOT_TYPE_OF, "NOT_TYPE_OF");
    assert opSymbolMap.size() == CompareOpProto.values().length
        : "Map is missing some of the CompareOpProto elements";
  }

  /**
   * Map from QueryCondition operators to their symbolic string.
   */
  static final Map<CompareOpProto, String> opJsonOpMap =
      Maps.newEnumMap(CompareOpProto.class);
//  static {
//    opJsonOpMap.put(ConditionOpProto.CND_UNKNOWN, "!ERR!");
//    opJsonOpMap.put(ConditionOpProto.CND_LT, "$lt");
//    opJsonOpMap.put(ConditionOpProto.CND_LE, "$le");
//    opJsonOpMap.put(ConditionOpProto.CND_EQ, "$eq");
//    opJsonOpMap.put(ConditionOpProto.CND_NE, "$ne");
//    opJsonOpMap.put(ConditionOpProto.CND_GE, "$ge");
//    opJsonOpMap.put(ConditionOpProto.CND_GT, "$gt");
//    opJsonOpMap.put(ConditionOpProto.CND_IN, "$in");
//    opJsonOpMap.put(ConditionOpProto.CND_NOT_IN, "$nin");
//    opJsonOpMap.put(ConditionOpProto.CND_MATCHES, "$matches");
//    opJsonOpMap.put(ConditionOpProto.CND_NOT_MATCHES, "$notmatches");
//    opJsonOpMap.put(ConditionOpProto.CND_TYPE_OF, "$typeof");
//    opJsonOpMap.put(ConditionOpProto.CND_NOT_TYPE_OF, "$nottypeof");
//    assert opJsonOpMap.size() == ConditionOpProto.values().length
//        : "Map is missing some of the ConditionOpProto elements";
//  }

  static final List<RowkeyRange> FULL_TABLE_RANGE = ImmutableList.of(
      new RowkeyRange(MapRConstants.EMPTY_BYTE_ARRAY, MapRConstants.EMPTY_BYTE_ARRAY));
  public static class RowkeyRange {
    final protected byte[] startRow;
    final protected byte[] stopRow;

    public RowkeyRange(byte[] startRow, byte[] stopRow) {
      this.startRow = startRow;
      this.stopRow = stopRow;
    }

    public byte[] getStartRow() {
      return startRow == null ? MapRConstants.EMPTY_BYTE_ARRAY : startRow;
    }

    public byte[] getStopRow() {
      return stopRow == null ? MapRConstants.EMPTY_BYTE_ARRAY : stopRow;
    }

    @Override
    public String toString() {
      return "RowkeyRange [startRow=" + Bytes.toStringBinary(startRow)
          + ", stopRow=" + Bytes.toStringBinary(stopRow) + "]";
    }

    @Override
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + Arrays.hashCode(startRow);
      result = prime * result + Arrays.hashCode(stopRow);
      return result;
    }

    @Override
    public boolean equals(Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      RowkeyRange other = (RowkeyRange) obj;
      if (!Arrays.equals(startRow, other.startRow))
        return false;
      if (!Arrays.equals(stopRow, other.stopRow))
        return false;
      return true;
    }
  }

}
