/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import static com.mapr.db.impl.Constants.DEFAULT_FAMILY_MAP;
import static com.mapr.db.impl.Constants.DEFAULT_NAME_MAP;
import static com.mapr.db.impl.Constants.DUMMY_FIELDPATH_V;
import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;
import static com.mapr.fs.proto.Dbfilters.ComparatorModeProto.CMP_PATTERN;
import static com.mapr.fs.proto.Dbfilters.ComparatorModeProto.CMP_VALUE;
import static com.mapr.fs.proto.Dbfilters.CompareOpProto.EQUAL;
import static org.ojai.DocumentConstants.ID_FIELD;
import static org.ojai.DocumentConstants.ID_KEY;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Pattern;

import org.ojai.Document;
import org.ojai.FieldPath;
import org.ojai.FieldSegment;
import org.ojai.Value;
import org.ojai.annotation.API;
import org.ojai.store.QueryCondition;
import org.ojai.util.Values;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.protobuf.ByteString;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.KeyValue;
import com.mapr.db.rowcol.RowcolCodec;
import com.mapr.db.rowcol.SerializationAction;
import com.mapr.db.rowcol.SerializedFamilyInfo;
import com.mapr.fs.jni.MapRConstants;
import com.mapr.fs.proto.Dbfilters.BinaryComparatorProto;
import com.mapr.fs.proto.Dbfilters.ComparatorModeProto;
import com.mapr.fs.proto.Dbfilters.ComparatorProto;
import com.mapr.fs.proto.Dbfilters.ComparatorProto.Builder;
import com.mapr.fs.proto.Dbfilters.CompareOpProto;
import com.mapr.fs.proto.Dbfilters.ConditionFilterProto;
import com.mapr.fs.proto.Dbfilters.FilterComparatorProto;
import com.mapr.fs.proto.Dbfilters.FilterMsg;
import com.mapr.fs.proto.Dbfilters.NullComparatorProto;
import com.mapr.fs.proto.Dbfilters.RegexStringComparatorProto;
import com.mapr.fs.proto.Dbfilters.RowFilterProto;
import com.mapr.fs.proto.Dbfilters.SizeComparatorProto;
import com.mapr.fs.proto.Dbfilters.TypeComparatorProto;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

@API.Internal
class ConditionLeaf extends ConditionNode {

  static Logger logger = LoggerFactory.getLogger(ConditionLeaf.class);

  private static final String REGEX_OPTIONAL_CHARS = "?*";
  private static final String REGEX_SPECIAL_CHARS = "^.+*?()[{\\|$";

  /* never null */
  final FieldPath field;

  /* never null */
  final CompareOpProto op;

  /* never null */
  final ComparatorModeProto cmpMode;

  /* nullable */
  final KeyValue value;

  ConditionLeaf(FieldPath path, QueryCondition.Op op, KeyValue initFrom) {
    this(path, opProtoMap.get(op), initFrom);
  }

  ConditionLeaf(FieldPath field, CompareOpProto op, KeyValue value) {
    this(field, op, value, CMP_VALUE);
  }

  ConditionLeaf(FieldPath field, QueryCondition.Op op, KeyValue value,
      ComparatorModeProto cmpMode) {
    this(field, opProtoMap.get(op), value, cmpMode);
  }

  ConditionLeaf(FieldPath field, CompareOpProto op, KeyValue value,
      ComparatorModeProto cmpMode) {
    this.op = op;
    this.field = field;
    this.value = value;
    this.cmpMode = cmpMode;
    checkArgs();
  }

  ConditionLeaf(String filterId, ByteString serializedState)  {
    FilterComparatorProto filterComparator = null;
    try {
      switch (filterId) {
      case HASH_OF_CONDITION_FILTER: {
        ConditionFilterProto proto = ConditionFilterProto.parseFrom(serializedState);
        this.field = FieldPath.parseFrom(proto.getFieldPath());
        filterComparator = proto.getFilterComparator();
        break;
      }
      case HASH_OF_ROW_FILTER: {
        this.field = ID_FIELD;
        RowFilterProto proto = RowFilterProto.parseFrom(serializedState);
        filterComparator = proto.getFilterComparator();
        break;
      }
      default:
        throw new IllegalArgumentException("Invalid filter id: " + filterId);
      }

      this.op = filterComparator.getCompareOp();

      if (filterComparator.hasComparator()
          && filterComparator.getComparator().hasSerializedComparator()) {
        String name = filterComparator.getComparator().getName();
        ByteString state = filterComparator.getComparator().getSerializedComparator();
        switch (name) {
        case HASH_OF_JSON_COMPARATOR:
          BinaryComparatorProto binaryComparator = BinaryComparatorProto.parseFrom(state);
          ByteBuffer encoded = binaryComparator.getComparable().asReadOnlyByteBuffer();
          encoded.order(ByteOrder.LITTLE_ENDIAN);

          Map <Integer, ByteBuffer> bufferMap = new HashMap<Integer, ByteBuffer>();
          bufferMap.put(0, encoded);
          DBDocumentImpl doc = RowcolCodec.decode(bufferMap, DEFAULT_FAMILY_MAP,
                                                    DEFAULT_NAME_MAP);
          this.value = (KeyValue)doc.getValue(DUMMY_FIELD_X);
          break;
        case HASH_OF_REGEX_COMPARATOR:
          RegexStringComparatorProto regexComparator = RegexStringComparatorProto.parseFrom(state);
          this.value = KeyValueBuilder.initFrom(regexComparator.getPattern().toStringUtf8());
          break;
        case HASH_OF_BINARY_COMPARATOR:
          ByteString byteString = BinaryComparatorProto.parseFrom(state).getComparable();
          this.value = (KeyValue)IdCodec.decode(byteString.asReadOnlyByteBuffer());
          break;
        case HASH_OF_SIZE_COMPARATOR:
          this.value = KeyValueBuilder.initFrom((int)SizeComparatorProto.parseFrom(state).getSize());
          break;
        case HASH_OF_NULL_COMPARATOR:
          this.value = null;
          break;
        case HASH_OF_TYPE_COMPARATOR:
          this.value = KeyValueBuilder.initFrom((int)TypeComparatorProto.parseFrom(state).getType());
          break;
        default:
          throw new IllegalArgumentException("Invalid comparator id: " + name);
        }
      } else {
        this.value = null;
      }

      this.cmpMode = filterComparator.hasComparatorMode() ? filterComparator.getComparatorMode() : null;
    } catch (IOException e) {
      throw new IllegalArgumentException("Failed to decode message", e);
    }
    checkArgs();
  }

  @Override
  protected ConditionLeaf clone() {
    ConditionLeaf newLeaf = (ConditionLeaf) super.clone();
    return newLeaf;
  }

  CompareOpProto getOp() {
    return op;
  }

  FieldPath getField() {
    return field;
  }

  @Override
  boolean checkAndPrune() {
    if (isOnId()
        && (cmpMode == CMP_VALUE)
        && op != CompareOpProto.NOT_EQUAL
        && opProtoMap.containsValue(op)
        && value != null) {
      /*
       * We are assuming that the condition on the _id
       * field has been replaced by the appropriate range
       * scan and hence condition evaluation is not required.
       */
      return true;
    }
    return false;
  }

  @Override
  void addProjections(Set<FieldPath> proj) {
    proj.add(field);
  }

  @Override
  StringBuilder expressionBuilder(StringBuilder sb) {
    return treeBuilder(sb, 0);
  }

  @Override
  StringBuilder treeBuilder(StringBuilder sb, int level) {
    return sb.append(OPEN_PARAN)
        .append(getFieldStr(sb))
        .append(SPACE_CHAR).append(getOpStr(sb)).append(SPACE_CHAR)
        .append(getValueStr(sb))
        .append(CLOSE_PARAN);
  }

  private String getFieldStr(StringBuilder sb) {
    switch (cmpMode) {
    case CMP_PATTERN:
    case CMP_VALUE:
      sb.append(field.asPathString());
      break;
    case CMP_TYPE:
      sb.append("TYPE_OF(").append(field.asPathString()).append(CLOSE_PARAN);
      break;
    case CMP_SIZE:
      sb.append("SIZE_OF(").append(field.asPathString()).append(CLOSE_PARAN);
      break;
    default:
      throw new UnsupportedOperationException(cmpMode + " is currently not supported.");
    }
    return EMPTY_STR;
  }

  private String getValueStr(StringBuilder sb) {
    switch (cmpMode) {
    case CMP_VALUE:
    case CMP_PATTERN:
    case CMP_SIZE:
      sb.append((value != null ? Values.asJsonString(value) : null));
      break;
    case CMP_TYPE:
      sb.append(Value.Type.valueOf(value.getInt()));
      break;
    default:
      throw new UnsupportedOperationException(cmpMode + " is currently not supported.");
    }
    return EMPTY_STR;
  }

  private String getOpStr(StringBuilder sb) {
    switch (cmpMode) {
    case CMP_VALUE:
    case CMP_TYPE:
    case CMP_SIZE:
      sb.append(opSymbolMap.get(op));
      break;
    case CMP_PATTERN:
      sb.append(op == EQUAL ? "MATCHES" : "NOT_MATCHES");
      break;
    default:
      throw new UnsupportedOperationException(cmpMode + " is currently not supported.");
    }
    return EMPTY_STR;
  }

  @Override
  StringBuilder jsonBuilder(StringBuilder sb) {
    return sb.append(OPEN_MAP)
        .append(QUOTE_CHAR).append(field.asPathString()).append(QUOTE_CHAR)
        .append(COLON_CHAR).append(SPACE_CHAR).append(OPEN_MAP)
        .append(QUOTE_CHAR).append(opJsonOpMap.get(op).toLowerCase()).append(QUOTE_CHAR)
        .append(COLON_CHAR).append(SPACE_CHAR)
        //.append(type != null ? ("\"" + type + "\"") : (value != null ? value.toString() : null))
        .append(CLOSE_MAP).append(CLOSE_MAP);
  }

  @Override
  ConditionDescriptor getDescriptor(BiMap<FieldPath, Integer> pathIdMap) {
    if (isOnId()) {
      return toRowKeyFilter(pathIdMap);
    } else {
      return toConditionFilter(pathIdMap);
    }
  }

  boolean isOnId() {
    return (field.getRootSegment().isLastPath()
        && ID_KEY.equals(field.getRootSegment().getNameSegment().getName()));
  }

  private void checkArgs() {
    if (cmpMode == null) {
      throw new IllegalArgumentException("ComparatorModeProto can not be null.");
    } else if (op == null) {
      throw new IllegalArgumentException("CompareOpProto can not be null.");
    } else if (field == null) {
      throw new IllegalArgumentException("FieldPath can not be null.");
    } else if (isOnId()
        && (cmpMode == CMP_VALUE || cmpMode == CMP_PATTERN)
        && value != null
        && !IdCodec.isSupportedType(value.getType())) {
      throw new IllegalArgumentException(String.format(
          "A %s value can not be used for '_id' field.", value.getType()));
    } else if (cmpMode == CMP_PATTERN) {
      Pattern.compile(value.getString()); // test for valid regex
    }
  }

  private ConditionDescriptor toRowKeyFilter(BiMap<FieldPath, Integer> pathIdMap) {
    RowFilterProto rowFilter = RowFilterProto.newBuilder()
        .setFilterComparator(getComparator(true))
        .build();
    FilterMsg filterMsg = FilterMsg.newBuilder()
        .setId(HASH_OF_ROW_FILTER)
        .setSerializedState(rowFilter.toByteString())
        .build();

    ImmutableMap.Builder<Integer, Set<FieldPath>> mapBuilder = ImmutableMap.builder();
    for (Entry<Integer, FieldPath> pathIdEntry : pathIdMap.inverse().entrySet()) {
      mapBuilder.put(pathIdEntry.getKey(), ImmutableSet.of(pathIdEntry.getValue()));
    }

    return new ConditionDescriptor(filterMsg, mapBuilder.build());
  }

  private ConditionDescriptor toConditionFilter(BiMap<FieldPath, Integer> pathIdMap) {
    /*
     * Start with assuming "default" column family.
     */
    FieldPath fieldPath = field;
    Integer familyId = pathIdMap.get(FieldPath.EMPTY);
    if (familyId == null) {
      throw new IllegalStateException("Unable to find family id for the default column family.");
    }

    /*
     * Now see if the field path of this condition
     * leaf is under a column family's FieldPath.
     */
    FieldPath lastParent = null;
    for (Entry<FieldPath, Integer> kv : pathIdMap.entrySet()) {
      FieldPath familyPath = kv.getKey();
      if (familyPath.equals(FieldPath.EMPTY)) continue;
      FieldSegment progeny = field.segmentAfterAncestor(familyPath);
      if (progeny != null) {
        if (lastParent == null || lastParent.isAtOrAbove(familyPath)) {
          lastParent = familyPath;
          fieldPath = progeny.equals(FieldPath.EMPTY.getRootSegment())
              ? DUMMY_FIELDPATH_V : DUMMY_FIELDPATH_V.cloneWithNewChild(progeny);
          familyId = kv.getValue();
        }
      }
    }

    ConditionFilterProto conditionFilter = ConditionFilterProto.newBuilder()
        .setFieldPath(fieldPath.asPathString(false))
        .setFamilyId(familyId)
        .setFilterComparator(getComparator(false))
        .build();
    FilterMsg filterMsg = FilterMsg.newBuilder()
        .setId(HASH_OF_CONDITION_FILTER)
        .setSerializedState(conditionFilter.toByteString())
        .build();
    return new ConditionDescriptor(filterMsg, ImmutableMap.of(familyId, ImmutableSet.of(fieldPath)));
  }

  private FilterComparatorProto getComparator(boolean onRowKey) {
    FilterComparatorProto.Builder filterComparator = FilterComparatorProto
        .newBuilder()
        .setComparatorMode(cmpMode)
        .setCompareOp(op);
    filterComparator.setComparator(toComparator(onRowKey));
    return filterComparator.build();
  }

  private ComparatorProto toComparator(boolean onId) {
    Builder builder = ComparatorProto.newBuilder();
    switch (cmpMode) {
    case CMP_VALUE:
      return buildValueComparator(builder, onId);
    case CMP_TYPE:
      return buildTypeComparator(builder, onId);
    case CMP_PATTERN:
      return buildPatternComparator(builder, onId);
    case CMP_SIZE:
      return buildSizeComparator(builder, onId);
    default:
      throw new UnsupportedOperationException(cmpMode + " is currently not supported.");
    }
  }

  private ComparatorProto buildSizeComparator(Builder builder, boolean onId) {
    ByteString state = SizeComparatorProto.newBuilder()
        .setSize(value.getInt())
        .setOnIdField(onId)
        .build()
        .toByteString();
    return builder
        .setName(HASH_OF_SIZE_COMPARATOR)
        .setSerializedComparator(state).build();
  }

  private ComparatorProto buildPatternComparator(Builder builder, boolean onId) {
    ByteString pattern = ByteString.copyFrom(Bytes.toBytes(value.getString()));
    ByteString state = RegexStringComparatorProto.newBuilder()
        .setPattern(pattern)
        .setIsUTF8(true)
        .setOnIdField(onId)
        .build()
        .toByteString();
    return builder
        .setName(HASH_OF_REGEX_COMPARATOR)
        .setSerializedComparator(state).build();
  }

  private ComparatorProto buildTypeComparator(Builder builder, boolean onId) {
    ByteString state = TypeComparatorProto.newBuilder()
        .setType(value.getInt())
        .setOnIdField(onId)
        .build()
        .toByteString();
    return builder
        .setName(HASH_OF_TYPE_COMPARATOR)
        .setSerializedComparator(state).build();
  }

  private ComparatorProto buildValueComparator(Builder builder, boolean onId) {
    String name = null;
    ByteString state = null;
    if (value == null) {
      name = HASH_OF_NULL_COMPARATOR;
      state = NullComparatorProto.newBuilder().build().toByteString();
    } else {
      ByteString comparable = null;
      if (onId) {
        name = HASH_OF_BINARY_COMPARATOR;
        comparable = ByteString.copyFrom(IdCodec.encode(value));
      } else if (value != null) {
        name = HASH_OF_JSON_COMPARATOR;
        Document rec = new DBDocumentImpl();
        rec.set(DUMMY_FIELD_X, value);
        SerializedFamilyInfo[] famInfo = RowcolCodec.encode(rec, DEFAULT_FAMILY_MAP);
        assert(famInfo.length == 1);
        assert(famInfo[0].getAction() == SerializationAction.SET);
        comparable = ByteString.copyFrom(famInfo[0].getByteBuffer());
      }
      state = BinaryComparatorProto.newBuilder()
          .setComparable(comparable)
          .setOnIdField(onId)
          .build()
          .toByteString();
    }
    return builder
        .setName(name)
        .setSerializedComparator(state).build();
  }

  @Override
  List<RowkeyRange> getRowkeyRanges() {
    if (!isOnId() || value == null
        || (cmpMode != CMP_VALUE && cmpMode != CMP_PATTERN)) {
      return FULL_TABLE_RANGE;
    }

    byte[] fieldValue = IdCodec.encodeAsBytes(value);

    byte[] startRow = MapRConstants.EMPTY_BYTE_ARRAY;
    byte[] stopRow = MapRConstants.EMPTY_BYTE_ARRAY;
    switch (op) {
    case GREATER_OR_EQUAL:
      startRow = fieldValue;
      break;
    case GREATER:
      startRow = Bytes.appendZeroByte(fieldValue);
      break;
    case LESS_OR_EQUAL:
      stopRow = Bytes.appendZeroByte(fieldValue);
      break;
    case LESS:
      stopRow = fieldValue;
      break;
    case EQUAL:
      if (cmpMode == ComparatorModeProto.CMP_PATTERN) {
        /*
         * Traverse the supplied regex pattern and see if there is
         * a literal prefix, e.g. "device.*" or "^user.*".
         */
        boolean lastEscaped = false;
        boolean inRegexQuote = false; // \Q...\E
        StringBuilder prefixSB = new StringBuilder();
        String regexString = value.getString();
        if (regexString.charAt(0) == '^') {
          // remove the leading anchor, if exist
          regexString = regexString.substring(1);
        }
        for (int i = 0; i < regexString.length(); i++) {
          char ch = regexString.charAt(i);
          if (inRegexQuote) {
            prefixSB.append(ch);
          }
          if (lastEscaped) {
            lastEscaped = false;
            if (ch == 'Q') {
              inRegexQuote = true;
              continue;
            } else if (ch == 'E') {
              if (inRegexQuote) {
                prefixSB.setLength(prefixSB.length()-2);
              }
              inRegexQuote = false;
              continue;
            } else if (REGEX_SPECIAL_CHARS.indexOf(ch) != -1) {
              prefixSB.append(ch);
              continue;
            } else {
              break; // must be a character class, e.g. \d, \w, etc
            }
          } else if (ch == '\\') {
            lastEscaped = true;
            continue;
          } else if (!inRegexQuote) {
            if (REGEX_OPTIONAL_CHARS.indexOf(ch) != -1) {
              if (prefixSB.length() > 0) {
                // we encountered an optional meta-char (.|*)
                // must remove the last char from prefix string.
                prefixSB.setLength(prefixSB.length()-1);
              }
              break; // search for prefix ends
            } else if (REGEX_SPECIAL_CHARS.indexOf(ch) != -1) {
              break; // search for prefix ends
            }
            // literal char, append to the prefix
            prefixSB.append(ch);
          }
        }
        if (prefixSB.length() > 0) { // found some prefix
          String prefix = prefixSB.toString();
          startRow = Bytes.getBytes(IdCodec.encode(prefix));
          stopRow = Bytes.unsignedCopyAndIncrement(startRow);
          if (stopRow.length > startRow.length) { //overflow
            stopRow = MapRConstants.EMPTY_BYTE_ARRAY;
          }
        }
      } else {
        startRow = fieldValue;
        stopRow = Bytes.appendZeroByte(fieldValue);
      }
    default:
      break;
    }

    if (startRow != MapRConstants.EMPTY_BYTE_ARRAY
        || stopRow != MapRConstants.EMPTY_BYTE_ARRAY) {
      return ImmutableList.of(new RowkeyRange(startRow, stopRow));
    }
    return FULL_TABLE_RANGE;
  }

}
