/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import static com.mapr.db.impl.ConditionBlock.BlockType.and;
import static com.mapr.db.impl.ConditionBlock.BlockType.or;
import static com.mapr.db.impl.Constants.DEFAULT_FAMILY_MAP;
import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;
import static com.mapr.fs.proto.Dbfilters.ComparatorModeProto.CMP_PATTERN;
import static com.mapr.fs.proto.Dbfilters.ComparatorModeProto.CMP_SIZE;
import static com.mapr.fs.proto.Dbfilters.ComparatorModeProto.CMP_TYPE;
import static com.mapr.fs.proto.Dbfilters.CompareOpProto.EQUAL;
import static com.mapr.fs.proto.Dbfilters.CompareOpProto.NOT_EQUAL;
import static org.ojai.DocumentConstants.ID_FIELD;

import java.math.BigDecimal;
import java.nio.ByteBuffer;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import org.ojai.FieldPath;
import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.annotation.API;
import org.ojai.store.QueryCondition;
import org.ojai.types.ODate;
import org.ojai.types.OInterval;
import org.ojai.types.OTime;
import org.ojai.types.OTimestamp;

import com.google.common.collect.BiMap;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.db.impl.ConditionBlock.BlockType;
import com.mapr.db.impl.ConditionNode.RowkeyRange;
import com.mapr.fs.proto.Dbfilters.FilterMsg;

/**
 * Do not instantiate this class directly. Use the {@code QueryCondition} interface.
 */
@API.Internal
public class ConditionImpl implements QueryCondition {

  private static final String EMPTY = "<EMPTY>";
  private static final String EMPTY_DOC = "{}";
  private static final ConditionDescriptor EMPTY_DESC = new ConditionDescriptor(ByteBuffer.wrap(new byte[0]));

  private ConditionNode root;
  private boolean built = false;
  private Stack<ConditionBlock> groupsStack;
  private List<RowkeyRange> rowkeyRanges;

  /**
   * Empty constructor.
   */
  public ConditionImpl() { }

  ConditionImpl(String id, ByteString serializedState) {
    switch (id) {
    case ConditionNode.HASH_OF_ALWAYSFALSE_FILTER:
        root = new AlwaysFalseCondition();
        break;
    case ConditionNode.HASH_OF_ROW_FILTER:
    case ConditionNode.HASH_OF_CONDITION_FILTER:
      root = new ConditionLeaf(id, serializedState);
      break;
    case ConditionNode.HASH_OF_FILTER_LIST:
      root = new ConditionBlock(serializedState);
      break;
    default:
      throw new IllegalArgumentException("Unknown filter in the serialized message: " + id);
    }
    build();
  }

  @Override
  public boolean isEmpty() {
    return root == null || root.isEmpty();
  }

  @Override
  public boolean isBuilt() {
    return built;
  }

  /**
   * @return a {@code String} representing the prefix form of this
   *         condition expression
   */
  public String asPrefix() {
    if (root == null) return EMPTY;

    StringBuilder sb = new StringBuilder();
    sb.append("[\n");
    root.treeBuilder(sb);
    if (built) {
      sb.append("\n]");
    }
    return sb.toString();
  }

  public String jsonString() {
    if (root == null || root.isEmpty()) return EMPTY_DOC;
    StringBuilder sb = new StringBuilder();
    root.jsonBuilder(sb);
    return sb.toString();
  }

  // Find all the fields/paths in the condition which will be treated as
  // projections for get/scan
  public Set<FieldPath> getProjections() {
    Set<FieldPath> proj = new HashSet<FieldPath>();
    if (!isEmpty()) {
      root.addProjections(proj);
    }

    // Do not add "_id" even if it is in condition, as it causes default
    // CF to be fetched
    proj.remove(ID_FIELD);
    return proj;
  }

  /**
   * @return a {@code String} representing the infix form of this
   *         condition expression.
   */
  public String asInfix() {
    return isEmpty() ? EMPTY : root.expressionBuilder(new StringBuilder()).toString();
  }

  /**
   * Returns a string representation of the object. Same as returned
   * by {@link #asInfix()}.
   */
  @Override
  public String toString() {
    return asInfix();
  }

  @Override
  public int hashCode() {
    return getDescriptor(DEFAULT_FAMILY_MAP).hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    ConditionImpl other = (ConditionImpl) obj;
    // we do not compare the row key ranges
    // since they are not part of serialized state
    return getDescriptor(DEFAULT_FAMILY_MAP).equals(other.getDescriptor(DEFAULT_FAMILY_MAP));
  }

  @Override
  public ConditionImpl and() {
    return addGroup(new ConditionBlock(BlockType.and));
  }

  @Override
  public ConditionImpl or() {
    return addGroup(new ConditionBlock(BlockType.or));
  }

  @Override
  public ConditionImpl close() {
    if (groupsStack == null || groupsStack.isEmpty()) {
      throw new IllegalStateException("Not in a condition block.");
    }
    groupsStack.pop().close();
    return this;
  }

  @Override
  public ConditionImpl build() {
    return build(false);
  }

  public ConditionImpl cloneOptimized() {
    ConditionImpl pruned = new ConditionImpl();
    if (root != null) {
      pruned.root = root.clone();
    }
    return pruned.build(true);
  }

  @Override
  public ConditionImpl condition(QueryCondition conditionToAdd) {
    if (conditionToAdd == null) {
      throw new IllegalArgumentException(
          "A null condition can not be added.");
    } else if (conditionToAdd == this) {
      throw new IllegalArgumentException(
          "A condition can not be added to itself");
    } else if (!conditionToAdd.isBuilt()) {
      throw new IllegalArgumentException(
          "The specified condition is not built\n" + conditionToAdd);
    } else if (conditionToAdd.isEmpty()) {
      throw new IllegalArgumentException("Can not add an empty condition");
    }
    checkStateForModification();

    ConditionImpl other = (ConditionImpl) conditionToAdd;
    if (root == null) {
      root = other.getRoot().clone();
    } else {
      groupsStack.peek().add(other.getRoot().clone());
    }
    return this;
  }

  @Override
  public ConditionImpl exists(String path) {
    return exists(FieldPath.parseFrom(path));
  }

  @Override
  public ConditionImpl exists(FieldPath path) {
    return addLeaf(new ConditionLeaf(path, NOT_EQUAL, null));
  }

  @Override
  public ConditionImpl notExists(String path) {
    return notExists(FieldPath.parseFrom(path));
  }

  @Override
  public ConditionImpl in(String path, List<? extends Object> listOfValue) {
    return in(FieldPath.parseFrom(path), listOfValue);
  }

  @Override
  public ConditionImpl in(FieldPath path, List<? extends Object> listOfValue) {
    if (listOfValue.isEmpty()) {
        return addAlwaysFalseCondition(new AlwaysFalseCondition());
    }

    ConditionBlock inBlock = new ConditionBlock(or);
    for (Object value : listOfValue) {
      inBlock.add(new ConditionLeaf(path, EQUAL, KeyValueBuilder.initFromObject(value)));
    }
    return addGroup(inBlock.close()).close();
  }

  @Override
  public ConditionImpl notIn(String path, List<? extends Object> listOfValue) {
    return notIn(FieldPath.parseFrom(path), listOfValue);
  }

  @Override
  public ConditionImpl notIn(FieldPath path, List<? extends Object> listOfValue) {
    ConditionBlock notInBlock = new ConditionBlock(and);
    for (Object value : listOfValue) {
      notInBlock.add(new ConditionLeaf(path, NOT_EQUAL, KeyValueBuilder.initFromObject(value)));
    }
    return addGroup(notInBlock.close()).close();
  }

  @Override
  public ConditionImpl notExists(FieldPath path) {
    return addLeaf(new ConditionLeaf(path, EQUAL, null));
  }

  @Override
  public ConditionImpl typeOf(String path, Value.Type type) {
    return typeOf(FieldPath.parseFrom(path), type);
  }

  @Override
  public ConditionImpl typeOf(FieldPath path, Value.Type type) {
    return addLeaf(new ConditionLeaf(path, EQUAL,
        KeyValueBuilder.initFrom((int)type.getCode()), CMP_TYPE));
  }

  @Override
  public ConditionImpl notTypeOf(String path, Type type) {
    return notTypeOf(FieldPath.parseFrom(path), type);
  }

  @Override
  public ConditionImpl notTypeOf(FieldPath path, Type type) {
    return addLeaf(new ConditionLeaf(path, NOT_EQUAL,
        KeyValueBuilder.initFrom((int)type.getCode()), CMP_TYPE));
  }

  @Override
  public ConditionImpl matches(String path, String regex) {
    return matches(FieldPath.parseFrom(path), regex);
  }

  @Override
  public ConditionImpl matches(FieldPath path, String regex) {
    return addLeaf(new ConditionLeaf(path, EQUAL, KeyValueBuilder.initFrom(regex), CMP_PATTERN));
  }

  @Override
  public ConditionImpl notMatches(String path, String regex) {
    return notMatches(FieldPath.parseFrom(path), regex);
  }

  @Override
  public ConditionImpl notMatches(FieldPath path, String regex) {
    return addLeaf(new ConditionLeaf(path, NOT_EQUAL, KeyValueBuilder.initFrom(regex), CMP_PATTERN));
  }

  @Override
  public QueryCondition like(String path, String likeExpression) {
    return matches(path,
        new LikeToRegexConvertor(likeExpression).parse().getRegexString());
  }

  @Override
  public QueryCondition like(FieldPath path, String likeExpression) {
    return matches(path,
        new LikeToRegexConvertor(likeExpression).parse().getRegexString());
  }

  @Override
  public QueryCondition like(String path, String likeExpression, Character escapeChar) {
    return matches(path,
        new LikeToRegexConvertor(likeExpression, escapeChar).parse().getRegexString());
  }

  @Override
  public QueryCondition like(FieldPath path, String likeExpression, Character escapeChar) {
    return matches(path,
        new LikeToRegexConvertor(likeExpression, escapeChar).parse().getRegexString());
  }

  @Override
  public QueryCondition notLike(String path, String likeExpression) {
    return notMatches(path,
        new LikeToRegexConvertor(likeExpression).parse().getRegexString());
  }

  @Override
  public QueryCondition notLike(FieldPath path, String likeExpression) {
    return notMatches(path,
        new LikeToRegexConvertor(likeExpression).parse().getRegexString());
  }

  @Override
  public QueryCondition notLike(String path, String likeExpression, Character escapeChar) {
    return notMatches(path,
        new LikeToRegexConvertor(likeExpression, escapeChar).parse().getRegexString());
  }

  @Override
  public QueryCondition notLike(FieldPath path, String likeExpression, Character escapeChar) {
    return notMatches(path,
        new LikeToRegexConvertor(likeExpression, escapeChar).parse().getRegexString());
  }

  @Override
  public QueryCondition is(String path, QueryCondition.Op op, boolean value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, boolean value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, String value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, String value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, byte value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, byte value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, short value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, short value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, int value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, int value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, long value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, long value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, float value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, float value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, double value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, double value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, BigDecimal value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, BigDecimal value) {
    throw new UnsupportedOperationException("BigDecimal type not supported");
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, ODate value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, ODate value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, OTime value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, OTime value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, OTimestamp value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, OTimestamp value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, OInterval value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, OInterval value) {
    throw new UnsupportedOperationException("Interval type not supported");
  }

  @Override
  public ConditionImpl is(String path, QueryCondition.Op op, ByteBuffer value) {
    return is(FieldPath.parseFrom(path), op, value);
  }

  @Override
  public ConditionImpl is(FieldPath path, QueryCondition.Op op, ByteBuffer value) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public QueryCondition sizeOf(String path, QueryCondition.Op op, long size) {
    return sizeOf(FieldPath.parseFrom(path), op, size);
  }

  @Override
  public QueryCondition sizeOf(FieldPath path, QueryCondition.Op op, long size) {
    return addLeaf(new ConditionLeaf(path, op, KeyValueBuilder.initFrom((int) size), CMP_SIZE));
  }

  @Override
  public ConditionImpl equals(String path, Map<String, ? extends Object> value) {
    return equals(FieldPath.parseFrom(path), value);
  }

  @Override
  public ConditionImpl equals(FieldPath path, Map<String, ? extends Object> value) {
    return addLeaf(new ConditionLeaf(path, EQUAL, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl equals(String path, List<? extends Object> value) {
    return equals(FieldPath.parseFrom(path), value);
  }

  @Override
  public ConditionImpl equals(FieldPath path, List<? extends Object> value) {
    return addLeaf(new ConditionLeaf(path, EQUAL, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl notEquals(String path, Map<String, ? extends Object> value) {
    return notEquals(FieldPath.parseFrom(path), value);
  }

  @Override
  public ConditionImpl notEquals(FieldPath path, Map<String, ? extends Object> value) {
    return addLeaf(new ConditionLeaf(path, NOT_EQUAL, KeyValueBuilder.initFrom(value)));
  }

  @Override
  public ConditionImpl notEquals(String path, List<? extends Object> value) {
    return notEquals(FieldPath.parseFrom(path), value);
  }

  @Override
  public ConditionImpl notEquals(FieldPath path, List<? extends Object> value) {
    return addLeaf(new ConditionLeaf(path, NOT_EQUAL, KeyValueBuilder.initFrom(value)));
  }

  /**
   * @return a {@code ByteBuffer} containing the serialized form of this condition
   */
  @API.Internal
  public ConditionDescriptor getDescriptor() {
    return getDescriptor(DEFAULT_FAMILY_MAP);
  }

  /**
   * @param pathIdMap a Mapping between {@link FieldPath} and Column Family ID
   *
   * @return a {@code ByteBuffer} containing the serialized form of this condition
   */
  @API.Internal
  public ConditionDescriptor getDescriptor(BiMap<FieldPath, Integer> pathIdMap) {
    checkIfBuilt();
    return isEmpty() ? EMPTY_DESC : root.getDescriptor(pathIdMap);
  }

  /**
   * @return a {@code List} of {@linkplain RowkeyRange} over which
   */
  @API.Internal
  public List<RowkeyRange> getRowkeyRanges() {
    checkIfBuilt();
    return rowkeyRanges;
  }

  /**
   * Parses the serialized {@code QueryCondition} from the bytes in the specified {@code ByteBuffer}.
   * @param buf the {@code ByteBuffer} to parse
   * @return the parsed condition
   * @throws IllegalArgumentException if the bytes in the specified {@code ByteBuffer}
   *         cannot be parsed as a {@code QueryCondition}
   */
  @API.Internal
  public static QueryCondition parseFrom(ByteBuffer buf) {
    try {
      if (buf.remaining() > 0) {
        FilterMsg msg = FilterMsg.parseFrom(ByteString.copyFrom(buf));
        return new ConditionImpl(msg.getId(), msg.getSerializedState());
      } else {
        return new ConditionImpl().build();
      }
    } catch (InvalidProtocolBufferException e) {
      throw new IllegalArgumentException("Unable to parse the provided data.", e);
    }
  }

  /* package methods */

  ConditionNode getRoot() {
    return root;
  }

  /* private methods */

  /**
   * Adds the new leaf to appropriate node
   * @param leaf the leaf to add
   * @return {@code this}
   */
  private ConditionImpl addLeaf(ConditionLeaf leaf) {
    checkStateForModification();
    if (root == null) {
      root = leaf;
    } else {
      groupsStack.peek().add(leaf);
    }
    return this;
  }

  private ConditionImpl addAlwaysFalseCondition(AlwaysFalseCondition fc) {
    checkStateForModification();
    if (root == null) {
      root = fc;
    } else {
      groupsStack.peek().add(fc);
    }
    return this;
  }

  private ConditionImpl addGroup(ConditionBlock newGroup) {
    checkStateForModification();
    if (groupsStack == null) {
      groupsStack = new Stack<ConditionBlock>();
    }

    if (root == null) {
      root = newGroup;
    } else {
      groupsStack.peek().add(newGroup);
    }
    groupsStack.push(newGroup);
    return this;
  }

  private ConditionImpl build(boolean optimize) {
    if (built) {
      throw new IllegalStateException("The condition is already built.");
    }
    if (groupsStack != null && !groupsStack.isEmpty()) {
      throw new IllegalStateException(
          "At least one condition group is not closed." + asPrefix());
    }
    if (root != null && !root.isEmpty()) {
      rowkeyRanges = root.getRowkeyRanges();
      if (optimize) {
        if (root.checkAndPrune()) {
          root = null;
        }
      }
    } else {
      rowkeyRanges = ConditionNode.FULL_TABLE_RANGE;
    }
    built = true;
    return this;
  }

  private void checkIfBuilt() {
    if (!built) {
      throw new IllegalStateException("The condition is not built yet.\n" + asPrefix());
    }
  }

  private void checkStateForModification() {
    if (built) {
      throw new IllegalStateException("The condition is already built.");
    } else if (root != null && groupsStack == null) {
      throw new IllegalArgumentException("A condition can only be"
          + " added as root or a child of another logical connecter.");
    }
  }

}
