package com.mapr.db.impl;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.ojai.FieldPath;
import org.ojai.annotation.API;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.ByteString;
import com.mapr.fs.jni.MapRConstants;
import com.mapr.fs.proto.Dbfilters.FilterListProto;
import com.mapr.fs.proto.Dbfilters.FilterListProto.OperatorProto;
import com.mapr.fs.proto.Dbfilters.FilterMsg;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

@API.Internal
class ConditionBlock extends ConditionNode {
  enum BlockType {and, or}

  private BlockType type;
  private List<ConditionNode> children;
  private boolean closed = false;

  ConditionBlock(BlockType type) {
    this.type = type;
    this.children = Lists.newArrayList();
  }

  ConditionBlock(ByteString serializedState) {
    try {
      FilterListProto proto = FilterListProto.parseFrom(serializedState);
      this.type = proto.getOperator() == OperatorProto.MUST_PASS_ALL ? BlockType.and : BlockType.or;
      List<ConditionNode> nodeList = new ArrayList<ConditionNode>(proto.getFiltersCount());
      for (FilterMsg filter : proto.getFiltersList()) {
        String nodeId = filter.getId();
        ByteString nodeState = filter.getSerializedState();
        switch (nodeId) {
        case ConditionNode.HASH_OF_ALWAYSFALSE_FILTER:
          nodeList.add(new AlwaysFalseCondition());
          break;
        case ConditionNode.HASH_OF_ROW_FILTER:
        case ConditionNode.HASH_OF_CONDITION_FILTER:
          nodeList.add(new ConditionLeaf(nodeId, nodeState));
          break;
        case ConditionNode.HASH_OF_FILTER_LIST:
          nodeList.add(new ConditionBlock(nodeState));
          break;
        default:
          throw new IllegalArgumentException("Unknown filter in the serialized message: " + nodeId);
        }
      }
      this.children = nodeList;
      closed = true;
    } catch (IOException e) {
      throw new IllegalArgumentException("Failed to decode message", e);
    }
  }

  @Override
  protected ConditionBlock clone() {
    ConditionBlock newBlock = (ConditionBlock) super.clone();
    newBlock.children = Lists.newArrayList();
    for (int i = 0; i < children.size(); i++) {
      newBlock.children.add((ConditionNode) children.get(i).clone());
    }
    return newBlock;
  }

  BlockType getType() {
    return type;
  }

  void add(ConditionNode node) {
    children.add(node);
  }

  @Override
  public boolean isEmpty() {
    return children == null || children.size() == 0;
  }

  @Override
  boolean isLeaf() {
    return false;
  }

  List<ConditionNode> getChildren() {
    return children;
  }

  void addProjections(Set<FieldPath> proj) {
    for (int i = 0; i < children.size(); i++) {
      children.get(i).addProjections(proj);
    }
  }

  @Override
  StringBuilder expressionBuilder(StringBuilder sb) {
    sb.append(OPEN_PARAN);
    for (int i = 0; i < children.size(); i++) {
      children.get(i).expressionBuilder(sb);
      if (i < children.size()-1) {
        sb.append(SPACE_CHAR).append(type).append(SPACE_CHAR);
      }
    }
    sb.append(CLOSE_PARAN);
    return sb;
  }

  @Override
  StringBuilder treeBuilder(StringBuilder sb, int level) {
    int indent = 2 * level; level++;
    sb.append(type).append(" [");
    for (ConditionNode node : children) {
      sb.append(NEWLINE_CHAR);
      indent(sb, indent);
      node.treeBuilder(sb, level);
    }
    if (closed) {
      sb.append(NEWLINE_CHAR);
      indent(sb, indent-2).append(']');
    }
    return sb;
  }

  @Override
  StringBuilder jsonBuilder(StringBuilder sb) {
    sb.append(OPEN_MAP)
      .append(QUOTE_CHAR)
      .append(DOLLAR_CHAR)
      .append(type)
      .append(QUOTE_CHAR)
      .append(COLON_CHAR)
      .append(SPACE_CHAR)
      .append(OPEN_BRACKET);
    for (ConditionNode node : children) {
      node.jsonBuilder(sb).append(COMMA_CHAR).append(SPACE_CHAR);
    }
    if (sb.charAt(sb.length()-1) != OPEN_BRACKET) {
      sb.delete(sb.length()-2, sb.length());
    }
    sb.append(CLOSE_BRACKET)
      .append(CLOSE_MAP);
    return sb;
  }

  @Override
  ConditionDescriptor getDescriptor(BiMap<FieldPath, Integer> pathIdMap) {
    FilterListProto.Builder builder = FilterListProto.newBuilder();
    builder.setOperator(type == BlockType.and ? OperatorProto.MUST_PASS_ALL : OperatorProto.MUST_PASS_ONE);
    Map<Integer, Set<FieldPath>> blockFamilyFieldPathsMap = Maps.newHashMap();
    for (ConditionNode filter : children) {
      ConditionDescriptor child = filter.getDescriptor(pathIdMap);
      builder.addFilters(child.getFilterMsg());
      Map<Integer, ? extends Set<FieldPath>> childFamilyFieldPathsMap = child.getFamilyFieldPathsMap();
      for (Entry<Integer, ? extends Set<FieldPath>> childFamilyFieldPathsEntry : childFamilyFieldPathsMap.entrySet()) {
        Integer familyId = childFamilyFieldPathsEntry.getKey();
        Set<FieldPath> set = blockFamilyFieldPathsMap.get(familyId);
        if (set == null) {
          blockFamilyFieldPathsMap.put(familyId, (set = Sets.newTreeSet()));
        }
        set.addAll(childFamilyFieldPathsEntry.getValue());
      }
    }
    FilterMsg filterMsg = FilterMsg.newBuilder()
        .setId(HASH_OF_FILTER_LIST)
        .setSerializedState(builder.build().toByteString())
        .build();
    return new ConditionDescriptor(filterMsg, blockFamilyFieldPathsMap);
  }

  @Override
  List<RowkeyRange> getRowkeyRanges() {
    if (children.size() == 0) {
      return FULL_TABLE_RANGE;
    }

    List<RowkeyRange> nodeRanges = children.get(0).getRowkeyRanges();

    for (int i = 1; i < children.size(); ++i) {
      nodeRanges = mergeNodeRanges(nodeRanges, children.get(i).getRowkeyRanges());
    }
    return nodeRanges;
  }

  ConditionBlock close() {
    closed = true;
    return this;
  }

  @Override
  boolean checkAndPrune() {
    if (type == BlockType.and) {
      Iterator<ConditionNode> itr = children.iterator();
      while(itr.hasNext()) {
        if (itr.next().checkAndPrune()) {
          itr.remove();
        }
      }
    }
    if (children.size() == 1 && !children.get(0).isLeaf()) {
      ConditionBlock rootBlock = (ConditionBlock) children.get(0);
      children = rootBlock.children;
      type = rootBlock.type;
    }
    return children.size() == 0;
  }

  private List<RowkeyRange> mergeNodeRanges(
      List<RowkeyRange> leftRanges, List<RowkeyRange> rightRanges) {
    byte[] startRow = MapRConstants.EMPTY_BYTE_ARRAY;
    byte[] stopRow = MapRConstants.EMPTY_BYTE_ARRAY;

    byte[] leftStartRow = leftRanges.get(0).getStartRow();
    byte[] leftStopRow = leftRanges.get(0).getStopRow();
    byte[] rightStartRow = rightRanges.get(0).getStartRow();
    byte[] rightStopRow = rightRanges.get(0).getStopRow();

    switch (type) {
    case and:
      startRow = Bytes.maxOfStartRows(leftStartRow, rightStartRow);
      stopRow = Bytes.minOfStopRows(leftStopRow, rightStopRow);
      break;
    case or:
      startRow = Bytes.minOfStartRows(leftStartRow, rightStartRow);
      stopRow = Bytes.maxOfStopRows(leftStopRow, rightStopRow);
      break;
    default:
      assert false : type + " is not handled.";
    }

    if (startRow != MapRConstants.EMPTY_BYTE_ARRAY
        || stopRow != MapRConstants.EMPTY_BYTE_ARRAY) {
      return ImmutableList.of(new RowkeyRange(startRow, stopRow));
    }
    return FULL_TABLE_RANGE;
 }

}
