/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.impl;

import static com.mapr.db.impl.Constants.DEFAULT_FAMILY;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.ojai.FieldPath;
import org.ojai.Value;
import org.ojai.annotation.API;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.mapr.db.Admin;
import com.mapr.db.FamilyDescriptor;
import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.TableDescriptor;
import com.mapr.db.exceptions.DBException;
import com.mapr.db.exceptions.ExceptionHandler;
import com.mapr.db.exceptions.FamilyExistsException;
import com.mapr.db.exceptions.FamilyNotFoundException;
import com.mapr.db.exceptions.OpNotPermittedException;
import com.mapr.db.exceptions.TableExistsException;
import com.mapr.db.exceptions.TableNotFoundException;
import com.mapr.fs.ErrnoException;
import com.mapr.fs.MapRFileStatus;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.jni.Errno;
import com.mapr.fs.jni.MapRConstants;
import com.mapr.fs.proto.Dbserver.ColumnFamilyAttr;
import com.mapr.fs.proto.Dbserver.SchemaFamily;
import com.mapr.fs.tables.TableProperties;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

@API.Internal
public class AdminImpl implements Admin {
  final private MapRFileSystem  maprfs_;
  final private Configuration conf_;
  private boolean shoulCloseFS_ = false;

  public AdminImpl() throws DBException {
    this(new Configuration());
  }

  public AdminImpl(Configuration conf) throws DBException {
    this(newMapRFS(conf));
    shoulCloseFS_ = true;
  }

  public AdminImpl(MapRFileSystem maprfs) throws DBException {
    maprfs_ = maprfs;
    conf_ = maprfs.getConf();
  }

  private static MapRFileSystem newMapRFS(Configuration conf) throws DBException {
    try {
      MapRFileSystem maprfs = new MapRFileSystem();
      maprfs.initialize(new URI(MapRConstants.MAPRFS_PREFIX), conf);
      return maprfs;
    } catch (Exception e) {
      throw new DBException(e); /* not happening */
    }
  }

  @Override
  public synchronized void close() throws DBException {
    try {
      if (shoulCloseFS_) {
        maprfs_.close();
        shoulCloseFS_ = false;
      }
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "close()");
    }
  }

  @Override
  public List<Path> listTables() throws DBException {
    return listTables((Path)null);
  }

  @Override
  public List<Path> listTables(String folderOrPattern) throws DBException {
    return listTables(folderOrPattern == null ? null : new Path(folderOrPattern));
  }

  @Override
  public List<Path> listTables(Path folderOrPattern) throws DBException {
    try {
      return _listTables(folderOrPattern);
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "listTables()");
    }
  }

  private List<Path> _listTables(Path folderOrPattern) throws IOException {
    FileStatus[] children;
    try {
      if (folderOrPattern == null) {
        folderOrPattern = maprfs_.getWorkingDirectory();
      }
      MapRFileStatus patternStat = maprfs_.getMapRFileStatus(folderOrPattern);
      if (patternStat.isDirectory()) {
        children = maprfs_.listMapRStatus(folderOrPattern, false, false);
      } else {
        children = new MapRFileStatus[] {patternStat};
      }
    } catch (FileNotFoundException e) {
      children = maprfs_.globStatus(folderOrPattern);
    }
    List<Path> tables = Lists.newLinkedList();
    if (children != null) {
      for (FileStatus fileStatus : children) {
        if (fileStatus.isTable()) {
          // FIXME: this is not optimal, we should be able to get Json attribute in MapRFileStatus
          TableProperties prop = maprfs_.getTableProperties(fileStatus.getPath());
          if (prop.getAttr().getJson()) {
            tables.add(new Path(fileStatus.getPath().toUri().getPath()));
          }
        }
      }
    }
    return tables;
  }

  @Override
  public Table createTable(String tablePath)
      throws TableExistsException, DBException {
    return _createTable(new TableDescriptorImpl(new Path(tablePath)), null/*splitPoints*/, true);
  }

  @Override
  public Table createTable(Path tablePath) throws DBException {
    return _createTable(new TableDescriptorImpl(tablePath), null/*splitPoints*/, true);
  }

  @Override
  public Table createTable(TableDescriptor tableDesc)
      throws TableExistsException, DBException {
    return _createTable(tableDesc, null/*splitPoints*/, true);
  }

  //@Override
  public Table createTable(TableDescriptor tableDesc, Value[] splitPoints)
      throws TableExistsException, DBException {
    byte[][] encodedSplitKeys = null;
    if (splitPoints != null && splitPoints.length > 0) {
      encodedSplitKeys = new byte[splitPoints.length][];
      for (int i = 0; i < splitPoints.length; i++) {
        Preconditions.checkArgument(IdCodec.isSupportedType(splitPoints[i].getType()),
            splitPoints[i].getType() + " is not a supported type for split points.");
        encodedSplitKeys[i] = IdCodec.encodeAsBytes(splitPoints[i]);
      }
    }
    return _createTable(tableDesc, encodedSplitKeys, true);
  }

  @Override
  public Table createTable(TableDescriptor tableDesc, String[] splitPoints)
      throws TableExistsException, DBException {
    byte[][] encodedSplitKeys = null;
    if (splitPoints != null && splitPoints.length > 0) {
      encodedSplitKeys = new byte[splitPoints.length][];
      for (int i = 0; i < splitPoints.length; i++) {
        encodedSplitKeys[i] = IdCodec.encodeAsBytes(splitPoints[i]);
      }
    }
    return _createTable(tableDesc, encodedSplitKeys, true);
  }

  @Override
  public Table createTable(TableDescriptor tableDesc, ByteBuffer[] splitPoints)
      throws TableExistsException, DBException {
    byte[][] encodedSplitKeys = null;
    if (splitPoints != null && splitPoints.length > 0) {
      encodedSplitKeys = new byte[splitPoints.length][];
      for (int i = 0; i < splitPoints.length; i++) {
        encodedSplitKeys[i] = IdCodec.encodeAsBytes(splitPoints[i]);
      }
    }
    return _createTable(tableDesc, encodedSplitKeys, true);
  }

  @API.Internal
  public Table createTable(String tablePath, boolean openTable)
      throws TableExistsException, DBException {
    return _createTable(new TableDescriptorImpl(new Path(tablePath)), null/*splitPoints*/, openTable);
  }

  @API.Internal
  public Table createTable(TableDescriptor tableDesc, boolean openTable)
      throws TableExistsException, DBException {
    return _createTable(tableDesc, null/*splitPoints*/, openTable);
  }

  private Table _createTable(TableDescriptor tableDesc,
      byte[][] splitPoints, boolean openTable) throws DBException {
    try {
      if (maprfs_.exists(tableDesc.getPath())) {
        throw new TableExistsException(tableDesc.getPath());
      }

      if (splitPoints != null) {
        // sort the provided split points.
        Arrays.sort(splitPoints, Bytes.BYTES_COMPARATOR);
      }

      TableDescriptorImpl descImpl = ((TableDescriptorImpl)tableDesc);
      maprfs_.createTable(tableDesc.getPath(), null,
          descImpl.getTableAttr().build(), descImpl.getTableAces().build(),
          splitPoints, false, -1 /*auditValue*/); //FIXME: audit

      if (tableDesc.getFamilies().isEmpty()) {
        // create the default column family if none specified.
        maprfs_.createColumnFamily(tableDesc.getPath(),
            DEFAULT_FAMILY, FamilyDescriptorImpl.DEFAULT_COLUMN_FAMILY_ATTR);
      } else {
        int numFam = tableDesc.getNumFamilies();
        boolean isStream = ((TableDescriptorImpl)tableDesc).isStream();
        for (FamilyDescriptor family : tableDesc.getFamilies()) {
          // NOTE: There is a race with numFam check here. TODO - move checks
          // to server.
          if (!isStream && (numFam > 1) && (family.getTTL() != 0)) {
            deleteTable(tableDesc.getPath());
            throw new OpNotPermittedException("TLL cannot be set with " +
                                              "multiple column families.");
          }

          maprfs_.createColumnFamily(tableDesc.getPath(), family.getName(),
              getColumnFamilyAttr(false, family.getName(), family));
        }
      }

      return openTable ? new MapRDBTableImpl(tableDesc.getPath(), conf_) : null;
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "createTable()");
    }
  }

  @Override
  public void alterTable(TableDescriptor desc) throws TableNotFoundException, DBException {
    try {
      TableDescriptor oldDesc = getTableDescriptor(desc.getPath());
      if (!oldDesc.isBulkLoad() && desc.isBulkLoad()) {
        throw new OpNotPermittedException("Bulk load mode can only be turned off.");
      } else if (desc.isInsertionOrder() != oldDesc.isInsertionOrder()) {
        throw new OpNotPermittedException("Insertion order property can not be altered.");
      } else if (!desc.getFamilies().equals(oldDesc.getFamilies())) {
        throw new OpNotPermittedException("alterTable() can not be used to modify families of the table.");
      }
      TableDescriptorImpl descImpl = ((TableDescriptorImpl)desc);
      maprfs_.modifyTableAttr(desc.getPath(),
          descImpl.getTableAttr().clearJson().build(), descImpl.getTableAces().build());
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "alterTable()");
    }
  }

  //@Override
  public void addFamily(Path tablePath, FamilyDescriptor familyDesc)
      throws TableNotFoundException, FamilyExistsException, DBException {
    try {
      if (familyDesc.getTTL() != 0) {
        throw new OpNotPermittedException("TLL cannot be set with " +
                                          "multiple column families.");
      }

      maprfs_.createColumnFamily(tablePath,
          familyDesc.getName(), getColumnFamilyAttr(false, null, familyDesc));
    } catch (IOException e) {
      if (e instanceof ErrnoException && ((ErrnoException)e).getErrno() == Errno.EEXIST) {
        throw new FamilyExistsException(tablePath, familyDesc, e);
      } else {
        throw ExceptionHandler.handle(e, "addFamily()");
      }
    }
  }

  @Override
  public boolean deleteFamily(String tablePath, String familyName) throws TableNotFoundException,
      FamilyNotFoundException, DBException {
    return deleteFamily(new Path(tablePath), familyName);
  }

  @Override
  public boolean deleteFamily(Path tablePath, String familyName)
      throws TableNotFoundException, FamilyNotFoundException, DBException {
    if (familyName.equals(DEFAULT_FAMILY)) {
      throw new OpNotPermittedException(DEFAULT_FAMILY + " family cannot be deleted.");
    }

    try {
      maprfs_.deleteColumnFamily(tablePath, familyName);
      return true;
    } catch (IOException e) {
      if (e instanceof ErrnoException && ((ErrnoException)e).getErrno() == Errno.ENOENT) {
        return false;
      } else {
        throw ExceptionHandler.handle(e, "deleteFamily()");
      }
    }
  }

  @Override
  public void alterFamily(String tablePath, String familyName, FamilyDescriptor familyDesc)
      throws TableNotFoundException, FamilyNotFoundException, OpNotPermittedException, DBException {
    alterFamily(new Path(tablePath), familyName, familyDesc);
  }

  @Override
  public void alterFamily(Path tablePath, String familyName, FamilyDescriptor familyDesc)
      throws TableNotFoundException, FamilyNotFoundException, DBException {
    if (familyName.equals(DEFAULT_FAMILY) && familyDesc.hasName() &&
        !familyDesc.getName().equals(DEFAULT_FAMILY)) {
      throw new OpNotPermittedException(DEFAULT_FAMILY + " family name cannot be altered.");
    }

    TableDescriptor tableDesc = getTableDescriptor(tablePath);
    FamilyDescriptor oldFamilyDesc = tableDesc.getFamily(familyName);
    if (oldFamilyDesc == null) {
      throw new FamilyNotFoundException(tablePath, familyName);
    } else if (!oldFamilyDesc.getJsonFieldPath().equals(familyDesc.getJsonFieldPath())) {
      throw new OpNotPermittedException("A family's Json path can not be altered.");
    }

    try {
      maprfs_.modifyColumnFamily(tablePath, familyName,
          getColumnFamilyAttr(true, familyName, familyDesc));
    } catch (IOException e) {
      if (e instanceof ErrnoException) {
        switch (((ErrnoException)e).getErrno()) {
        case Errno.ENOENT:
          throw new FamilyNotFoundException(tablePath, familyName, e);
        case Errno.EEXIST:
          throw new FamilyExistsException(tablePath, familyDesc, e);
        default:
          // falls through
        }
      }
      throw ExceptionHandler.handle(e, "alterFamily()");
    }
  }

  @Override
  public boolean tableExists(String tablePath) throws DBException {
    return tableExists(new Path(tablePath));
  }

  @Override
  public boolean tableExists(Path tablePath) throws DBException {
    try {
      TableProperties tableProp = maprfs_.getTableProperties(tablePath);
      return tableProp.getAttr().getJson();
    } catch (FileNotFoundException e) {
      return false;
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "tableExists()");
    }
  }

  @Override
  public TableDescriptor getTableDescriptor(String tablePath) throws DBException {
    return getTableDescriptor(new Path(tablePath));
  }

  @Override
  public TableDescriptor getTableDescriptor(Path tablePath) throws DBException {
    try {
      TableProperties props = maprfs_.getTableProperties(tablePath);
      List<ColumnFamilyAttr> cfAttrs = maprfs_.listColumnFamily(tablePath, false);
      return new TableDescriptorImpl(tablePath, cfAttrs, props,
                                     props.getAttr().getInsertionOrder());
    } catch (IOException e) {
      throw new TableNotFoundException(tablePath, e);
    }
  }

  @Override
  public boolean deleteTable(String tablePath) throws DBException {
    return deleteTable(new Path(tablePath));
  }

  @Override
  public boolean deleteTable(Path tablePath) throws DBException {
    if (!tableExists(tablePath)) {
      return false;
    } else try {
      return maprfs_.delete(tablePath);
    } catch (IOException e) {
      throw ExceptionHandler.handle(e, "deleteTable()");
    }
  }

  /**
   * Creates a JSON DB Table as CopyTable destination
   * TODO JJS: Remove with Bug 19295 fixed
   */
  @Deprecated
  public static void createTableForCopy(String replicaPath, String srcPath,
      List<String> cfList, boolean isBulkload) throws DBException {

      try (AdminImpl admin = new AdminImpl()) {
        if (admin.tableExists(replicaPath)) {
          throw new TableExistsException(replicaPath);
        } else if (!admin.tableExists(srcPath)) {
          throw new TableNotFoundException(srcPath);
        }

        TableDescriptor desc = MapRDB.newTableDescriptor(replicaPath)
            .addFamily(MapRDB.newDefaultFamilyDescriptor())
            .setBulkLoad(isBulkload);

        MapRDBTableImpl srcTable = (MapRDBTableImpl) MapRDB.getTable(srcPath);
        List<ColumnFamilyAttr> cfAttrs = srcTable.maprTable().getMapRFS()
            .listColumnFamily(srcTable.maprTable().getTablePath(), false);
        //Construct the CF Name => JSON FieldPath map
        Map<FieldPath, Integer> srcIdPathMap = MapRDBTableImplHelper.getCFIdPathMap(cfAttrs);
        if (srcIdPathMap.size() == 0) {
          throw new DBException("Source table has no column families.");
        }
        for (Entry<FieldPath, Integer> entry : srcIdPathMap.entrySet()) {
          String cfName = srcTable.maprTable().getFamilyName(entry.getValue());
          if (!cfName.equals(DEFAULT_FAMILY)  //default should be created with the table
              && (cfList.size() == 0 || cfList.contains(cfName))) {
            desc.addFamily(MapRDB.newFamilyDescriptor(cfName, entry.getKey().asPathString()));
          }
        }
        srcTable.close();

        admin.createTable(desc, false);
      } catch (IOException e) {
        throw ExceptionHandler.handle(e, "createTableForCopy()");
      }
  }

  private ColumnFamilyAttr getColumnFamilyAttr(boolean forAlter,
      String familyName, FamilyDescriptor familyDesc) {
    FamilyDescriptorImpl familyDescImpl = (FamilyDescriptorImpl)familyDesc;
    SchemaFamily.Builder schemaFamily = familyDescImpl.getSchemaFamily();
    schemaFamily.clearId(); // clear the family id
    if (familyName == null || familyDesc.getName().equals(familyName)) {
      schemaFamily.clearName();
    }

    ColumnFamilyAttr.Builder builder = ColumnFamilyAttr.newBuilder();
    FieldPath jsonFamilyPath = familyDescImpl.getJsonFieldPath();
    if (!forAlter && jsonFamilyPath != null && !jsonFamilyPath.equals(FieldPath.EMPTY)) {
      builder.setJsonFamilyPath(jsonFamilyPath.asPathString());
    }

    return builder.setSchFamily(schemaFamily).build();
  }

}
