/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.exceptions;

import java.io.FileNotFoundException;
import java.io.IOException;

import org.apache.hadoop.security.AccessControlException;

import com.mapr.fs.ErrnoException;
import com.mapr.fs.PathNotFoundException;
import com.mapr.fs.RetryException;
import com.mapr.fs.StaleFileException;
import com.mapr.fs.jni.Errno;
import com.mapr.fs.jni.IOExceptionWithErrorCode;

public class ExceptionHandler {

  public static DBException handle(IOException e, String op) {
    DBException dbx;
    if (e instanceof ErrnoException) {
      ErrnoException ex = (ErrnoException)e;
      switch (ex.getErrno()) {
      case Errno.EACCES:
        dbx = new AccessDeniedException(op, e);
        break;
      case Errno.EPERM:
        dbx = new OpNotPermittedException(op, e);
        break;
      case Errno.EROFS:
        dbx = new ReadOnlyException(op, e);
        break;
      case Errno.EAGAIN:
        dbx = new DBRetryException(op, e);
        break;
      case Errno.ENOENT:
      case Errno.ESTALE:
        /* this should never be reached.
         *
         * if you notice a code causing this, replace the thrown
         * ErrnoException with StaleFileException or PathNotFoundException
         */
        dbx = new TableNotFoundException(String.format(
            "(unknown table, error code = %d)", ex.getErrno()), e);
        break;
      default:
        dbx = new DBException(String.format(
            "%s failed with err code = %d,", op, ex.getErrno()), e);
        break;
      }
    } else if (e instanceof StaleFileException) {
      dbx = new TableNotFoundException(((StaleFileException)e).getFilePath(), e);
    } else if (e instanceof RetryException) {
      dbx = new DBRetryException(((RetryException)e).getFilePath(), e);
    } else if (e instanceof PathNotFoundException) {
      dbx = new TableNotFoundException(((PathNotFoundException)e).getFilePath(), e);
    } else if (e instanceof FileNotFoundException) {
      dbx = new TableNotFoundException(e.getMessage(), e);
    } else if (e instanceof AccessControlException) {
      dbx = new AccessDeniedException(e.getMessage(), e);
    } else if (e instanceof IOExceptionWithErrorCode) {
      dbx = new DBException(e.getMessage(), e);
    } else {
      dbx = new DBException(String.format("%s failed.,", op), e);
    }

    return dbx;
  }

}
