/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.Arrays;

import org.ojai.store.QueryCondition;

import com.mapr.db.impl.ConditionImpl;
import com.mapr.db.util.ByteBufs;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

public class TableSplitInternal {

  private String tableName;
  private ConditionImpl cond;
  private String[] locations = null;
  /* length of the split */
  private long length ;

  public TableSplitInternal() {}

  public TableSplitInternal(TableSplitInternal splitInt) {
    tableName = splitInt.getTableName();
    cond = splitInt.getCondition();
    locations = splitInt.getLocations();
    length = splitInt.getLength();
  }

  public TableSplitInternal(String tabName,
                            QueryCondition c,
                            String[] locs,
                            long len) {
    tableName = tabName;
    cond = (ConditionImpl)c;
    locations = locs;
    length = len;
  }

  public String[] getLocations() {
    return  locations;
  }

  public void setLocations(String[] locs) {
    locations = locs;
  }

  public ConditionImpl getCondition() {
    return cond;
  }

  public long getLength() {
    return length;
  }

  public byte[] readField(final DataInput in) throws IOException {
    /* extract length of the field */
    /* create byte array, fill in */
    /* and return.                */
    int len = in.readInt();
    byte[] buff = new byte[len];
    in.readFully(buff);
    return buff;
  }

  public void readFields(DataInput in) throws IOException {
    tableName = new String(readField(in), "UTF-8");
    cond = (ConditionImpl)ConditionImpl.parseFrom(ByteBufs.wrap(readField(in)));
    locations = new String[] {new String(readField(in), "UTF-8")};
    length = in.readLong();
  }

  public void write(DataOutput out) throws IOException {
    byte[] tabName = tableName.getBytes(Charset.forName("UTF-8"));
    out.writeInt(tabName.length);
    out.write(tabName);
    ByteBuffer buff = cond.getDescriptor().getSerialized();
    int condBuffLength = buff.remaining();
    byte[] condBytes = new byte[condBuffLength];
    buff.get(condBytes);
    out.writeInt(condBuffLength);
    out.write(condBytes);
    byte[] locationsInBytes = concatLocationString().getBytes(Charset.forName("UTF-8"));
    out.writeInt(locationsInBytes.length);
    out.write(locationsInBytes);
    out.writeLong(length);
  }

  private String concatLocationString() {
    return Arrays.toString(locations);
  }


  /**
   * Returns the details about this instance as a string.
   *
   * @return The values of this instance as a string.
   * @see java.lang.Object#toString()
   */
  @Override
  public String toString() {
    StringBuffer buf = new StringBuffer();
    buf.append("data-size : " + length + "\n");
    buf.append("startRow : " + Bytes.toStringBinary(getStartRow()) + "\n");
    buf.append("stopRow : " + Bytes.toStringBinary(getStopRow()) + "\n");
    buf.append("condition : " + cond + "\n");
    buf.append("locations : " + "\n");
    for (String loc : locations) {
      buf.append("  " + loc + "\n");
    }
    return buf.toString();
  }

  public String getTableName() {
    return tableName;
  }

  public byte[] getStartRow() {
    return cond.getRowkeyRanges().get(0).getStartRow();
  }

  public byte[] getStopRow() {
    return cond.getRowkeyRanges().get(0).getStopRow();
  }

  @Override
  public boolean equals(Object o) {
    if (o == null || !(o instanceof TableSplitInternal)) {
      return false;
    }
    return tableName.equals(((TableSplitInternal)o).tableName) &&
      cond.equals(((TableSplitInternal)o).getCondition()) &&
      locations.equals(((TableSplitInternal)o).locations);

  }

    @Override
    public int hashCode() {
      int result = tableName != null ? tableName.hashCode() : 0;
      result = 31 * result + (cond != null ? cond.hashCode() : 0);
      result = 31 * result + (locations != null ? Arrays.hashCode(locations) : 0);
      return result;
    }
}
