/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db;

import java.util.List;

import org.apache.hadoop.fs.Path;

/**
 * A TableDescriptor contains the details about a table such as the descriptors of all the families,
 * the size at which a tablet split should occur, whether or not bulk load mode is turn on, and so on.
 */
public interface TableDescriptor extends Cloneable {

  /**
   * @return the Path of this table
   */
  public Path getPath();

  /**
   * Sets the path of this table in MapR FileSystem.
   *
   * @param tablePath the table path
   * @return {@code this} for chain invocation
   */
  public TableDescriptor setPath(String tablePath);

  /**
   * Sets the path of this table in MapR FileSystem.
   *
   * @param tablePath the table path
   * @return {@code this} for chain invocation
   */
  public TableDescriptor setPath(Path tablePath);

  /**
   * @return {@code true} if the map entries are stored in the order they are added, {@code false}
   *         if they are stored in the lexicographical order of their key name
   */
  public boolean isInsertionOrder();

  /**
   * Sets the map entries to be stored in the order they are added or not to be stored in order.
   *
   * @param insertionOrder {@code true} if the map entries are stored in the order they are added
   * @return {@code this} for chain invocation
   */
  public TableDescriptor setInsertionOrder(boolean insertionOrder);

  /**
   * @return {@code true} if the bulk load mode is set on this table
   */
  public boolean isBulkLoad();

  /**
   * Sets if the table is created with bulk load mode turned on. This can be set to true only when
   * creating the table and can only be set to false later.
   *
   * @param bulkLoad {@code true} if the bulk load mode is to be turned on
   * @return {@code this} for chain invocation
   */
  public TableDescriptor setBulkLoad(boolean bulkLoad);

  /**
   * @return {@code true} if a tablet should split automatically if it grows beyond a certain size.
   */
  public boolean isAutoSplit();

  /**
   * Sets the tablets to split automatically, or not, if they grow beyond a certain size.
   *
   * @param autoSplit {@code true} if the split should happen automatically
   * @return {@code this} for chain invocation
   */
  public TableDescriptor setAutoSplit(boolean autoSplit);

  /**
   * @return The size in mebibytes at which a tablet in this table will split.
   */
  public long getSplitSize();

  /**
   * Sets the size in mebibytes at which a tablet in this table will split.
   * Default value is 4096 (4 gibibytes).
   *
   * @param splitSizeMB split size in mebibytes
   * @return {@code this} for chain invocation
   */
  public TableDescriptor setSplitSize(long splitSizeMB);

  /**
   * Returns the FamilyDescriptor of the family with the specified name from this TableDescriptor
   * or {@code null} if a family by this name does not exist in the TableDescriptor.
   * Modifying the returned object does not modify this TableDescriptor.
   *
   * @param familyName the family name
   * @return the FamilyDescriptor
   */
  public FamilyDescriptor getFamily(String familyName);

  /**
   * Returns the list of family descriptors for this table. If the table does not have any column
   * family defined, an empty list is returned. Modifying the returned list or the FamilyDescriptor
   * in the list does not modify this TableDescriptor.
   *
   * @return the FamilyDescriptor list for this table
   */
  public List<FamilyDescriptor> getFamilies();

  /**
   * Returns the number of family descriptors for this table descriptor.
   */
  public int getNumFamilies();

  /**
   * Sets the list of families for this TableDescriptor. The first family must be named 'default'
   * and should have an empty JSON field path. This is only applicable when creating a table.
   *
   * @param families the list of families to add
   * @return {@code this} for chain invocation
   */
  public TableDescriptor setFamilies(List<FamilyDescriptor> families);

  /**
   * Appends a FamilyDescriptor to this TableDescriptor. This is only applicable when creating a table.
   *
   * @param family
   * @return {@code this} for chain invocation
   */
  public TableDescriptor addFamily(FamilyDescriptor family);


  /**
   * Creates and returns a copy of this TableDescriptor.
   * @return a clone of this TableDescriptor
   */
  public TableDescriptor clone();

}
