/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db;

import java.math.BigDecimal;
import java.nio.ByteBuffer;

import org.apache.hadoop.fs.Path;
import org.ojai.Document;
import org.ojai.DocumentStream;
import org.ojai.FieldPath;
import org.ojai.Value;
import org.ojai.annotation.API;
import org.ojai.store.DocumentMutation;
import org.ojai.store.DocumentStore;
import org.ojai.store.OpListener;
import org.ojai.store.QueryCondition;
import org.ojai.store.exceptions.DocumentExistsException;
import org.ojai.store.exceptions.DocumentNotFoundException;
import org.ojai.store.exceptions.MultiOpException;

import com.mapr.db.exceptions.AccessDeniedException;
import com.mapr.db.exceptions.DBException;
import com.mapr.db.exceptions.OpNotPermittedException;
import com.mapr.db.exceptions.ReadOnlyException;
import com.mapr.db.exceptions.TableNotFoundException;

@API.Public
public interface Table extends DocumentStore {
  /**
   * Following are different table options:<br/><br/>
   *
   * <b>{@code EXCLUDEID}:</b> If this option is set to true then rowkey
   * (field "_id") won't be returned in the result of a find or findById operation.
   * The default value of this option is false.<br/><br/>
   *
   * <b>{@code BUFFERWRITE}:</b>  If this option is set to false then any insert,
   * update, delete operation on the table would be synchronously sent in RPC
   * to server. If the option is true, then buffering of these operations on
   * the client are allowed and buffered updates are sent in batches to the server.
   * The buffering operation is optimized for performance as it reduces the number
   * of RPCs to the server.<br/>
   * The default value of this option is true, meaning data is buffered on the
   * client and flushed to the server periodically.<br/><br/>
   *
   * <b>{@code KEEPINSERTIONORDER}:</b>  If this option is set to true, then
   * documents are returned to the end user with the fields in the same order
   * that they were inserted into the document.<br/>
   * Suppose the document is inserted as { "name" : 1, "age" : 2, " }.
   * On the server we will store it in lexicographical sorted format as
   * { "age" : 2, "name" : 1}. However, if this option is not specified then on read
   * we will return it as { "age" : 2, "name" : 1}, but if this option is
   * specified, then on read we will return it as { "name" : 1, "age" : 2, " }.
   *
   * There is a little extra processing needed to maintain the fields in
   * the document in insertion order. Hence applications that don't care about
   * order should set this option to false. The default value of this option is
   * true.<br/><br/>
   *
   */
  public enum TableOption {
    EXCLUDEID,
    BUFFERWRITE,
    KEEPINSERTIONORDER,
  }

  /**
   * Returns just the name of the table (not the full path).
   */
  public String getName();

  /**
   * Returns the full path of the table.
   */
  public Path getPath();

  /**
   * @return The TableDescriptor for this table
   */
  public TableDescriptor getTableDescriptor();

  public Table setOption(TableOption option, boolean value);

  public Value getOption(TableOption option);

  /**
   * Flushes all the buffered update operations on the client for this table.
   * Throws an exception if the flush failed or if the flush of any buffered
   * operation resulted in an error.
   * @throws: DBException
   */
  @Override
  public void flush() throws DBException;

  /**
   * Reads the entire document for a given key.
   * NULL is returned if the document with a given id doesn't exist in the table.
   * Blocking operation.
   * Throws DBException if there is an error in getting the document.
   * @param id document id
   * @return document object
   * @throws DBException
   */
  public Document findById(String id) throws DBException;
  public Document findById(ByteBuffer id) throws DBException;
  public Document findById(Value id) throws DBException;

  /**
   * Reads the row and projects only the field paths requested. Blocking
   * operation.
   * NULL is returned if the document with the given id doesn't exist in the table.
   * Throws DBException if there is an error in getting the document.
   * If no path parameter is specified then it returns a full document for a
   * given row.
   * @param id document id
   * @param paths list of fields that should be returned in the read document
   * @return document object with the requested projection/conditions
   * @throws DBException
   */
  public Document findById(String id, String...paths) throws DBException;
  public Document findById(String id, FieldPath...paths) throws DBException;
  public Document findById(ByteBuffer id, String...paths) throws DBException;
  public Document findById(ByteBuffer id, FieldPath...paths) throws DBException;

  /**
   * Reads the row parts that match the condition and projects only the
   * field paths requested. Blocking operation.
   * If no path parameter is specified then it returns a full document for a
   * given row.
   * NULL is returned if the document with a given id doesn't exist in the table.
   * Throws DBException if there is an error in getting the document.
   *
   * @param id document id
   * @param paths list of fields that should be returned in the read document
   * @param c condition to check for the query
   * @return document object with the requested projection/conditions
   * @throws DBException
   */
  public Document findById(String id, QueryCondition c) throws DBException;
  public Document findById(ByteBuffer id, QueryCondition c) throws DBException;

  public Document findById(String id, QueryCondition c, String...paths) throws DBException;
  public Document findById(String id, QueryCondition c, FieldPath...paths) throws DBException;

  public Document findById(ByteBuffer id, QueryCondition c, String...paths) throws DBException;
  public Document findById(ByteBuffer id, QueryCondition c, FieldPath...paths) throws DBException;

  /**
   * Non-blocking read operation that gets called back on the listener object
   * on success (with the document) or failure (with the error).
   * @param listener object to be used as the callback
   * @param id document id
   * @return nothing
   */
  public void findById(OpListener listener, String id);
  public void findById(OpListener listener, ByteBuffer id);

  /**
   * Non-blocking read operation which gets called back on the listener object
   * on success or failure. Returns only the field paths requested.
   * If no path parameter is specified then it returns a full document for a
   * given row.
   * @param listener object to be used as the callback
   * @param id document id
   * @param paths list of fields that should be returned in the read document
   * @return nothing
   */
  public void findById(OpListener listener, String id, String...paths);
  public void findById(OpListener listener, String id, FieldPath...paths);
  public void findById(OpListener listener, ByteBuffer id, String...paths);
  public void findById(OpListener listener, ByteBuffer id, FieldPath...paths);

  /**
   * Non-blocking read operation that gets called back on the listener object
   * on success or failure. Returns only the row data only if the condition
   * matches for the given row. Also, returns only the field paths requested in
   * the document to the listener.
   *
   * If no path parameter is specified then it returns a full document for a
   * given row.
   *
   * QueryCondition is used to specify additional criteria for reading the row. For
   * example, I want to read from row with id as "abc" but only if the
   * field "age" > 50. Then "age" > 50 is specified as condition.
   *
   * @param listener object to be used as the callback
   * @param id document id
   * @param c condition to check for the query
   * @param paths list of fields that should be returned in the read document
   * @return nothing
   */
  public void findById(OpListener listener, String id, QueryCondition c);
  public void findById(OpListener listener, ByteBuffer id, QueryCondition c);

  public void findById(OpListener listener, String id, QueryCondition c, String... paths);
  public void findById(OpListener listener, String id, QueryCondition c, FieldPath...paths);

  public void findById(OpListener listener, ByteBuffer id, QueryCondition c, String... paths);
  public void findById(OpListener listener, ByteBuffer id, QueryCondition c, FieldPath...paths);

  /**
   * Returns a DocumentStream for all the rows in the table.
   * @return A DocumentStream that can be used to retrieve each row in the table.
   *         The DocumentStream must be closed after retrieving the documents
   * @throws DBException
   */
  @Override
  public DocumentStream find() throws DBException;

  /**
   * Returns a DocumentStream for all the rows in the table. Each Document will
   * contain only the paths that are specified in the argument. If a given
   * field doesn't exist in the row or the user doesn't have permissions to
   * access that field then that field won't be returned in the read document.
   *
   * If no path parameter is specified then it returns a full document for a
   * given row.
   *
   * @param paths list of fields that should be returned in the read document
   * @return A DocumentStream that can be used to requested paths.
   *         The DocumentStream must be closed after retrieving the documents.
   * @throws DBException
   */
  @Override
  public DocumentStream find(String...paths) throws DBException;
  @Override
  public DocumentStream find(FieldPath...paths) throws DBException;

  /**
   * Returns a DocumentStream for all the rows in the table that match the
   * condition.
   *
   * @param c condition to check for the query
   * @return A DocumentStream that can be used to get rows.
   *         The DocumentStream must be closed after retrieving the documents.
   * @throws TableNotFoundException when the table from which to get any row
   * does not exist
   */
   @Override
  public DocumentStream find(QueryCondition c) throws DBException;

  /**
   * Returns a document reader for all the rows in the table that match the
   * condition. Each Document will
   * contain only the paths that are specified in the argument. If a given
   * field doesn't exist in the row or the user doesn't have permissions to
   * access that field then that field won't be returned in the read document.
   *
   * If no path parameter is specified then it returns a full document for a
   * given row.
   *
   * @param c condition to check for the query
   * @param paths list of fields that should be returned in the read document
   * @return A DocumentStream that can be used to read docuemnts with requested
   *         paths. The DocumentStream must be closed after retrieving the documents
   * @throws DBException
   */
  @Override
  public DocumentStream find(QueryCondition c, String...paths) throws DBException;
  @Override
  public DocumentStream find(QueryCondition c, FieldPath...paths) throws DBException;

  /**
   * Inserts a new document in the table. The document id is either explicitly specified
   * as parameter "id" or it is implicitly specified as the field "_id" in the
   * passed document. If the document id is explicitly passed then the document should
   * not contain an "_id" field or its value should be same as the explicitly
   * specified id. Otherwise the operation will fail.
   *
   * If the row with the given key exists on the server then that document will
   * be overwritten by the new document.
   *
   * "fieldAsKey", when provided, will also be stored as the "_id" field in the
   * written document for the row. If "_id" already existed in the document, then
   * an error will be thrown. When reading the document back from the DB, the
   * key will be returned as usual as an "_id" field.
   *
   * @param r JSON document as the new value for the given row
   * @param id to be used as the key for the document
   * @param fieldAsKey document's field to be used as the key when an id is not
   *                     passed in and the document doesn't have an "_id" field
   * @throws DBException
   */
  @Override
  public void insertOrReplace(Document r) throws DBException;

  public void insertOrReplace(String id, Document r) throws DBException;
  public void insertOrReplace(ByteBuffer id, Document r) throws DBException;

  @Override
  public void insertOrReplace(Document r, FieldPath fieldAsKey) throws DBException;
  @Override
  public void insertOrReplace(Document r, String fieldAsKey) throws DBException;

  /**
   * Inserts a set of documents represented by the DocumentStream into the table.
   * This is a synchronous API and it won't return until all the documents
   * in the DocumentStream are written to the table or some error occurs while
   * writing the documents. Each document read from the DocumentStream must have a
   * field "_id" of type BINARY or UTF8-string; otherwise, the operation will fail.
   *
   * If there is an error in reading from the stream or in writing to
   * the table then a MultiOpException will be thrown that contains a list of
   * documents that failed to write to the table. Reading from a stream stops on the
   * first read error. If only write errors occur, the iterator will stop and the rest
   * of the documents will remain unconsumed.
   *
   * "fieldAsKey", when provided, will also be stored as the "_id" field in the
   * written document for the row. If "_id" already existed in the document, then
   * an error will be thrown. When reading the document back from the DB, the
   * key will be returned as usual as the "_id" field.
   *
   * If there is an error in reading from the stream or in writing to
   * the table then a MultiOpException will be thrown that contains a list of
   * documents that failed to write to the table. Reading from a stream stops on
   * the first read error. If only write errors occur, the iterator will stop
   * and the current list of failed document is returned in a multi op exception.
   * The untouched documents will remain in the DocumentStream.
   *
   * @param rs DocumentStream
   * @param fieldAsKey field from each document whose value is to be used as
   *                     the document key for insertion
   *
   * @throws MultiOpException that has a list of write-failed documents and
   *                          their errors
   */
  @Override
  public void insertOrReplace(DocumentStream rs) throws MultiOpException;
  @Override
  public void insertOrReplace(DocumentStream rs, FieldPath fieldAsKey)
      throws MultiOpException;
  @Override
  public void insertOrReplace(DocumentStream rs, String fieldAsKey)
      throws MultiOpException;

  /**
   * Applies a row mutation on the row identified by the row id. All the updates
   * specified by the mutation object are applied atomically, meaning either
   * all of the updates in the mutation are applied or none of them is applied.
   *
   * For example, the mutation object specified an increment on field
   * "address" by 5 and a set of field "zip" as "95134". If the "address" field
   * on the server is not of the numeric type then the increment operation will
   * fail. Because the set of field "zip" is also part of same mutation, that
   * operation will also not be applied on the server.
   *
   * @param id document id
   * @param m mutation object specifying the operation on paths of a row
   * @throws DBException
   */
  public void update(String id, DocumentMutation m) throws DBException;
  public void update(ByteBuffer id, DocumentMutation m) throws DBException;

  /**
   * Deletes a row with the given rowkey. This operation is successful even
   * when the row with the given id doesn't exist.
   *
   * "fieldAsKey", when provided, will also be stored as the "_id" field in the
   * written document for the row. If "_id" already existed in the document, then
   * an error will be thrown. When reading the document back from the DB, the
   * key will be returned as usual as "_id" field.
   *
   * @param id document id
   * @param r JSON document to be deleted
   * @param fieldAsKey document's field to be used as the key when an id is not
   *                     passed in and a document doesn't have an "_id" field
   * @throws DBException
   */
  public void delete(String id) throws DBException;
  public void delete(ByteBuffer id) throws DBException;
  @Override
  public void delete(Document r) throws DBException;

  @Override
  public void delete(Document r, FieldPath fieldAsKey) throws DBException;
  @Override
  public void delete(Document r, String fieldAsKey) throws DBException;

  /**
   * Deletes a set of documents from the table represented by the DocumentStream.
   * This is a synchronous API and it won't return until all the documents
   * in the DocumentStream are written to the table or some error occurs while
   * writing the documents. Each document read from the DocumentStream must have a
   * field "_id" of type BINARY or UTF8-string; otherwise the operation will fail.
   *
   * If there is an error in reading from the stream or in writing to
   * the table then a MultiOpException will be thrown that contains a list of
   * documents that failed to write to the table. Reading from a stream stops on
   * the first read error. If only write errors occur, the iterator will stop
   * and the current list of failed documents is returned in a multi op exception.
   * The untouched documents will remain in the DocumentStream.
   *
   * @param rs DocumentStream
   * @param fieldAsKey a field from each document whose value is to be used as
   *                     the document key for deletion
   *
   * @throws MultiOpException that has a list of write-failed documents and
   *                          their errors
   */
  @Override
  public void delete(DocumentStream rs) throws MultiOpException;
  @Override
  public void delete(DocumentStream rs, FieldPath fieldAsKey)
      throws MultiOpException;
  @Override
  public void delete(DocumentStream rs, String fieldAsKey)
      throws MultiOpException;

  /**
   * Inserts a row with the given rowkey. This operation is successful even
   * when the row with the given id doesn't exist.
   *
   * "fieldAsKey", when provided, will also be stored as the "_id" field in the
   * written document for the row. If "_id" already existed in the document, then
   * an error will be thrown. When reading the document back from the DB, the
   * key will be returned as usual as the "_id" field.
   *
   * Note that an insertOrReplace() operation would be more efficient than an
   * insert() call.
   *
   * @param r JSON document as the new value for the given row
   * @param id to be used as the key for the document
   * @param fieldAsKey document's field to be used as the key when the id is not
   *                     passed in and document doesn't have an "_id" field
   * @throws TableNotFoundException when a table does not exist to add this row
   * @throws AccessDeniedException when user credentials are a mismatch
   * @throws ReadOnlyException when a table is not accepting writes
   * @throws OpNotPermittedException when the server returned EPERM
   * @throws DocumentExistsException when a document with id already exists in table
   */
  public void insert(String id, Document r) throws DBException;
  public void insert(ByteBuffer id, Document r) throws DBException;

  @Override
  public void insert(Document r) throws DBException;
  @Override
  public void insert(Document r, FieldPath fieldAsKey) throws DBException;
  @Override
  public void insert(Document r, String fieldAsKey) throws DBException;

  /**
   * Inserts a set of documents represented by the DocumentStream into the table.
   * This is a synchronous API that won't return until all the documents
   * in the DocumentStream are written to the table or some error occurs while
   * writing the documents. Each document read from the DocumentStream must have a
   * field "_id" of type BINARY or UTF8-string; otherwise, the operation will
   * fail or it will be of the Document type.
   *
   * If a row with the given key exists on the server then it throws a document
   * exists exception, similar to the non-documentstream-based insert() API.
   *
   * If there is an error in reading from the stream or in writing to
   * the table then a MultiOpException will be thrown that contains a list of
   * documents that failed to write to the table. Reading from a stream stops on
   * the first read error. If only write errors occur, the iterator will stop
   * and the current list of failed documents is returned in a multi op exception.
   * The untouched documents will remain in the DocumentStream.
   *
   * @param rs DocumentStream
   * @param fieldAsKey a field from each document whose value is to be used as
   *                     the document key for deletion
   *
   * @throws MultiOpException that has a list of write-failed documents and
   *                          their errors
   */
  @Override
  public void insert(DocumentStream rs) throws MultiOpException;
  @Override
  public void insert(DocumentStream rs, FieldPath fieldAsKey)
      throws MultiOpException;
  @Override
  public void insert(DocumentStream rs, String fieldAsKey)
      throws MultiOpException;

  /**
   * Replaces a document in the table. The document id is either explicitly specified
   * as parameter "id" or it is implicitly specified as the field "_id" in the
   * passed document. If the document id is explicitly passed then the document should
   * not contain an "_id" field or its value should be the same as the explicitly
   * specified id; otherwise, the operation will  fail.
   *
   * If the row with the given key does not exist on the server then it will
   * throw a DocumentNotFoundException.
   *
   * "fieldAsKey", when provided, will also be stored as the "_id" field in the
   * written document for the row. If "_id" already existed in the document, then
   * an error will be thrown. When reading the document back from the DB, the
   * key would be returned as usual as an "_id" field.
   *
   * Note that an insertOrReplace() operation would be more efficient than an
   * replace() call.
   *
   * @param r JSON document as the new value for the given row
   * @param id to be used as the key for the document
   * @param fieldAsKey document's field to be used as the key when an id is not
   *                     passed in and the document doesn't have an "_id" field
   * @throws TableNotFoundException when a table does not exist to add this row
   * @throws AccessDeniedException when user credentials are a mismatch
   * @throws ReadOnlyException when a table is not accepting writes
   * @throws OpNotPermittedException when the server returns EPERM
   * @throws DocumentNotFoundException when a document with the id does not exist in table
   */
  public void replace(String id, Document r) throws DBException;
  public void replace(ByteBuffer id, Document r) throws DBException;

  @Override
  public void replace(Document r) throws DBException;
  @Override
  public void replace(Document r, FieldPath fieldAsKey) throws DBException;
  @Override
  public void replace(Document r, String fieldAsKey) throws DBException;

  /**
   * Replaces a set of documents represented by the DocumentStream into the table.
   * This is a synchronous API and it won't return until all the documents
   * in the DocumentStream are written to the table or some error occurs while
   * writing the documents. Each document read from the DocumentStream must have a
   * field "_id" of type BINARY or UTF8-string; otherwise, the operation will
   * fail or it will be of the Document type.
   *
   * If the row with the given key does not exist on the server then it throws,
   * a document not exists exception, similar to the non-DocumentStream based
   * replace() API.
   *
   * If there is an error in reading from the stream or in writing to
   * the table then a MultiOpException will be thrown that contains a list of
   * documents that failed to write to the table. Reading from a stream stops on
   * the first read error. If only write errors occur, the iterator will stop
   * and the current list of failed documents is returned in a multi op exception.
   * The untouched documents will remain in the DocumentStream.
   *
   * @param rs DocumentStream
   * @param fieldAsKey field from each document whose value is to be used as
   *                     the document key for deletion
   *
   * @throws MultiOpException that has a list of write-failed documents and
   *                          their errors
   */
  @Override
  public void replace(DocumentStream rs) throws MultiOpException;
  @Override
  public void replace(DocumentStream rs, FieldPath fieldAsKey)
      throws MultiOpException;
  @Override
  public void replace(DocumentStream rs, String fieldAsKey)
      throws MultiOpException;

  /**
   * Atomically applies an increment to a given field (in dot-separated notation)
   * of the given row id. If the field doesn't exist on the server
   * then it will be created with the type of the incremental value.
   * The increment operation can be applied on any of the numeric
   * types, such as byte, short, int, long, float, double, or decimal,
   * of a field. The operation will fail if the increment is applied to a
   * field that is of a non-numeric type.
   *
   * If an id doesn't exist, it gets created (similar to the insertOrReplace
   * behavior), and it is created, with the value of 'inc' parameter. The same
   * logic applies to intermittent paths in the path: they get created top to
   * bottom.
   *
   * If the type is different than the field in the operation, it fails.
   *
   * The increment operation won't change the type of an existing value stored in
   * the given field for the row. The resultant value of the field will be
   * truncated based on the original type of the field.
   *
   * For example, if a field 'score' is of type int and contains 60 and an
   * increment of double '5.675' is applied, then the resultant value of the
   * field will be 65 (65.675 will be truncated to 65).
   *
   * If the type to which the increment is applied is a byte, short, or int,
   * then it needs to use long as the operation.
   *
   * @param id document id
   * @param field the field name in dot separated notation
   * @param inc increment to apply to a field. Can be positive or negative.
   * @throws DBException
   */
  public void increment(String id, String field, byte inc) throws DBException;
  public void increment(ByteBuffer id, String field, byte inc) throws DBException;

  public void increment(String id, String field, short inc) throws DBException;
  public void increment(ByteBuffer id, String field, short inc) throws DBException;

  public void increment(String id, String field, int inc) throws DBException;
  public void increment(ByteBuffer id, String field, int inc) throws DBException;

  public void increment(String id, String field, long inc) throws DBException;
  public void increment(ByteBuffer id, String field, long inc) throws DBException;

  public void increment(String id, String field, float inc) throws DBException;
  public void increment(ByteBuffer id, String field, float inc) throws DBException;

  public void increment(String id, String field, double inc) throws DBException;
  public void increment(ByteBuffer id, String field, double inc) throws DBException;

  public void increment(String id, String field, BigDecimal inc) throws DBException;
  public void increment(ByteBuffer id, String field, BigDecimal inc) throws DBException;

  /**
   * Atomically evaluates the condition on a given row and if the
   * condition holds true for the row then a mutation is applied on the row.
   *
   * If an id doesn't exist, the function returns false (no exception is thrown).
   * If the mutation operation fails, it throws exception.
   *
   * @param id document id
   * @param condition the condition to evaluate on the row
   * @param m the mutation to apply on the row
   * @return True if the condition is true for the row; otherwise, false
   * @throws DBException if the condition passes but the mutation fails
   */
  public boolean checkAndMutate(String id,
      QueryCondition condition, DocumentMutation m) throws DBException;
  public boolean checkAndMutate(ByteBuffer id,
      QueryCondition condition, DocumentMutation m) throws DBException;

  /**
   * Atomically evaluates the condition on given row and if the
   * condition holds true for the row then it is atomically deleted.
   *
   * If id doesn't exist, returns false (no exception is thrown).
   * If the deletion operation fails, it throws an exception.
   *
   * @param id document id
   * @param condition the condition to evaluate on the row
   * @return True if the condition is valid for the row, otherwise false.
   * @throws DBException if the condition passes but the delete fails
   */
  public boolean checkAndDelete(String id, QueryCondition condition) throws DBException;
  public boolean checkAndDelete(ByteBuffer id, QueryCondition condition) throws DBException;

  /**
   * Atomically evaluates the condition on the given row and if the
   * condition holds true for the row then it atomically replaces the row
   * with the given document.
   *
   * If the id doesn't exist, the function returns false (no exception is thrown).
   * If the replace operation fails, it throws an exception.
   *
   * @param id document id
   * @param condition the condition to evaluate on the row
   * @param r the document to replace
   * @return True if the condition is true for the row; otherwise, false
   * @throws DBException if the condition passes but the replace fails
   */
  public boolean checkAndReplace(String id, QueryCondition condition, Document r) throws DBException;
  public boolean checkAndReplace(ByteBuffer id, QueryCondition condition, Document r) throws DBException;

  /**
   * @param _id the document id for which the {@code TabletInfo} is required
   * @return {@code TabletInfo} for the tablet of this table
   *         to which the specified key belongs
   */
  public TabletInfo getTabletInfo(String _id) throws DBException;

  /**
   * @param _id the document id for which the {@code TabletInfo} is required
   * @return {@code TabletInfo} for the tablet of this table
   *         to which the specified key belongs
   */
  public TabletInfo getTabletInfo(ByteBuffer _id) throws DBException;

  /**
   * @return {@code TabletInfo} for all current tablets of this table
   * @throws DBException
   */
  public TabletInfo[] getTabletInfos() throws DBException;

  /**
   * @param QueryCondition a condition object provided by the user or application
   * @return {@code TabletInfo} for all current tablets of this table, which
   *         has a start and end key matching the range specified by the
   *         condition object
   * @throws DBException
   */
  public TabletInfo[] getTabletInfos(QueryCondition c) throws DBException;

  /**
   * Overrides {@link AutoCloseable#close()} to avoid declaring a checked exception.
   */
  @Override
  void close() throws DBException;
}
