/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db;

import static com.mapr.db.impl.Constants.DEFAULT_INSERTION_ORDER;
import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;

import java.util.Map;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.ojai.Document;
import org.ojai.DocumentBuilder;
import org.ojai.DocumentReader;
import org.ojai.FieldPath;
import org.ojai.beans.BeanCodec;
import org.ojai.exceptions.DecodingException;
import org.ojai.json.Json;
import org.ojai.store.DocumentMutation;
import org.ojai.store.QueryCondition;

import com.mapr.db.exceptions.AccessDeniedException;
import com.mapr.db.exceptions.DBException;
import com.mapr.db.exceptions.TableExistsException;
import com.mapr.db.exceptions.TableNotFoundException;
import com.mapr.db.impl.AdminImpl;
import com.mapr.db.impl.ConditionImpl;
import com.mapr.db.impl.Constants;
import com.mapr.db.impl.FamilyDescriptorImpl;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.impl.TableDescriptorImpl;
import com.mapr.db.ojai.DBDocumentBuilder;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.MutationImpl;

public class MapRDB {

  /**
   * Test for the existence of Table at the given path.
   * This is a shortcut method for {@link Admin#tableExists(String)}.
   *
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to the user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in a default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @return {@code true} if the specified path exists and is a table
   */
  public static boolean tableExists(String tablePath) throws DBException {
    try (Admin admin = newAdmin()) {
      return admin.tableExists(tablePath);
    }
  }

  /**
   * Creates a JSON table at the specified path with a 'default' column family and default options.
   * This is a shortcut method for {@link Admin#createTable(String)}.
   *
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in a default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @return Table object for accessing the table
   * @throws TableExistsException if a table with the specified path already exists
   * @throws DBException if some DBException occurs
   */
  public static Table createTable(String tablePath) throws DBException {
    try (Admin admin = newAdmin()) {
      return admin.createTable(
          MapRDB.newTableDescriptor(tablePath)
          // TODO: remove the following line once default is handled on the server.
          .setInsertionOrder(DEFAULT_INSERTION_ORDER));
    }
  }

  /**
   * Deletes a Table.
   * This is a shortcut method for {@link Admin#deleteTable(String)}.
   *
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in a default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @throws AccessDeniedException  if the user does not have permission to delete
   *                                the table.
   * @return  {@code true} if and only if the table is successfully deleted;
   *          otherwise, {@code false}
   */
  public static boolean deleteTable(String tablePath) throws DBException {
    try (Admin admin = newAdmin()) {
      return admin.deleteTable(tablePath);
    }
  }

  /**
   * Returns an object for accessing the table having the given path name.
   * The table is opened with default table options.
   *
   * @param pathName table path. It can be specified as
   *               /tablepath to access a table in a default cluster or as
   *               /mapr/clustername/tabletpath to access a table in the given cluster
   * @return table object for accessing the table
   * @throws TableNotFoundException when the table doesn't exist
   */
  public static Table getTable(String pathName) throws DBException {
    return getTable(new Path(pathName));
  }

  /**
   * Returns an object for accessing the table having the given path name.
   * The table is opened with default table options.
   *
   * @param pathName table path. It can be specified as
   *               /tablepath to access a table in a default cluster or as
   *               /mapr/clustername/tabletpath to access a table in the given cluster
   * @return table object for accessing the table
   * @throws TableNotFoundException when the table doesn't exist
   */
  public static Table getTable(Path path) throws DBException {
    return new MapRDBTableImpl(path, new Configuration());
  }

  /**
   * Creates and returns an Admin instance with default configuration.
   * @return an instance of Admin interface
   */
  public static Admin newAdmin() throws DBException {
    return new AdminImpl();
  }

  /**
   * Creates and returns an Admin instance with specified configuration.
   * @return an instance of Admin interface
   */
  public static Admin newAdmin(Configuration c)
      throws DBException {
    return new AdminImpl(c);
  }

  /**
   * Creates and returns a TableDescriptor instance.
   * @return an instance of TableDescriptor interface
   */
  public static TableDescriptor newTableDescriptor() {
    return new TableDescriptorImpl();
  }

  /**
   * Creates and returns a TableDescriptor instance initialized with the specified path.
   * @return an instance of TableDescriptor interface
   */
  public static TableDescriptor newTableDescriptor(String tablePath) {
    return new TableDescriptorImpl(new Path(tablePath));
  }

  /**
   * Creates and returns a TableDescriptor instance initialized with the specified path.
   * @return an instance of TableDescriptor interface
   */
  public static TableDescriptor newTableDescriptor(Path tablePath) {
    return new TableDescriptorImpl(tablePath);
  }

  @Deprecated
  public static TableDescriptor newTableDescriptor(boolean insertionOrder) {
    return new TableDescriptorImpl(insertionOrder);
  }

  /**
   * Creates and returns a FamilyDescriptor instance.
   * @return an instance of FamilyDescriptor interface
   */
  public static FamilyDescriptor newFamilyDescriptor() {
    return new FamilyDescriptorImpl();
  }

  /**
   * Creates and returns a FamilyDescriptor instance for the "default" family.
   * @return an instance of FamilyDescriptor interface
   */
  public static FamilyDescriptor newDefaultFamilyDescriptor() {
    return new FamilyDescriptorImpl(Constants.DEFAULT_FAMILY, FieldPath.EMPTY);
  }

  /**
   * Creates and returns a FamilyDescriptor instance initialized with the specified family
   * name and its JSON path.
   * @return an instance of FamilyDescriptor interface
   */
  public static FamilyDescriptor newFamilyDescriptor(
      String familyName, String jsonPath) {
    return new FamilyDescriptorImpl(familyName, jsonPath);
  }

  /**
   * Creates and returns a FamilyDescriptor instance initialized with the specified family
   * name and its JSON path.
   * @return an instance of FamilyDescriptor interface
   */
  public static FamilyDescriptor newFamilyDescriptor(
      String familyName, FieldPath jsonPath) {
    return new FamilyDescriptorImpl(familyName, jsonPath);
  }

  /**
   * Creates and returns a new, empty instance of Document.
   */
  public static Document newDocument() {
    return new DBDocumentImpl();
  }

  /**
   * Returns a new instance of Document parsed from the specified JSON string.
   */
  public static Document newDocument(String jsonString)
      throws DecodingException {
    DocumentReader reader = Json.newDocumentReader(jsonString);
    DBDocumentBuilder writer = new DBDocumentBuilder();
    Json.writeReaderToBuilder(reader, writer);
    return writer.getDocument();
  }

  /**
   * Returns a new instance of Document constructed from the specified Map
   */
  public static Document newDocument(Map<String, Object> map)
      throws DecodingException {
    return (Document) KeyValueBuilder.initFrom(map);
  }

  /**
   * Returns a new instance of Document built from the specified Java bean.
   */
  public static Document newDocument(Object bean) throws DecodingException {
    return (Document) BeanCodec.decode(newDocumentBuilder(), bean);
  }

  /**
   * Returns the MapRDB storage implementation of DocumentBuilder interface.
   */
  public static DocumentBuilder newDocumentBuilder() {
    return new DBDocumentBuilder();
  }

  /**
   * Creates and returns a new DocumentMutation object.
   */
  public static DocumentMutation newMutation() {
    return new MutationImpl();
  }

  /**
   * Creates and returns a new QueryCondition object.
   */
  public static QueryCondition newCondition() {
    return new ConditionImpl();
  }

  /** This class cannot be instantiated. */
  private MapRDB() { }

}
