/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db;

import org.ojai.FieldPath;

/**
 * A FamilyDescriptor describes a family such as its JSON path in the document,
 * time to live, compression settings, and so on.
 *
 * It can be used as input when creating a table or adding/modifying a family.
 */
public interface FamilyDescriptor extends Cloneable {

  public enum Compression {
    /**
     * No compression.
     */
    None,

    /**
     * Inherit from the parent folder. Valid only while creating the family.
     */
    Inherited,

    /**
     * The LZ4 compression scheme. This is the default.
     */
    LZ4,

    /**
     * The LZF compression scheme.
     */
    LZF,

    /**
     * The ZLIB compression scheme. This offers highest compression at cost of speed.
     */
    ZLIB
  }

  /**
   * @return true if the descriptor has the name of the family name set
   */
  public boolean hasName();

  /**
   * @return the name of the family
   */
  public String getName();

  /**
   * Sets the name of this family.
   *
   * @param name family name
   * @return {@code this} for chain invocation
   */
  public FamilyDescriptor setName(String name);

  /**
   * @return the JSON field path of this family
   */
  public FieldPath getJsonFieldPath();

  /**
   * Sets the JSON field path of this family
   *
   * @param fieldPath
   * @return {@code this} for chain invocation
   */
  public FamilyDescriptor setJsonFieldPath(String fieldPath);

  /**
   * Sets the JSON field path of this family
   *
   * @param fieldPath
   * @return {@code this} for chain invocation
   */
  public FamilyDescriptor setJsonFieldPath(FieldPath fieldPath);

  /**
   * @return the Time-To-Live in seconds for the data stored in this family
   */
  public long getTTL();

  /**
   * Sets the Time-To-Live for the data stored in this family
   *
   * @param ttlInSec time to live in seconds
   * @return {@code this} for chain invocation
   */
  public FamilyDescriptor setTTL(long ttlInSec);

  /**
   * @return {@code true} if higher preference is given to the data stored in this family while
   *         evicting data from in-memory cache
   */
  public boolean isInMemory();

  /**
   * Sets whether preference is given to data of this family when data is evicted from the
   * in-memory cache.
   *
   * @param inMemory
   * @return {@code this} for chain invocation
   */
  public FamilyDescriptor setInMemory(boolean inMemory);

  /**
   * @return the {@link Compression} scheme used to store data in the family
   */
  public Compression getCompression();

  /**
   * Sets the {@link Compression} scheme used to store data in the family
   *
   * @param compression the Compression scheme
   * @return {@code this} for chain invocation
   */
  public FamilyDescriptor setCompression(Compression compression);

  /**
   * Creates and returns a copy of this FamilyDescriptor.
   * @return a clone of this FamilyDescriptor
   */
  public FamilyDescriptor clone();

}
