/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db;

import java.nio.ByteBuffer;
import java.util.List;

import org.apache.hadoop.fs.Path;

import com.mapr.db.exceptions.AccessDeniedException;
import com.mapr.db.exceptions.DBException;
import com.mapr.db.exceptions.FamilyNotFoundException;
import com.mapr.db.exceptions.OpNotPermittedException;
import com.mapr.db.exceptions.TableExistsException;
import com.mapr.db.exceptions.TableNotFoundException;

public interface Admin extends AutoCloseable {

  /**
   * List all tables in user's home directory.
   *
   * @return a list of table paths
   * @throws DBException
   */
  public List<Path> listTables() throws DBException;

  /**
   * List all tables in the specified folder and/or with the matching pattern.
   *
   * @param folderOrPattern the parent folder or pattern
   * @return a list of table paths
   * @throws DBException
   */
  public List<Path> listTables(String folderOrPattern) throws DBException;

  /**
   * List all tables in the specified folder and/or with the matching pattern.
   *
   * @param folderOrPattern the parent folder or pattern
   * @return a list of table paths
   * @throws DBException
   */
  public List<Path> listTables(Path folderOrPattern) throws DBException;

  /**
   * Creates a Json table at the specified path with 'default' column family and default options.
   *
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @return Table object for accessing the table
   * @throws TableExistsException if a table with the specified path already exits
   * @throws DBException if some DBException occurs
   */
  public Table createTable(String tablePath) throws TableExistsException, DBException;

  /**
   * Creates a Json table at the specified path with 'default' column family and default options.
   *
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @return Table object for accessing the table
   * @throws TableExistsException if a table with the specified path already exits
   * @throws DBException if some DBException occurs
   */
  public Table createTable(Path tablePath) throws TableExistsException, DBException;

  /**
   * Creates a Json table using the specified TableDescriptor.
   *
   * @param tableDesc the TableDescriptor containing various attributes of this table
   * @return a Table object connected to the newly created table
   * @throws TableExistsException if the table already exist at the specified path
   * @throws DBException if an DBException occurs
   */
  public Table createTable(TableDescriptor tableDesc) throws TableExistsException, DBException;

  /**
   * Creates a pre-split Json table using the specified TableDescriptor and the split points
   * as the tablet boundaries. The number of tablets is 1 + the number of split points.
   * The split points must be in sorted order.
   *
   * @param tableDesc the TableDescriptor containing various attributes of this table
   * @param splitPoints the split points for this table
   * @return
   * @throws TableExistsException
   * @throws DBException
   */
  public Table createTable(TableDescriptor tableDesc, String[] splitPoints)
      throws TableExistsException, DBException;

  /**
   * Creates a pre-split JSON table using the specified TableDescriptor and the split points
   * as the tablet boundaries. The number of tablets is 1 + the number of split points.
   * The split points must be in sorted order.
   *
   * @param tableDesc the TableDescriptor containing various attributes of this table
   * @param splitPoints the split points for this table
   * @return
   * @throws TableExistsException
   * @throws DBException
   */
  public Table createTable(TableDescriptor tableDesc, ByteBuffer[] splitPoints)
      throws TableExistsException, DBException;

  /**
   * Modify the attributes of an existing table. This API cannot be used to modify
   * (add/remove/alter) families of the table. Please use {@linkplain #addFamily(Path, FamilyDescriptor)},
   * {@linkplain #deleteFamily(Path, String)}, and {@linkplain #alterFamily(Path, String, FamilyDescriptor)}
   * for this purpose.
   *
   * @param tableDesc the TableDescriptor containing various attributes of this table
   * @throws TableNotFoundException
   * @throws OpNotPermittedException if an attribute is requested to be modified but is not allowed
   * @throws DBException
   */
  public void alterTable(TableDescriptor tableDesc)
      throws TableNotFoundException, OpNotPermittedException, DBException;

  /**
   * Deletes a family from a Json table.
   *
   * @param tablePath
   * @param familyName
   * @throws DBException
   * @return  {@code true} if and only if the family is successfully deleted;
   *          {@code false} otherwise
   */
  public boolean deleteFamily(String tablePath, String familyName)
      throws TableNotFoundException, FamilyNotFoundException, DBException;
  
  /**
   * Deletes a family from a Json table.
   *
   * @param tablePath
   * @param familyName
   * @throws DBException
   * @return  {@code true} if and only if the family is successfully deleted;
   *          {@code false} otherwise
   */
  public boolean deleteFamily(Path tablePath, String familyName)
      throws TableNotFoundException, FamilyNotFoundException, DBException;
  

  /**
   *
   * @param tablePath
   * @param familyName
   * @param familyDesc
   * @throws TableNotFoundException
   * @throws FamilyNotFoundException
   * @throws OpNotPermittedException if an attribute is requested to be modified but is not allowed
   * @throws DBException
   */
  public void alterFamily(String tablePath, String familyName, FamilyDescriptor familyDesc)
      throws TableNotFoundException, FamilyNotFoundException, OpNotPermittedException, DBException;
  /**
   *
   * @param tablePath
   * @param familyName
   * @param familyDesc
   * @throws TableNotFoundException
   * @throws FamilyNotFoundException
   * @throws OpNotPermittedException if an attribute is requested to be modified but is not allowed
   * @throws DBException
   */
  public void alterFamily(Path tablePath, String familyName, FamilyDescriptor familyDesc)
      throws TableNotFoundException, FamilyNotFoundException, OpNotPermittedException, DBException;


  /**
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @return {@code true} if the specified path exists and is a table
   */
  public boolean tableExists(String tablePath) throws DBException ;

  /**
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @return {@code true} if the specified path exists and is a table
   */
  public boolean tableExists(Path tablePath) throws DBException;

  /**
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @return TableDescriptor for the specified table
   */
  public TableDescriptor getTableDescriptor(String tablePath) throws DBException;

  /**
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @return TableDescriptor for the specified table
   */
  public TableDescriptor getTableDescriptor(Path tablePath) throws DBException;

  /**
   * Deletes a Table.
   *
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @throws AccessDeniedException  if the user does not have permission to delete
   *                                the table.
   * @return  {@code true} if and only if the table is successfully deleted;
   *          otherwise, {@code false}
   */
  public boolean deleteTable(String tablePath) throws DBException;

  /**
   * Deletes a Table.
   *
   * @param tablePath The table path.
   *        The table path can be either absolute or relative to user's home directory.
   *        The absolute path can be specified as "<i>/&lt;tablepath></i>" for tables in default
   *        cluster or as "/mapr/&lt;<i>clustername&gt;</i>/&lt;<i>tablepath&gt;</i>" for tables
   *        in the specified cluster
   *
   * @throws AccessDeniedException  If the user does not have permission to delete
   *                                the table.
   * @return  {@code true} if and only if the table is successfully deleted;
   *          otherwise, {@code false}
   */
  public boolean deleteTable(Path tablePath) throws DBException;

  /**
   * Override {@link AutoCloseable#close()} to avoid declaring a checked exception.
   */
  void close() throws DBException;

}
