/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.spark.writers

import java.nio.ByteBuffer

import org.ojai.Document
import com.mapr.db.rowcol.DBValueBuilderImpl
import com.mapr.db.spark.condition.DBQueryCondition
import org.ojai.store.{DocumentMutation, DocumentStore}

private[spark] case class TableInsertOrReplaceWriter(@transient table: DocumentStore) extends Writer {

  def write(doc: Document, key: ByteBuffer) = {
    write(doc,DBValueBuilderImpl.KeyValueBuilder.initFrom(key))
  }

  def write(doc: Document, key: String) = {
    write(doc, DBValueBuilderImpl.KeyValueBuilder.initFrom(key))
  }

  def write(doc: Document, key: org.ojai.Value) = {
    table.insertOrReplace(doc.setId(key))
  }

  def close() = {
    table.flush()
    table.close()
  }
}

private[spark] case class TableInsertWriter(@transient table: DocumentStore) extends Writer {

  def write(doc: Document, key: ByteBuffer) = {
    write(doc, DBValueBuilderImpl.KeyValueBuilder.initFrom(key))
  }

  def write(doc: Document, key: String) = {
    write(doc, DBValueBuilderImpl.KeyValueBuilder.initFrom(key))
  }

  def write(doc: Document, key: org.ojai.Value) = {
    table.insert(doc.setId(key))
  }

  def close() = {
    table.flush()
    table.close()
  }
}

private[spark] case class TableUpdateWriter(@transient table: DocumentStore) {

  def write(mutation: DocumentMutation, key: ByteBuffer) : Unit = {
    write(mutation, DBValueBuilderImpl.KeyValueBuilder.initFrom(key))
  }

  def write(mutation: DocumentMutation, key: String) : Unit = {
    write(mutation, DBValueBuilderImpl.KeyValueBuilder.initFrom(key))
  }

  def write(mutation: DocumentMutation, key: org.ojai.Value) : Unit = {
    table.update(key, mutation)
  }

  def close() = {
    table.flush()
    table.close()
  }
}

private[spark] case class TableCheckAndMutateWriter(@transient table: DocumentStore) {

  def write(mutation: DocumentMutation, queryCondition: DBQueryCondition, key: ByteBuffer) : Unit = {
    write(mutation, queryCondition, DBValueBuilderImpl.KeyValueBuilder.initFrom(key))
  }

  def write(mutation: DocumentMutation, queryCondition: DBQueryCondition, key: String) : Unit = {
    write(mutation, queryCondition, DBValueBuilderImpl.KeyValueBuilder.initFrom(key))
  }

  def write(mutation: DocumentMutation, queryCondition: DBQueryCondition, key: org.ojai.Value) : Unit = {
    table.checkAndMutate(key, queryCondition.condition, mutation)
  }

  def close() = {
    table.flush()
    table.close()
  }
}