/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.shell.ops;

import static com.mapr.db.Table.TableOption.BUFFERWRITE;
import static com.mapr.db.Table.TableOption.EXCLUDEID;
import static com.mapr.db.Table.TableOption.KEEPINSERTIONORDER;

import java.io.IOException;
import java.util.List;
import java.util.Map.Entry;
import java.util.concurrent.ExecutionException;

import org.apache.hadoop.fs.Path;
import org.ojai.Document;
import org.ojai.DocumentConstants;
import org.ojai.DocumentStream;
import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.json.JsonOptions;
import org.ojai.store.DocumentMutation;
import org.ojai.store.DocumentStore;
import org.ojai.store.QueryCondition;
import org.ojai.store.QueryCondition.Op;

import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.util.concurrent.UncheckedExecutionException;
import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.exceptions.DBException;
import com.mapr.db.exceptions.ExceptionHandler;
import com.mapr.db.impl.AdminImpl;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.shell.ShellSession;
import com.mapr.fs.MapRFileSystem;
import com.mapr.streams.Streams;

public class TableOps {

  private static final String DOC_ID_NOT_FOUND = "A document id was neither provided nor found in the document.";

  private final AdminImpl admin;
  private final MapRFileSystem fs_;
  private final ShellSession session_;
  private final LoadingCache<Path, Table> tableCache;
  private final LoadingCache<String, DocumentStore> streamCache;

  public TableOps(ShellSession session_) throws IOException {
    fs_ = session_.getFS();
    admin = new AdminImpl(fs_);
    this.session_ = session_;
    tableCache = CacheBuilder.newBuilder()
        .maximumSize(50)
        .removalListener(new RemovalListener<Path, Table>() {
            @Override
            public void onRemoval(RemovalNotification<Path, Table> notification) {
              try {
                notification.getValue().close();
              } catch (Exception e) {}
            }})
          .build(new CacheLoader<Path, Table>() {
              @Override
              public Table load(Path tablePath) throws Exception {
                return MapRDB.getTable(tablePath);
              }
          });
    streamCache = CacheBuilder.newBuilder()
        .maximumSize(50)
        .removalListener(new RemovalListener<String, DocumentStore>() {
          @Override
          public void onRemoval(RemovalNotification<String, DocumentStore> notification) {
            try {
              notification.getValue().close();
            } catch (Exception e) {}
          }})
        .build(new CacheLoader<String, DocumentStore>() {
          @Override
          public DocumentStore load(String tablePath) throws Exception {
            return Streams.getMessageStore(tablePath);
          }
        });

  }

  public void create(String tablePath) throws IOException {
    putTable(tablePath, admin.createTable(tablePath));
    System.out.printf("Table %s created.\n", tablePath);
  }

  public void deleteTable(String tablePath) throws IOException {
    if (!admin.deleteTable(tablePath)) {
      System.out.printf("Unable to delete table: '%s'."
          + " Check if the path exists and is a table.\n", tablePath);
    } else {
      removeTable(tablePath);
      System.out.printf("Table %s deleted.\n", tablePath);
    }
  }

  public void findById(JsonOptions jsonOptions, String tablePath, String id) throws IOException {
    Table store = getTable(tablePath);
    if (!session_.isRawStreamScan() && isStream(store)) {
      // disable findById, except in raw mode, on Streams until we support it
      throw new IllegalArgumentException("'findbyid' is currently not supported with Streams");
    }

    Document readDocument = store.findById(id);
    String json = readDocument.asJsonString(jsonOptions);
    System.out.println(json);
  }

  public void find(JsonOptions jsonOptions, String tablePath, String fromId, String toId, String limitStr) throws IOException {
    QueryCondition c = null;
    if (fromId != null || toId != null) {
      c = MapRDB.newCondition();
      if (fromId != null && toId != null) {
        c.and();
      }
      if (fromId != null) {
        c.is(DocumentConstants.ID_FIELD, Op.GREATER_OR_EQUAL, fromId);
      }
      if (toId != null) {
        c.is(DocumentConstants.ID_FIELD, Op.LESS, toId);
      }
      if (fromId != null && toId != null) {
        c.close();
      }
      c.build();
    }
    long count = 0, limit = Long.MAX_VALUE;
    if (limitStr != null) {
      limit = Long.parseLong(limitStr);
    }

    DocumentStream stream = null;
    DocumentStore store = getTable(tablePath);
    if (!session_.isRawStreamScan() && isStream(store)) {
      if (c != null) {
        // disable find with start and stop id on Streams until we support it
        throw new IllegalArgumentException("'--fromid' or '--toid' are not supported with Streams");
      }
      stream = getStream(tablePath).find();
    } else {
      stream = store.find(c);
    }
    for (Document document : stream) {
      count++;
      String json = document.asJsonString(jsonOptions);
      System.out.printf("%s\n", json);
      if (count == limit) {
        break;
      }
    }
    System.out.printf("%d document(s) found.\n", count);
  }

  public void insert(String tablePath, String docId, String jsonStr)
      throws IOException {
    Document document = MapRDB.newDocument(jsonStr);
    String inDocId = null;
    boolean inDoc = false;
    if (docId == null) {
      inDocId = document.getString(DocumentConstants.ID_FIELD);
      inDoc = true;
    } else {
      inDocId = document.getString(DocumentConstants.ID_FIELD);
      if ((inDocId != null) && !inDocId.equals(docId)) {
        throw new UnsupportedOperationException(
            "ERROR: '_id' field in the json string '" + inDocId + "' should " +
            " match input parameter '" + docId +
            "' or only one of them can be provided.");
      }
      inDocId = docId;
    }

    if (inDocId == null) {
      throw new UnsupportedOperationException(
            "ERROR: Expected an '_id' field either in the json string or as " +
            "an input '--id' parameter.");
    }

    Table t = noStream(getTable(tablePath), "insert");
    if (inDoc) {
      // remove _id field and add back to make getId() working.
      document.delete(DocumentConstants.ID_FIELD);
      document.setId(inDocId);
      t.insertOrReplace(document);
    } else {
      t.insertOrReplace(inDocId, document);
    }

    System.out.printf("Document with id: \"%s\" inserted.\n", inDocId);
  }

  public void deleteRow(String tablePath, String docId) throws IOException {
    Table t = noStream(getTable(tablePath), "delete");
    t.delete(docId);
    System.out.printf("Document with id: \"%s\" deleted.\n", docId);
  }

  public void update(String tablePath, String docId, String keyValues)
      throws IOException {
    Document document = MapRDB.newDocument(keyValues);
    if (docId == null) {
      docId = document.getIdString();
      Preconditions.checkArgument(docId != null, DOC_ID_NOT_FOUND);
      document.delete(DocumentConstants.ID_FIELD);
    }
    // build the document mutation
    DocumentMutation mutation = MapRDB.newMutation();
    for (Entry<String, Value> entry : document) {
      Value v = entry.getValue();
      if (v.getType() == Type.MAP || v.getType() == Type.ARRAY) {
        throw new UnsupportedOperationException(
            "ERROR: update of " + v.getType() + " type is not supported yet.");
      }
      mutation.setOrReplace(entry.getKey(), v);
    }

    noStream(getTable(tablePath), "update").update(docId, mutation);
    System.out.printf("Document with id: \"%s\" updated.\n", docId);
  }

  public void exists(String tablePath) throws IOException {
    System.out.println(admin.tableExists(tablePath));
  }

  public void list(String parent) throws IOException {
    List<Path> tables = admin.listTables(parent);
    if (tables == null) {
      System.out.printf("No tables found.\n");
      return;
    }
    for (Path path : tables) {
      System.out.println(path);
    }
    System.out.printf("%d table(s) found.\n", tables.size());
  }

  public void desc(String tablePath) throws IOException {
    System.out.println(admin.getTableDescriptor(tablePath));
  }

  /*
   * Private methods
   */

  private void removeTable(String tablePath) throws IOException {
    tableCache.invalidate(fs_.makeAbsolute(new Path(tablePath)));
  }

  private void putTable(String tablePath, Table table) throws IOException {
    tableCache.put(fs_.makeAbsolute(new Path(tablePath)), table);
  }

  private Table noStream(Table table, String op) {
    if (isStream(table)) {
      throw new UnsupportedOperationException("'" + op +
          "' is not supported with a MapR Stream.");
    }
    return table;
  }

  private boolean isStream(DocumentStore store) {
    return (store instanceof MapRDBTableImpl)
        && ((MapRDBTableImpl)store).isStream();
  }

  private Table getTable(String tablePath) throws IOException {
    try {
      Table table = tableCache.get(fs_.makeAbsolute(new Path(tablePath)))
          .setOption(EXCLUDEID, session_.getTableOptions().isExcludeId())
          .setOption(BUFFERWRITE, session_.getTableOptions().isBufferWrite());
      if (!isStream(table)) {
        table.setOption(KEEPINSERTIONORDER, session_.getTableOptions().isKeepInsertionOrder());
      }
      return table;
    } catch (ExecutionException | UncheckedExecutionException e) {
      Throwable cause = e.getCause();
      if (cause instanceof DBException) {
        throw (DBException)cause;
      }
      throw ExceptionHandler.handle((cause instanceof IOException)
        ? (IOException)cause : new IOException(cause), "getTable()");
    }
  }

  private DocumentStore getStream(String streamPath) throws IOException {
    try {
      return streamCache.get(streamPath);
    } catch (ExecutionException | UncheckedExecutionException e) {
      Throwable cause = e.getCause();
      if (cause instanceof DBException) {
        throw (DBException)cause;
      }
      throw ExceptionHandler.handle((cause instanceof IOException)
          ? (IOException)cause : new IOException(cause), "getStream()");
    }
  }

}
