/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.shell.ops;

import java.io.BufferedReader;
import java.io.IOException;

import jline.internal.InputStreamReader;

import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FsShell;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathExistsException;
import org.apache.hadoop.util.ToolRunner;

import com.mapr.db.shell.ShellSession;
import com.mapr.fs.MapRFileStatus;

public class FileOps {
  private final ShellSession session_;
  
  private String oldpwd = null;

  public FileOps(ShellSession session) throws IOException {
    session_ = session;
  }

  public String getWorkingDirectory() {
    return session_.getCWD().toUri().getPath();
  }

  public void setWorkingDirectory(String pathStr) throws IOException {
    Path path = null;
    if (pathStr != null) {
      if ("-".equals(pathStr)) {
        if (oldpwd != null) {
          pathStr = oldpwd;
        } else {
          throw new IllegalStateException("OLDPWD not set");
        }
      }
      path = new Path(pathStr);
      MapRFileStatus stat = session_.getFS().getMapRFileStatus(path);
      if (!stat.isDirectory()) {
        throw new IllegalArgumentException(pathStr + " is not a directory");
      }
    }
    oldpwd = session_.getCWD().toUri().getPath();
    session_.getFS().setWorkingDirectory(path == null ? session_.getHomeDir() : path);
  }

  public void listFiles(String pathStr) throws Exception {
    FsShell cli = new FsShell(session_.getFS().getConf());
    try {
      if (pathStr == null) {
        pathStr = getWorkingDirectory();
      }
      ToolRunner.run(cli, new String[] {"-ls", pathStr});
    } finally {
      cli.close();
    }
  }

  public void pwd() {
    System.out.println(getWorkingDirectory());
  }

  public void cat(String pathStr) throws IOException {
    Path path = new Path(pathStr);
    MapRFileStatus stat = session_.getFS().getMapRFileStatus(path);
    if (!stat.isRegular()) {
      throw new IllegalArgumentException(pathStr + " is not a regular file");
    }

    try (FSDataInputStream in = session_.getFS().open(path);
         BufferedReader reader = new BufferedReader(new InputStreamReader(in, "UTF-8"))) {
      String line;
      while ((line = reader.readLine()) != null) {
        System.out.println(line);
      }
    }
  }

  public void mkdir(String pathStr) throws IOException {
    Path path = new Path(pathStr);
    if (session_.getFS().exists(path)) {
      throw new PathExistsException(pathStr);
    }
    session_.getFS().mkdirs(path);
  }

}
