/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.shell.commands;

import java.io.IOException;

import org.ojai.json.JsonOptions;
import org.springframework.shell.core.annotation.CliAvailabilityIndicator;
import org.springframework.shell.core.annotation.CliCommand;
import org.springframework.shell.core.annotation.CliOption;
import org.springframework.stereotype.Component;

import com.mapr.db.shell.ops.TableOps;

@Component
public class TableCommands extends ShellCommands {

  private static final String HELP_PARAM_ID = "Document Id.";
  private static final String HELP_PARAM_FROM_ID = "Document Id to start from (inclusive)";
  private static final String HELP_PARAM_TO_ID = "Document Id to stop at (exclusive)";
  private static final String HELP_PARAM_1 = "A path and/or a file pattern.";
  private static final String HELP_PARAM_2 = "Json document to insert or replace.";
  private static final String HELP_PARAM_3 = "Json document containing field and their values.";
  private static final String HELP_PARAM_4 = "Maximum number of documents to return.";
  private static final String HELP_PARAM_5 = "Enables/disables printing with extended Type Tags.";
  private static final String HELP_PARAM_6 = "Enables/disables pretty printing of the document.";

  private static final String TABLE_PATH = "Table Path.";
  private static final String UPDATE_HELP = "Update field in a single document.";
  private static final String DELETE_HELP = "Delete a document from the table.";
  private static final String FINDBYID_HELP = "Retrieves a single document from the table.\nExample: findbyid /idsTable --id 001";
  private static final String FIND_HELP = "Retrieves one or more documents from the table.\nExample: find /idsTable\n         find /idsTable --fromid 001 --toid 00a --limit 32";
  private static final String INSERT_HELP = "Inserts or replaces a document into the table.\nExample: insert /idsTable --value '{\"_id\":\"001\" , \"first_name\":\"John\", \"last_name\":\"Doe\"}'\n         insert /idsTable --id 002 --value '{\"first_name\":\"Jane\", \"last_name\":\"Dane\"}'\n";
  private static final String DROP_HELP = "Deletes a json table.";
  private static final String EXISTS_HELP = "Returns true if the table exists.";
  private static final String CREATE_HELP = "Create a json table at the given path.";
  private static final String DESC_HELP = "Describes the properties of a table.";
  private static final String LIST_HELP = "Lists all tables in a folder or matching the specified pattern.";
  private TableOps tableOps_;

  public TableCommands() throws IOException {
    tableOps_ = new TableOps(session_);
  }

  @CliAvailabilityIndicator({"find", "insert", "delete", "exists", "create", "drop", "list", "desc"})
  public boolean isDMLAvailable() {
    return session_.inTableMode();
  }

  @CliCommand(value = "exists", help = EXISTS_HELP)
  public String exists(
      @CliOption(key = {"", "path", "p"}, mandatory = true, help = TABLE_PATH)
      final String path) throws IOException {
    tableOps_.exists(path);
    return null;
  }

  @CliCommand(value = "list", help = LIST_HELP)
  public String list(
      @CliOption(key = {"", "p", "patternOrPath"}, help = HELP_PARAM_1)
      final String patternOrPath) throws IOException {
    tableOps_.list(patternOrPath);
    return null;
  }

  @CliCommand(value = "desc", help = DESC_HELP)
  public String desc(
      @CliOption(key = {"", "t", "table"}, mandatory = true, help = TABLE_PATH)
      final String tablePath) throws IOException {
    tableOps_.desc(tablePath);
    return null;
  }

  @CliCommand(value = "create", help = CREATE_HELP)
  public String create(
      @CliOption(key = {"", "t", "table"}, mandatory = true, help = TABLE_PATH)
      final String tablePath) throws IOException {
    tableOps_.create(tablePath);
    return null;
  }

  @CliCommand(value = "drop", help = DROP_HELP)
  public String drop(
      @CliOption(key = {"", "t", "table"}, mandatory = true, help = TABLE_PATH)
      final String tablePath) throws IOException {
    tableOps_.deleteTable(tablePath);
    return null;
  }

  @CliCommand(value = "insert", help = INSERT_HELP)
  public String insert(
      @CliOption(key = {"", "t", "table"}, mandatory = true, help = TABLE_PATH)
      final String tablePath,
      @CliOption(key = {"id"}, help = HELP_PARAM_ID)
      final String _id,
      @CliOption(key = {"v", "value"}, mandatory = true, help = HELP_PARAM_2)
      final String jsonStr) throws IOException {
    tableOps_.insert(tablePath, _id, jsonStr);
    return null;
  }

  @CliCommand(value = "findbyid", help = FINDBYID_HELP)
  public String get(
      @CliOption(key = {"", "table", "t"}, mandatory = true, help = TABLE_PATH)
      final String tablePath,
      @CliOption(key = {"id"}, mandatory = true, help = HELP_PARAM_ID)
      final String _id,
      @CliOption(key = {"withtags", "withTags"}, help = HELP_PARAM_5)
      final String withTags,
      @CliOption(key = {"pretty"}, help = HELP_PARAM_6)
      final String pretty) throws IOException {
    JsonOptions jsonOptions = getCmdJsonOptions(withTags, pretty);
    tableOps_.findById(jsonOptions, tablePath, _id);
    return null;
  }

  @CliCommand(value = "find", help = FIND_HELP)
  public String find(
      @CliOption(key = {"", "t", "table"}, mandatory = true, help = TABLE_PATH)
      final String tablePath,
      @CliOption(key = {"id"}, help = HELP_PARAM_ID)
      final String _id,
      @CliOption(key = {"fromid"}, help = HELP_PARAM_FROM_ID)
      final String fromId,
      @CliOption(key = {"toid"}, help = HELP_PARAM_TO_ID)
      final String toId,
      @CliOption(key = {"limit"}, help = HELP_PARAM_4)
      final String limit,
      @CliOption(key = {"withtags", "withTags"}, help = HELP_PARAM_5)
      final String withTags,
      @CliOption(key = {"pretty"}, help = HELP_PARAM_6)
      final String pretty) throws IOException {
    JsonOptions jsonOptions = getCmdJsonOptions(withTags, pretty);
    if (_id == null) {
      tableOps_.find(jsonOptions, tablePath, fromId, toId, limit);
    } else {
      tableOps_.findById(jsonOptions, tablePath, _id);
    }
    return null;
  }

  @CliCommand(value = "delete", help = DELETE_HELP)
  public String delete(
      @CliOption(key = {"", "t", "table"}, mandatory = true, help = TABLE_PATH)
      final String tablePath,
      @CliOption(key = {"id"}, mandatory = true, help = HELP_PARAM_ID)
      final String id) throws IOException {
    tableOps_.deleteRow(tablePath, id);
    return null;
  }

  public String update(
      @CliOption(key = {"", "t", "table"}, mandatory = true, help = TABLE_PATH)
      final String tablePath,
      @CliOption(key = {"id"}, mandatory = true, help = HELP_PARAM_ID)
      final String _id,
      @CliOption(key = {"v", "value"}, mandatory = true, help = HELP_PARAM_3)
      final String kvs) throws IOException {
    tableOps_.update(tablePath, _id, kvs);
    return null;
  }

}
