/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.shell.commands;

import java.io.IOException;

import org.springframework.shell.core.annotation.CliAvailabilityIndicator;
import org.springframework.shell.core.annotation.CliCommand;
import org.springframework.shell.core.annotation.CliOption;
import org.springframework.stereotype.Component;

@Component
public class SessionCommnds extends ShellCommands {

  protected SessionCommnds() throws IOException {
  }

  @CliAvailabilityIndicator({"debug"})
  public boolean available() {
    return true; // session commands are always available
  }

  @CliAvailabilityIndicator({"mode"})
  public boolean modeAvailable() {
    return false; // currenly we only operate in table mode
  }

  @CliCommand(value = "debug", help = "Sets/shows the debug mode.")
  public String debug(
      @CliOption(key = {""}, help = "Set the debug mode to on/off")
      final String mode,
      @CliOption(key = {"rawstream"}, specifiedDefaultValue = "true", hidden = true /* undocumented */)
      final Boolean rawStream) throws IOException {
    StringBuilder sb = new StringBuilder();
    if (mode != null) {
      session_.setDebug("on".equalsIgnoreCase(mode));
    }
    sb.append("Debug Mode: ").append(session_.isDebug() ? "On" : "Off");
    if (rawStream != null) {
      session_.setRawStreamScan(rawStream);
      sb.append("\nStream Scan Mode: ").append(session_.isRawStreamScan() ? "Raw" : "Messages");
    }
    return sb.toString();
  }

  public String mode(
      @CliOption(key = {""}, help = "Sets/shows the active mode (global|files|tables|stream) of the session.")
      final String mode) throws IOException {
    if (mode != null) {
      session_.setMode(mode);
    }
    return "Session mode is set to " + session_.getMode() + ".";
  }

  @CliCommand(value = "jsonoptions", help = "Sets/shows the Json output options.")
  public String jsonoption(
      @CliOption(key = {"pretty"},
                 help = "Sets/clears the pretty printing mode.",
                 specifiedDefaultValue = "true")
      final Boolean pretty,
      @CliOption(key = {"withtags", "withTags"},
                 help = "Enables/disables printing with extended Type Tags.",
                 specifiedDefaultValue = "true")
      final Boolean withTags) throws IOException {
    if (pretty != null) {
      session_.getJsonOptions().setPretty(pretty);
    }
    if (withTags != null) {
      session_.getJsonOptions().setWithTags(withTags);
    }
    return String.valueOf(session_.getJsonOptions());
  }

  @CliCommand(value = "tableoptions", help = "Sets/shows the MapR-DB Table access options.")
  public String tableoption(
      @CliOption(key = {"bufferwrite", "bufferWrite"},
      help = "Sets/clears the buffered write mode when inserting documents.",
      specifiedDefaultValue = "false")
      final Boolean bufferwrite,
      @CliOption(key = {"excludeid", "excludeId"},
      help = "Enables/disables exclusion of '_id' field when printing documents.",
      specifiedDefaultValue = "false")
      final Boolean excludeId,
      @CliOption(key = {"insertionorder", "insertionOrder"},
      help = "Enables/disables insertion order when printing documents.",
      specifiedDefaultValue = "true")
      final Boolean insertionOrder) throws IOException {
    if (bufferwrite != null) {
      session_.getTableOptions().setBufferWrite(bufferwrite);
    }
    if (excludeId != null) {
      session_.getTableOptions().setExcludeId(excludeId);
    }
    if (insertionOrder != null) {
      session_.getTableOptions().setKeepInsertionOrder(insertionOrder);
    }
    return String.valueOf(session_.getTableOptions());
  }

}
