/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.shell;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.ojai.Document;
import org.ojai.json.JsonOptions;
import org.springframework.shell.ShellException;

import com.mapr.db.MapRDB;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.jni.MapRConstants;

public class ShellSession {
  private static final String DEBUG_ENABLED = "enabled";
  private static final String STREAM_RAW_SCAN = "STREAM_RAW_SCAN";

  final private MapRFileSystem fs_;
  final private Path homeDir;
  final private JsonOptions jsonOptions_;
  final private TableOptions tableOptions_;

  private SessionMode mode_ = SessionMode.TABLES;

  private Document debugOptions = MapRDB.newDocument()
      .set(DEBUG_ENABLED, false)
      .set(STREAM_RAW_SCAN, false);

  private ShellSession() throws IOException {
    try {
      fs_ = new MapRFileSystem();
      fs_.initialize(new URI(MapRConstants.MAPRFS_PREFIX), new Configuration());
      homeDir = fs_.getHomeDirectory();
      jsonOptions_ = new JsonOptions().withTags();
      tableOptions_ = new TableOptions();
    } catch (URISyntaxException e) {
      throw new RuntimeException(e); /* not happening */
    }
  }

  public Path getHomeDir() {
    return homeDir;
  }

  public MapRFileSystem getFS() {
    return fs_;
  }

  private static volatile ShellSession INSTACE;
  public static ShellSession getSession() {
    if (INSTACE == null) {
      synchronized (ShellSession.class) {
        if (INSTACE == null) {
          try {
            INSTACE = new ShellSession();
          } catch (IOException e) {
            throw new ShellException(e);
          }
        }
      }
    }
    return INSTACE;
  }

  public Document getDebugOptions() {
    return debugOptions;
  }

  public boolean isDebug() {
    return debugOptions.getBoolean(DEBUG_ENABLED);
  }

  public ShellSession setDebug(boolean isDebug) {
    debugOptions.set(DEBUG_ENABLED, isDebug);
    return this;
  }

  public boolean isRawStreamScan() {
    return debugOptions.getBoolean(STREAM_RAW_SCAN);
  }

  public ShellSession setRawStreamScan(boolean value) {
    debugOptions.set(STREAM_RAW_SCAN, value);
    return this;
  }

  public ShellSession setMode(String mode) {
    mode_ = SessionMode.valueOf(mode.toUpperCase());
    return this;
  }

  public ShellSession setMode(SessionMode mode) {
    mode_ = mode;
    return this;
  }

  public SessionMode getMode() {
    return mode_;
  }

  public boolean inGlobalMode() {
    return mode_ == SessionMode.GLOBAL;
  }

  public boolean inTableMode() {
    return mode_ == SessionMode.TABLES;
  }

  public Path getCWD() {
    return fs_.getWorkingDirectory();
  }

  public JsonOptions getJsonOptions() {
    return jsonOptions_;
  }

  public TableOptions getTableOptions() {
    return tableOptions_;
  }

}
