package com.mapr.db.mapreduce.test;

import static org.junit.Assert.assertEquals;

import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.SequenceFile;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.ojai.Document;

import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.impl.IdCodec;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.impl.MapRDBTableImpl.TablePrivateOption;
import com.mapr.db.mapreduce.impl.ByteBufWritableComparable;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.SequenceFileRowColCodec;
import com.mapr.db.tests.utils.DBTests;
import com.mapr.fs.MapRFileSystem;
import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

@Category(ClusterTest.class)
public class TestMCFEncodingDecoding extends BaseTest {

  private SequenceFile.Writer writer = null;
  FileSystem fs = null;
  String filePath = DBTests.getTestRoot() + "/fpath";
  String tableName = "mcftab1";
  String tabname2 = "mcftab2";
  Configuration conf = null;

  private void setUp() throws Exception {

    Map<String, String> cfPath = new HashMap<String, String>();

    cfPath.put("cf1", "a.b");
    cfPath.put("cf2", "a.c");

    DBTests.deleteTables(tableName);

    Table myTab = DBTests.createOrReplaceTable(tableName, cfPath);
    Document d = MapRDB.newDocument();

    d.set("a.b", 1234)
     .set("a.c.x1", true)
     .set("a.c.x2", "eureka")
     .set("a.e", "pqr");

    myTab.insertOrReplace("k1", d);
    myTab.insertOrReplace("k2", d);
    myTab.flush();
    myTab.close();

  }

  public void setUp2() throws Exception {
    setUp();
    Map<String, String> cfPath = new HashMap<String, String>();

    cfPath.put("cf1", "a.b");
    cfPath.put("cf2", "a.c");

    DBTests.deleteTables(tabname2);

    Table myTab = DBTests.createOrReplaceTable(tabname2, cfPath);
    myTab.close();
  }

  private void setUp3() throws Exception {

    Map<String, String> cfPath = new HashMap<String, String>();

    cfPath.put("cf1", "a.b");
    cfPath.put("cf2", "a.c");
    cfPath.put("cf3", "x.y");

    DBTests.deleteTables(tableName);

    Table myTab = DBTests.createOrReplaceTable(tableName, cfPath);
    Document d = MapRDB.newDocument();

    /* No data for CF1 and CF3 */
    d.set("a.c.x1", true)
     .set("a.c.x2", "eureka")
     .set("a.e", "pqr");

    myTab.insertOrReplace("k1", d);
    myTab.insertOrReplace("k2", d);
    myTab.flush();
    myTab.close();

  }

  private void createWriter() throws Exception {
    conf = new Configuration();
    fs = MapRFileSystem.get(conf);
    writer = SequenceFile.createWriter(fs, conf, new Path(filePath), ByteBufWritableComparable.class, ByteBufWritableComparable.class);
  }

  @Test
  public void testMCFEncoding() throws Exception {
    setUp();
    createWriter();
    /* retrieve records */
    MapRDBTableImpl myTab = (MapRDBTableImpl)DBTests.getTable(tableName);
    myTab.setPrivateOption(TablePrivateOption.PRESERVE_TIMESTAMP, true);
    Iterator<Document> iter = myTab.find().iterator();
    while(iter.hasNext()) {
      Document d = iter.next();
      ByteBuffer id = IdCodec.encode(d.getId());
      ByteBuffer encodedDocument = SequenceFileRowColCodec.encode(d);
      writer.append(new ByteBufWritableComparable(id), new ByteBufWritableComparable(encodedDocument));
    }

    writer.close();

    SequenceFile.Reader reader = new SequenceFile.Reader(fs, new Path(filePath), conf);

    ByteBufWritableComparable key = new ByteBufWritableComparable();
    ByteBufWritableComparable value = new ByteBufWritableComparable();

    while (reader.next(key, value)) {
      Document docValue = SequenceFileRowColCodec.decode(value.getByteBuf());
      assertEquals(true, docValue.getBoolean("a.c.x1"));
    }

    reader.close();
  }


  @Test
  public void test2() throws Exception {
    setUp2();
    MapRDBTableImpl myTab = (MapRDBTableImpl)DBTests.getTable(tableName);
    myTab.setPrivateOption(TablePrivateOption.PRESERVE_TIMESTAMP, true);



    MapRDBTableImpl tab2 = (MapRDBTableImpl)DBTests.getTable(tableName);
    Iterator<Document> iter = myTab.find().iterator();
    while(iter.hasNext()) {
      Document d = iter.next();
      tab2.insertOrReplace(d.getId(), d);
    }

    tab2.flush();
    myTab.close();
    tab2.close();
  }

  @Test
  public void testEmptyCFData() throws Exception {
    setUp3();
    createWriter();
    MapRDBTableImpl myTab = (MapRDBTableImpl)DBTests.getTable(tableName);
    myTab.setPrivateOption(TablePrivateOption.PRESERVE_TIMESTAMP, true);

    Iterator<Document> iter = myTab.find().iterator();
    while(iter.hasNext()) {
      Document d = iter.next();
      ByteBuffer id = IdCodec.encode(d.getId());
      ByteBuffer encodedDocument = SequenceFileRowColCodec.encode(d);
      writer.append(new ByteBufWritableComparable(id), new ByteBufWritableComparable(encodedDocument));
    }

    writer.close();

    SequenceFile.Reader reader = new SequenceFile.Reader(fs, new Path(filePath), conf);

    ByteBufWritableComparable key = new ByteBufWritableComparable();
    ByteBufWritableComparable value = new ByteBufWritableComparable();

    while (reader.next(key, value)) {
      DBDocumentImpl docValue = (DBDocumentImpl)SequenceFileRowColCodec.decode(value.getByteBuf());
      assertEquals(2,docValue.getCachedBuffers().size());
    }

  }

}