/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.db.mapreduce.test;

import static com.mapr.db.rowcol.DBValueBuilderImpl.KeyValueBuilder;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.ArrayList;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.ojai.Document;
import org.ojai.Value.Type;
import org.ojai.store.DocumentMutation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.Admin;
import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.Table.TableOption;
import com.mapr.db.TableDescriptor;
import com.mapr.db.FamilyDescriptor;
import com.mapr.db.mapreduce.BulkLoadRecordWriter;
import com.mapr.db.tests.utils.DBTests;
import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

@Category(ClusterTest.class)
public class TestBulkLoadRecordWriter extends BaseTest {
  private static final String TEST_BULKLOAD_JSON_TABLE = "/tmp/test_bulkload_json";
  private static final Logger _logger = LoggerFactory.getLogger(TestBulkLoadRecordWriter.class);
  private static final String TABLE_NAME = "testtable-TestBulkLoadRecordWriter";
  private static Table table;
  private static Admin admin;
  private static final Path tablePath = DBTests.getTablePath(TABLE_NAME);
  private static final int THR_SLEEP_INT = 5000; //milliseconds

  @Before
  public void setupTestBulkLoadRecordWriter() throws Exception {
    try {
      table = DBTests.createOrReplaceTable(TABLE_NAME);
      table.setOption(TableOption.BUFFERWRITE, false);
      table.setOption(TableOption.EXCLUDEID, true);
      admin = MapRDB.newAdmin();
    } catch (Exception e) {
      _logger.info("Failed to create a new table '{}'.", tablePath);
      throw e;
    }
  }

  @After
  public void cleanupTestBulkLoadRecordWriter() throws Exception {
    if (table != null) {
      table.close();
      DBTests.admin().deleteTable(tablePath);
    }
  }

  /*
   * Note: Table and Column Families are created using maprcli for
   * BulkLoad Record Delete test using the following specifications
   * Table Path: /tmp/test_bulkload_json
   * Col Families:
   * default = ""
   * cf1 => defaultchild4.family1
   * cf2 => defaultchild4.family2
   * cf3 => Family1
   * cf4 => Family1.child4.family1
   * cf5 => Family1.child4.family2

     maprcli table create -path /tmp/test_bulkload_json  -tabletype json
     maprcli table info -path /tmp/test_bulkload_json
     maprcli table cf create -path /tmp/test_bulkload_json -cfname default
     maprcli table cf create -path /tmp/test_bulkload_json -cfname cf1 -jsonpath defaultchild4.family1
     maprcli table cf create -path /tmp/test_bulkload_json -cfname cf2 defaultchild4.family2
     maprcli table cf create -path /tmp/test_bulkload_json -cfname cf2 -jsonpath defaultchild4.family2
     maprcli table cf create -path /tmp/test_bulkload_json -cfname cf3 -jsonpath Family1
     maprcli table cf create -path /tmp/test_bulkload_json -cfname cf4 -jsonpath Family1.child4.family1
     maprcli table cf create -path /tmp/test_bulkload_json -cfname cf5 -jsonpath Family1.child4.family2
     maprcli table cf list -path /tmp/test_bulkload_json
   */
  private void createBulkLoadTable(boolean fullBulkLoad)
    throws IOException, Exception {

    TableDescriptor desc = MapRDB.newTableDescriptor(TEST_BULKLOAD_JSON_TABLE)
        .setBulkLoad(fullBulkLoad);
    desc.addFamily(MapRDB.newDefaultFamilyDescriptor());
    desc.addFamily(MapRDB.newFamilyDescriptor("cf1", "defaultchild4.family1"));
    desc.addFamily(MapRDB.newFamilyDescriptor("cf2", "defaultchild4.family2"));
    desc.addFamily(MapRDB.newFamilyDescriptor("cf3", "Family1"));
    desc.addFamily(MapRDB.newFamilyDescriptor("cf4", "Family1.child4.family1"));
    desc.addFamily(MapRDB.newFamilyDescriptor("cf5", "Family1.child4.family2"));

    admin.deleteTable(TEST_BULKLOAD_JSON_TABLE);
    _logger.info("Creating table /tmp/test_bulkload_json with bulkload = " + fullBulkLoad);
    admin.createTable(desc);
  }

  private void deleteBulkLoadTable() {
    admin.deleteTable(TEST_BULKLOAD_JSON_TABLE);
  }

  private void clearBulkLoadTableFlag() throws IOException, Exception {
    TableDescriptor desc = admin.getTableDescriptor(TEST_BULKLOAD_JSON_TABLE);
    desc.setBulkLoad(false);
    admin.alterTable(desc);
  }

  @Test
  public void testBulkLoadRecordDeleteIncremental()
    throws IOException, Exception {
    createBulkLoadTable(false /*fullBulkLoad*/);
    testBulkLoadRecordDeleteCommon();
    deleteBulkLoadTable();
  }

  @Test
  public void testBulkLoadRecordDeleteFull()
    throws IOException, Exception {
    createBulkLoadTable(true /*fullBulkLoad*/);
    testBulkLoadRecordDeleteCommon();
    deleteBulkLoadTable();
  }

  private void testBulkLoadRecordDeleteCommon()
    throws IOException, Exception {
    assert(table != null);
    Path tabPath = new Path(TEST_BULKLOAD_JSON_TABLE);
    try {
      table = MapRDB.getTable(tabPath);
      table.setOption(TableOption.BUFFERWRITE, false);
      table.setOption(TableOption.EXCLUDEID, true);
    } catch (Exception e) {
      _logger.info("Failed to get table. Create table '{}'.", tabPath);
      throw e;
    }

    //delete  "Family1.child4.family2.child1"
    //delete  "defaultchild4.family2.child1"
    //delete  "defaultchild4.family2.child2"
    //replace "defaultchild4.family2"
    Configuration conf = new Configuration();
    BulkLoadRecordWriter bulkLoadWriter = new BulkLoadRecordWriter(conf, tabPath);

    String rowStr = new String("r1");

    Document rec = MapRDB.newDocument();
    rec.set("defaultchild1", 10)
    .set("defaultchild2", true)
    .set("defaultchild3", "Value")
    .set("defaultchild4.child1", "child1")
    .set("defaultchild4.child2", "child2")
    .set("defaultchild4.family1", "family1")
    .set("defaultchild4.family2", 25000)
    .set("Family1.child1", 10)
    .set("Family1.child2", true)
    .set("Family1.child3", "Value")
    .set("Family1.child4.child1", "child1")
    .set("Family1.child4.child2", "child2")
    .set("Family1.child4.family2.child2", 50000);

    bulkLoadWriter.write(KeyValueBuilder.initFrom(rowStr), rec);
    bulkLoadWriter.close(null);

    clearBulkLoadTableFlag();

    Document readRec = (Document) table.findById(rowStr);
    assertNotNull(readRec);

    assertEquals(10, readRec.getInt("defaultchild1"));
    assertEquals(true, readRec.getBoolean("defaultchild2"));
    assertEquals("Value", readRec.getString("defaultchild3"));
    assertEquals("child1", readRec.getString("defaultchild4.child1"));
    assertEquals("child2", readRec.getString("defaultchild4.child2"));
    assertEquals("family1", readRec.getString("defaultchild4.family1"));
    assertEquals(25000, readRec.getInt("defaultchild4.family2"));

    assertNull(readRec.getValue("defaultchild4.family2.child1"));
    assertNull(readRec.getValue("defaultchild4.family2.child2"));
    assertNull(readRec.getValue("Family1.child4.family2.child1"));
  }

  @Test
 /*
  * Note: Table and Column Families need to be created manually.
  * Table Path: /tmp/test_bulkload_full
  */
  public void testFullBulkLoadRecordWriterMCF()
    throws IOException, Exception {
    String tabName = "/tmp/test_bulkload_full_mcf";
    Path tabPath = new Path(tabName);

    List<FamilyDescriptor> families = new ArrayList<FamilyDescriptor>();

    TableDescriptor desc = MapRDB.newTableDescriptor(tabName)
        .setBulkLoad(true);
    desc.addFamily(MapRDB.newDefaultFamilyDescriptor());
    desc.addFamily(MapRDB.newFamilyDescriptor("af1", "person.address.office"));
    desc.addFamily(MapRDB.newFamilyDescriptor("cf1", "person.address.home"));
    desc.addFamily(MapRDB.newFamilyDescriptor("ef1", "person.salary"));
    desc.addFamily(MapRDB.newFamilyDescriptor("gf1", "person.contact"));

    _logger.info("Creating table: " + tabName);
    admin.deleteTable(tabName);
    admin.createTable(desc);

    try {
      _logger.info("Getting table: " + tabPath.toString());
      table = MapRDB.getTable(tabPath);
      table.setOption(TableOption.BUFFERWRITE, false);
      table.setOption(TableOption.EXCLUDEID, true);
    } catch (Exception e) {
      _logger.info("Failed to get table. Create table '{}'.", tabPath);
      throw e;
    }
    assert(table != null);
    Configuration conf = new Configuration();
    BulkLoadRecordWriter bulkLoadWriter = new BulkLoadRecordWriter(conf, tabPath);

    // Write mutation
    DocumentMutation mutation = MapRDB.newMutation();
    mutation.delete("person.address.office.street")
    .setOrReplace("person.address.home.city", "San Jose");

    String row3Str = "r3";
    ByteBuffer rowBytes3 = ByteBuffer.wrap(row3Str.getBytes());
    bulkLoadWriter.write(KeyValueBuilder.initFrom(row3Str), mutation);

    //Forced flush is not necessary, just for testing
    bulkLoadWriter.write(null, (DocumentMutation)null);

    admin.deleteTable(tabName);
 }

  @Test
 /*
  * Note: Table and Column Families need to be created manually.
  * Table Path: /tmp/test_bulkload_full
  */
  public void testFullBulkLoadRecordWriter()
    throws IOException, Exception {
    String tabName = "/tmp/test_bulkload_full";
    Path tabPath = new Path(tabName);
    TableDescriptor desc = MapRDB.newTableDescriptor(tabName)
        .setBulkLoad(true);
    desc.addFamily(MapRDB.newDefaultFamilyDescriptor());

    _logger.info("Creating table: " + tabName);
    admin.deleteTable(tabName);
    admin.createTable(tabName);

    try {
      _logger.info("Getting table: " + tabPath.toString());
      table = MapRDB.getTable(tabPath);
      table.setOption(TableOption.BUFFERWRITE, false);
      table.setOption(TableOption.EXCLUDEID, true);
    } catch (Exception e) {
      _logger.info("Failed to get table. Create table '{}'.", tabPath);
      throw e;
    }
    assert(table != null);
    Configuration conf = new Configuration();
    BulkLoadRecordWriter bulkLoadWriter = new BulkLoadRecordWriter(conf, tabPath);

    //Subtest 1: Simple writes
    Document rec = MapRDB.newDocument();
    rec.set("person.address.home.street", "525 E Maude Ave")
    .set("person.address.home.city", "Sunnyvale")
    .set("person.address.office.street", "San Jose")
    .set("person.address.office.zip", "95134");

    String row1Str = "r1";
    bulkLoadWriter.write(KeyValueBuilder.initFrom(row1Str), rec);

    rec = MapRDB.newDocument();
    rec.set("person.address.office.street", "350 Holger Way")
    .set("person.address.office.city", "San Jose")
    .set("person.address.office.zip", 95134);

    String row2Str = "r2";
    bulkLoadWriter.write(KeyValueBuilder.initFrom(row2Str), rec);

    //Force a flush
    bulkLoadWriter.write(null, (Document)null);

    //Subtest 2: Write mutation, incremental bulkload
    DocumentMutation mutation = MapRDB.newMutation();
    mutation.delete("person.address.office.street")
    .setOrReplace("person.address.home.city", "San Jose");

    String row3Str = "r3";
    ByteBuffer rowBytes3 = ByteBuffer.wrap(row3Str.getBytes());
    bulkLoadWriter.write(KeyValueBuilder.initFrom(row3Str), mutation);

    //Forced flush is not necessary, just for testing
    bulkLoadWriter.write(null, (DocumentMutation)null);

    //Subtest 3: Test spill splits (put size > 100 recordds or 1MB)
    ByteBuffer rowBytes = null;
    int i = 4;
    int j = 4000;
    for (; i < 999; ++i) {
      String rowStr = new String("r" + Long.toString(j++));
      bulkLoadWriter.write(KeyValueBuilder.initFrom(rowStr), rec);
    }

    bulkLoadWriter.close(null);

    desc.setBulkLoad(false);
    admin.alterTable(desc);

    //Verify all the writes
    Document readRec1 = table.findById(row1Str);
    assertEquals("525 E Maude Ave", readRec1.getString("person.address.home.street"));
    assertEquals("Sunnyvale", readRec1.getString("person.address.home.city"));

    Document readRec2 = table.findById(row2Str);
    assertEquals("350 Holger Way", readRec2.getString("person.address.office.street"));
    assertEquals("San Jose", readRec2.getString("person.address.office.city"));
    assertEquals(95134, readRec2.getInt("person.address.office.zip"));

    Document readRec3 = table.findById(row3Str);
    assertEquals("San Jose", readRec3.getString("person.address.home.city"));
    assertNull(readRec3.getString("person.address.office.street"));

    i = 4;
    j = 4000;
    Document readRec = null;
    for (; i < 999; i++) {
      String rowStr = new String("r" + Long.toString(j++));
      readRec = table.findById(rowStr);

      assertEquals("350 Holger Way", readRec.getString("person.address.office.street"));
      assertEquals("San Jose", readRec.getString("person.address.office.city"));
      assertEquals(95134, readRec.getInt("person.address.office.zip"));
    }

    admin.deleteTable(tabName);
 }

  @Test
  public void testIncrementalBulkLoadRecordWriter()
          throws IOException, Exception {
    assert(table != null);
    Configuration conf = new Configuration();
    BulkLoadRecordWriter bulkLoadWriter = new BulkLoadRecordWriter(conf, tablePath);

    //Subtest 1: Simple writes
    Document rec = MapRDB.newDocument();
    rec.set("person.address.home.street", "525 E Maude Ave")
    .set("person.address.home.city", "Sunnyvale")
    .set("person.address.office.street", "San Jose")
    .set("person.address.office.zip", "95134");

    String row1Str = "r1";
    bulkLoadWriter.write(KeyValueBuilder.initFrom(row1Str), rec);

    rec = MapRDB.newDocument();
    rec.set("person.address.office.street", "350 Holger Way")
    .set("person.address.office.city", "San Jose")
    .set("person.address.office.zip", 95134);

    String row2Str = new String("r2");
    bulkLoadWriter.write(KeyValueBuilder.initFrom(row2Str), rec);

    //Force a flush
    bulkLoadWriter.write(null, (Document)null);
    bulkLoadWriter.close(null);

    Document readRec1 = table.findById(row1Str);
    Document readRec2 = table.findById(row2Str);

    assertEquals("525 E Maude Ave", readRec1.getString("person.address.home.street"));
    assertEquals("Sunnyvale", readRec1.getString("person.address.home.city"));

    assertEquals("350 Holger Way", readRec2.getString("person.address.office.street"));
    assertEquals("San Jose", readRec2.getString("person.address.office.city"));
    assertEquals(95134, readRec2.getInt("person.address.office.zip"));

    //Subtest 2: Write mutation
    bulkLoadWriter = new BulkLoadRecordWriter(conf, tablePath);
    DocumentMutation mutation = MapRDB.newMutation();
    mutation.delete("person.address.office.street")
    .setOrReplace("person.address.home.city", "San Jose");

    bulkLoadWriter.write(KeyValueBuilder.initFrom(row2Str), mutation);
    //Forced flush is not necessary, just for testing
    bulkLoadWriter.write(null, (DocumentMutation)null);
    bulkLoadWriter.close(null);

    readRec1 = table.findById(row2Str);

    assertEquals("San Jose", readRec1.getString("person.address.home.city"));
    assertNull(readRec1.getString("person.address.office.street"));

    //Subtest 3: Test spill splits (put size > 100 recordds or 1MB)
    bulkLoadWriter = new BulkLoadRecordWriter(conf, tablePath);
    ByteBuffer rowBytes = null;
    int i = 4;
    int j = 4000;
    for (; i < 999; ++i) {
      row2Str = new String("r" + Long.toString(j++));
      bulkLoadWriter.write(KeyValueBuilder.initFrom(row2Str), rec);
    }
    bulkLoadWriter.close(null);

    i = 4;
    j = 4000;
    Document readRec = null;
    for (; i < 999; i++) {
      row2Str = new String("r" + Long.toString(j++));
      readRec = table.findById(row2Str);

      assertEquals("350 Holger Way", readRec.getString("person.address.office.street"));
      assertEquals("San Jose", readRec.getString("person.address.office.city"));
      assertEquals(95134, readRec.getInt("person.address.office.zip"));
    }
 }

 /*
  * Note: Table and Column Families need to be created using maprcli
  * Table Path: /tmp/test_bulkload_json
  * Col Families:
  *  default = ""
  *  cf1 => defaultchild4.family1
  *  cf2 => defaultchild4.family2
  *  cf3 => Family1
  *  cf4 => Family1.child4.family1
  *  cf5 => Family1.child4.family2
    maprcli table create -path /tmp/test_bulkload_json  -tabletype json
    maprcli table cf create -path /tmp/test_bulkload_json -cfname default
    maprcli table info -path /tmp/test_bulkload_json
    maprcli table cf create -path /tmp/test_bulkload_json -cfname cf1 -jsonpath defaultchild4.family1
    maprcli table cf create -path /tmp/test_bulkload_json -cfname cf2 -jsonpath defaultchild4.family2
    maprcli table cf create -path /tmp/test_bulkload_json -cfname cf3 -jsonpath Family1
    maprcli table cf create -path /tmp/test_bulkload_json -cfname cf4 -jsonpath Family1.child4.family1
    maprcli table cf create -path /tmp/test_bulkload_json -cfname cf5 -jsonpath Family1.child4.family2
    maprcli table cf list -path /tmp/test_bulkload_json
  */
 @Test
 public void testBulkLoadMultiCFFull()
   throws IOException, Exception {
   createBulkLoadTable(true /*fullBulkLoad*/);
   testBulkLoadMultiCFCommon();
   deleteBulkLoadTable();
 }

 @Test
 public void testBulkLoadMultiCFIncremental()
   throws IOException, Exception {
   createBulkLoadTable(false /*fullBulkLoad*/);
   testBulkLoadMultiCFCommon();
   deleteBulkLoadTable();
 }

 private void testBulkLoadMultiCFCommon()
   throws IOException, Exception {
   Path tabPath = new Path(TEST_BULKLOAD_JSON_TABLE);
   try {
     table = MapRDB.getTable(tabPath);
     table.setOption(TableOption.BUFFERWRITE, false);
     table.setOption(TableOption.EXCLUDEID, true);
   } catch (Exception e) {
     _logger.info("Failed to get table. Create table '{}'.", tabPath);
     throw e;
   }

   Document rec = MapRDB.newDocument();
   rec.set("defaultchild1", 10)
   .set("defaultchild2", true)
   .set("defaultchild3", "Value")
   .set("defaultchild4.child1", "child1")
   .set("defaultchild4.child2", "child2")
   .set("defaultchild4.family1", "family1")
   .set("defaultchild4.family2.child1", false)
   .set("defaultchild4.family2.child2", 50000)
   .set("Family1.child1", 10)
   .set("Family1.child2", true)
   .set("Family1.child3", "Value")
   .set("Family1.child4.child1", "child1")
   .set("Family1.child4.child2", "child2")
   .set("Family1.child4.family1", "family1")
   .set("Family1.child4.family2.child1", false)
   .set("Family1.child4.family2.child2", 50000);

   assert(table != null);
   Configuration conf = new Configuration();
   BulkLoadRecordWriter bulkLoadWriter = new BulkLoadRecordWriter(conf, tabPath);

   String rowStr = new String("r1");
   bulkLoadWriter.write(KeyValueBuilder.initFrom(rowStr), rec);

   bulkLoadWriter.close(null);

   clearBulkLoadTableFlag();

   Document readRec = (Document) table.findById(rowStr);

   assertNotNull(readRec);

   assertEquals(10, readRec.getInt("defaultchild1"));
   assertEquals(true, readRec.getBoolean("defaultchild2"));
   assertEquals("Value", readRec.getString("defaultchild3"));
   assertEquals("child1", readRec.getString("defaultchild4.child1"));
   assertEquals("child2", readRec.getString("defaultchild4.child2"));
   assertEquals("family1", readRec.getString("defaultchild4.family1"));
   assertEquals(Type.MAP, readRec.getValue("defaultchild4.family2").getType());
   assertEquals(false, readRec.getBoolean("defaultchild4.family2.child1"));
   assertEquals(50000, readRec.getInt("defaultchild4.family2.child2"));
 }
}


