package com.mapr.db.mapreduce.impl;

import static org.ojai.DocumentConstants.ID_FIELD;
import static org.ojai.store.QueryCondition.Op.GREATER_OR_EQUAL;
import static org.ojai.store.QueryCondition.Op.LESS_OR_EQUAL;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

import org.apache.hadoop.conf.Configuration;
import org.ojai.Document;
import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.util.Fields;

import com.google.common.io.BaseEncoding;
import com.mapr.db.Admin;
import com.mapr.db.MapRDB;
import com.mapr.db.impl.ConditionImpl;
import com.mapr.db.impl.TableDescriptorImpl;
import com.mapr.db.mapreduce.TableInputFormat;
import com.mapr.db.util.ByteBufs;
import com.mapr.fs.proto.Marlinserver.MarlinInternalDefaults;


public class MapReduceUtilMethods {

  public static void setStartStopRow(Configuration conf) {
    /* extract parameters */
    String startRow = conf.get(TableInputFormat.START_ROW);
    String endRow = conf.get(TableInputFormat.STOP_ROW);
    if ((startRow == null) && (endRow == null)) {
      return;
    }
    ConditionImpl cond = ((ConditionImpl)MapRDB.newCondition()).and()
          .is(ID_FIELD, GREATER_OR_EQUAL, startRow)
          .is(ID_FIELD, LESS_OR_EQUAL, endRow)
          .close()
          .build();

    ByteBuffer serializedCond = cond.getDescriptor().getSerialized();
    byte[] condArray = new byte[serializedCond.remaining()];
    serializedCond.get(condArray);
    //String conditionString = new String(Base64.encodeBase64(condArray));
    String conditionString = BaseEncoding.base64().encode(condArray);
    conf.set(TableInputFormat.COND_OBJ, conditionString);
  }

  public static String [] getStreamDefaultCFsForRepl() {
    MarlinInternalDefaults def = MarlinInternalDefaults.getDefaultInstance();
    String[] fields = new String[3];
    fields[0] = def.getCfTopicMeta();
    fields[1] = (def.getCfMessages().equals("default") ? new String("") : def.getCfMessages());
    fields[2] = def.getCfCursors();
    return fields;
  }

  public static String processColumnSpec(String columnSpec, String tabPath) {
    String fields[] = null;
    Admin admin = MapRDB.newAdmin();
    TableDescriptorImpl desc = (TableDescriptorImpl) admin.getTableDescriptor(tabPath);

    if (columnSpec != null) {
      fields = columnSpec.split(",", -1);
    } else {
      /*
       * if explicit columns are not provided and its a stream then
       * compare only the replicated cfs of the stream
       */
      if (desc.isStream()) {
        fields = getStreamDefaultCFsForRepl();
      }
    }

    if (fields != null) {
      int numFields = fields.length;
      StringBuilder cols = new StringBuilder();
      for (int i = 0; i < numFields; ++i) {
        cols.append(fields[i]);
        if (i < (numFields - 1)) {
          cols.append(",");
        }
      }
      return cols.toString();
    }
    return null;
  }

  //moving utility functions
  public static ByteBuffer getRowKey(Document record, String field){

    Value value = record.getValue(field);
    if (value.getType() == Type.BINARY) {
      return value.getBinary();
    }
    if (value.getType() == Type.STRING) {
      try {
        return ByteBufs.wrap(value.getString().getBytes("UTF-8"));
      } catch (Exception e) {
        return null;
      }
    }
    return null;
  }

  /* utility method to parse comma-separated fieldpaths for projection */
  public static String[] getFieldPaths(String fieldPaths) {
    String[] tokens = fieldPaths.split(",", -1);
    List<String> pathList = new ArrayList<String>();
    int length = tokens.length;

    int count = 0;
    String currentToken = null;
    while (count < length) {
      if (currentToken!=null) {
        currentToken += (","+tokens[count]);
       if (tokens[count].endsWith(String.valueOf(Fields.SEGMENT_QUOTE_CHAR))) {
          pathList.add(currentToken);
          currentToken = null;
        }
      } else {
        if (tokens[count].charAt(0) != Fields.SEGMENT_QUOTE_CHAR) {
          pathList.add(tokens[count]);
        } else {
          currentToken = tokens[count];
        }
      }
      count++;
    }

    String[] paths = new String[pathList.size()];
    paths = pathList.toArray(paths);
    return paths;

  }

  public static boolean checkBulkloadStatus(boolean blkOption, String dstPath) {
    Admin maprAdmin = MapRDB.newAdmin();
    if (maprAdmin.tableExists(dstPath)) {
      TableDescriptorImpl descriptor = (TableDescriptorImpl)maprAdmin.getTableDescriptor(dstPath);
      if ((descriptor.isBulkLoad()) && (!blkOption)) {
        return false;
      }
    }
    return true;
  }

}
